(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 20.6.99 by Hendrik
 *
 * Time-stamp: <Monday 8 October 01 17:58:00 tews@ithif51>
 *
 * dynamic name spaces with unique identifier creation
 *
 * $Id: name_space.ml,v 1.12 2002/01/23 16:00:23 tews Exp $
 *
 *)

open Util
open Top_variant_types;;

(***********************************************************************
 ***********************************************************************
 *
 * types for name spaces
 *
 *)

type 'a get_name_type = 'a
constraint 'a = < get_name : string; .. > 


type ('cl, 'mem) abstract = 
    {eq : ('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types -> bool;
     mutable top : string list;
     env : string list list;
       (* a name space may be flat, see sub_space and make_flat *)
     mutable flat : bool
    }

type ('cl, 'mem) t = ('cl get_name_type, 'mem) abstract

    (* this is for the status of name creation for one type
     * Unchecked 	-> nothing has been done so far
     * Spaces_checked 	-> a strategy was successful, the name has 
     * 	   been reserved, and the default field contains this name
     *)
type status = 
  | Unchecked
  | Spaces_checked


(* 
 * this is used to keep some information about the strings we try 
 * to use for variable names,
 * Arbitrary -> nothing unusual
 * Type_name -> the string corresponds (partly) with the type
 *)
type 'a proposal_info =
  | Arbitrary of 'a
  | Type_name of 'a


type ('cl, 'mem) type_description = {
  typ : ('cl, 'mem) top_pre_types;
  strings : string proposal_info list;	(* possible names *)
  mutable multiple : bool;		(* type is requested several times *)
  mutable status : status;
  mutable default : string;		(* the name that is found *)

  mutable counter : int;

  mutable solve_uniqe_start_fails : bool;
  mutable solve_whole_fails : bool;
}

(***********************************************************************
 ***********************************************************************
 *
 * the implementation
 * docu -> mli
 *
 *)

let create eq = 
  ({ eq = eq;
     top = [];
     env = [];
     flat = false
   }
  : ('cl, 'mem) t)


let make_flat ns = ns.flat <- true

let sub_space (ns : ('cl, 'mem) t) = 
  if ns.flat then ns
  else 
    ({ eq = ns.eq;
       top = [];
       env = ns.top :: ns.env;
       flat = ns.flat	       
     }
     : ('cl, 'mem) t)

let test ns name = 
  (not (List.mem name ns.top))
  &&
  List.for_all (fun name_list -> 
		 not (List.mem name name_list)
	      ) ns.env

let reserve_one ns name = 
  ns.top <- name :: ns.top

let reserve ns name_list = 
  ns.top <- name_list @ ns.top

(***********************************************************************
 ***********************************************************************
 *
 * name creation requires a bit more ....
 *
 * module inherent heuristic to create names
 *
 *)

let arb_list l = List.map (fun s -> Arbitrary s) l

let type_name s =
  let ls = String.lowercase s 
  in
    if ls = s 
    then Type_name ls
    else Arbitrary ls

let string_of_type = function
  | Groundtype(id,args) -> [type_name(id.id_token.token_name)]
  | TypeConstant(s, _,args) -> [ type_name(s) ]
  | BoundTypeVariable id -> [type_name(id.id_token.token_name)]
  | Self -> arb_list ["x"; "y"; "z"]
  | Carrier -> arb_list ["x"; "y"; "z"]
  | Bool -> arb_list ["x"; "y"; "z"]
  | Function(Product _, Bool) -> arb_list ["R"; "Q"; "S"]
  | Function(dom, Bool) -> arb_list ["P"; "Q"; "U"]
  | Function(dom, codom)  -> arb_list ["f"; "g"; "h"]
  | SmartFunction(doml, codom)  -> arb_list ["f"; "g"; "h"]
  | Product facs -> []
  | Record recs -> [Arbitrary "rec"]
  | IFace(cl, flag, args) -> [Arbitrary "iface"]
  | Class( cl, args) -> [type_name cl#get_name]
  | Adt(adt, flag,args) -> [type_name adt#get_name]

  | Array(_, types, size) -> [Arbitrary "a"]
  | Predtype form -> arb_list ["P"; "Q"]
					(* not allowed here *)
  | FreeTypeVariable _ -> (assert(false); raise Internal_error)




    (* get the string out of a proposal info *)
let get_string = function
  | Arbitrary s -> s
  | Type_name s -> s

let same_as_type = function
  | Arbitrary _ -> false
  | Type_name _ -> true

    (* various function to search a type description list for 
     * special entries
     *)
let rec find_default name = function
  | [] -> None
  | desc::rest -> 
      if desc.default = name then Some desc
      else find_default name rest

let rec find_type_maybe eq typ = function
  | [] -> None
  | desc::rest -> 
      if eq desc.typ typ then Some desc
      else find_type_maybe eq typ rest

let find_type eq typ desclist = 
  match find_type_maybe eq typ desclist with
    | Some d -> d
    | None -> (assert(false);raise Internal_error)


let rec find_same_start char result = 
  let rec iter_props desc restdescs result = function
    | s::props -> (if (get_string s).[0] = char 
		   then find_same_start char (desc :: result) restdescs
		   else iter_props desc restdescs result props)
    | _ -> find_same_start char result restdescs
  in function
    | [] -> result
    | desc :: rest -> 
	if desc.status = Unchecked 
	then iter_props desc rest result desc.strings
	else find_same_start char result rest

let rec find_same_string string result = function
  | [] -> result
  | desc :: rest -> 
      match desc.strings with 
	| s::_ when (get_string s) = string ->
	    find_same_string string (desc::result) rest
      	| _ -> find_same_string string result rest

let make_multiple desc =
  desc.multiple <- true

    (* Build the description records. If several ids are requested 
     * for the same type we want to generate names with digets.
     * Therefore for every type there is usually only one description 
     * record. The exception is for the generation with_preference. To get 
     * not confused we return both the record list and a list of 
     * typ, record pairs, where in the latter equal types share the 
     * same record.
     *)

let collect_types ns req_list = 
  let result = ref [] in
  let type_desc_list = 
    List.map
      (fun typ -> 
	 (match find_type_maybe ns.eq typ !result with
	    | Some desc -> (make_multiple desc;
			    (typ,desc)
			   )
	     | None -> 
		 let desc = { typ = typ;
			      strings = string_of_type typ;
			      default = "";
			      multiple = false;
			      status = Unchecked;
			      counter = 1;
			      solve_uniqe_start_fails = false;
			      solve_whole_fails = false
	   		    } in
		   result := desc :: !result;
		   (typ, desc)
	 ))
      req_list
  in
    (type_desc_list, List.rev( !result ))


    (* similar to above, but
     * - no sharing if one type is requested multiple times
     * - the prefered name is put into the strings list
     *)
let collect_types_with_preference req_list = 
  let type_desc_list =
    List.map
      (fun (name,typ) ->
	 let desc = { typ = typ;
		      strings = [Arbitrary name];
		      default = "";
		      multiple = false;
		      status = Unchecked;
		      counter = 1;
		      solve_uniqe_start_fails = false;
		      solve_whole_fails = false
		    }
	 in
	   (typ,desc)
      ) req_list
  in
    (type_desc_list, List.map snd type_desc_list)


let solve_uniqe_start ns desclist = 
    (* 
     * If desclist contains more than one description, than two records 
     * might conflict and the heuristic fails for the first. It is only 
     * fair, that it fails for the other conflicting records too. We use 
     * the flag solve_uniqe_start_fails for that. 
     *)
  let rec check_props desc restdescs = function
    | [] -> ()
    | s :: prop_list -> 
	if (String.length (get_string s) = 1) && (same_as_type s) 
	then
	  ()				(* do nothing *)
	else
      	  let char = (get_string s).[0] in
	  let name = Char.escaped(char) in
      	  let same = find_same_start char [] restdescs in
	    if same = [] then 
	      begin
	      	if test ns name then begin
	      	  reserve_one ns name;
	      	  desc.status <- Spaces_checked;
	      	  desc.default <- name
	      	end else begin
	      	  check_props desc restdescs prop_list
	      	end
	      end else begin
	      	List.iter (fun d -> d.solve_uniqe_start_fails <- true)
		  same;
	      	check_props desc restdescs prop_list
	      end
  in let rec doit = function
    | [] -> ()
    | { status = Unchecked; solve_uniqe_start_fails = false } 
	as desc :: rest ->
	  begin 
 	    check_props desc rest desc.strings;
	    doit rest
	  end
    | _ :: rest -> doit rest
  in 
    doit desclist

let solve_whole ns desclist =
  let rec check_props desc restdescs = function
    | [] -> ()
    | s :: prop_list ->
      	if (same_as_type s) or (String.length (get_string s) = 1 )
      	then 
	  check_props desc restdescs prop_list
 	else
      	  let name = get_string s in
      	  let same = find_same_string name [] restdescs  in
	    if same = [] then 
	      if test ns name then begin
		reserve_one ns name;
	      	desc.status <- Spaces_checked;
	      	desc.default <- name
	      end else begin
		check_props desc restdescs prop_list
	    end else begin
	      List.iter (fun d -> d.solve_whole_fails <- true)
		same;
	      check_props desc restdescs prop_list
	    end
  in let rec doit = function
    | [] -> ()
    | { multiple = true } :: rest -> doit rest
    | { status = Unchecked; solve_whole_fails = false } as desc :: rest ->
	begin
	  check_props desc rest desc.strings;
	  doit rest
	end 
    | _ :: rest -> doit rest
  in 
    doit desclist
	  
let create_stupid_name ns stem =
  let counter = ref 0 in
  let rec do_it () =
    let name = stem ^ (string_of_int !counter) in
    let _ = incr counter in
      if test ns name 
      then
	begin
	  reserve_one ns name;	
	  name
	end
      else do_it ()
  in
    do_it ()

(* I need an exception for jumping out *)
exception Found of string

let generate_ids ns type_desc_list = 
  let rec make_id desc = 
    let name = desc.default ^ (string_of_int desc.counter) in
    let _ = desc.counter <- desc.counter +1
    in
      if test ns name then name
      else make_id desc
  in let take_checked_default desc =
      if desc.status = Spaces_checked then 
	if desc.multiple then
	  raise (Found (make_id desc))
	else 
	  raise (Found desc.default)
  in let try_full_names s =
      let ls = get_string s in
      	if not (same_as_type s)  && test ns ls 
      	then
	  begin
	    reserve_one ns ls;
	    raise (Found ls)
	  end
  in let do_stupid () =
      raise (Found (create_stupid_name ns "p"))
  in let do_typ (typ,desc) =
	(* each of the following function tries something,
	 * the first success raises Found(name)
	 *)
	try 
	  take_checked_default desc;
	  if desc.multiple = false then
	    List.iter try_full_names desc.strings;
	  (* the last one is always successful *)
	  do_stupid ();
	  assert(false);raise Internal_error
	with Found name -> name,typ
  in 
    List.map do_typ type_desc_list

let create_ids ns typ_list =
  let type_desc_list, desclist = collect_types ns typ_list in
  let _ = solve_uniqe_start ns desclist in
  let _ = solve_whole ns desclist in
  let result = generate_ids ns type_desc_list in
    result

let create_one_id ns typ = fst (List.hd (create_ids ns [typ]))

let create_id_pairs ns typ_list =
  let l = List.length typ_list in
  let vars = create_ids ns (typ_list @ typ_list) in
    (Util.take vars l, Util.tail vars l)


let create_ids_with_variance ns vtype_list = 
  let ids = create_ids ns (List.map fst vtype_list) in
  let change_name name s =
    let nname = name ^ s in
      reserve_one ns nname;
      nname	    
  in
    List.map2
      (fun (t,v) (name,t') ->
	 let _ = assert( t == t' ) 
	 in match Logic_util.make_simple v with
	   | Unused
	   | Pos -> (None, Some((name,t)))
	   | Neg ->
	       (Some(( (change_name name "Neg"), t)), None)
	   | Mixed -> 
	       (Some(( (change_name name "Neg"), t)),
		Some(( (change_name name "Pos"), t)))
	   | _ -> assert(false); raise Internal_error)
      vtype_list ids


    (* 
     * create_id_with_preference ns name stem
     * reserves name and returns it, if possible;
     * otherwise use stem to create a name with appending digits
     * 
     * val create_id_with_preference :
     * 	('cl,'mem) t -> string -> string -> string
     * 	  
     *)

let create_id_with_preference ns name stem =
  if test ns name 
  then
    begin
      reserve_one ns name;	
      name
    end
  else
    create_stupid_name ns stem


let create_ids_with_preference ns typ_pref_list =
  let type_desc_list, desclist = collect_types_with_preference typ_pref_list in
  let _ = solve_whole ns desclist in
  let result = generate_ids ns type_desc_list in
    result





(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** End: ***)

