(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 15.1.99 by Hendrik
 *
 * Time-stamp: <Wednesday 22 May 02 11:02:43 tews@ithif51>
 *
 * global control variables of the loop compiler
 *
 * $Id: global.ml,v 1.15 2002/05/22 13:42:45 tews Exp $
 *
 *)


open Util
open Top_variant_types
;;

(***********************************************************************
 ***********************************************************************
 *
 * constant string section
 *
 *)


let compiler_version = "CCSL Compiler Version 2.1.3"


(***********************************************************************
 ***********************************************************************
 *
 * compiler behaviour flag section
 *
 *)


    (* set to Pvs_mode via -pvs in main.ml
     * set to Isa_mode via -isa in main.ml
     * initially set to None
     *)
let output_mode = ref (Pvs_mode : output_mode)


    (* if true parse the prelude(s)
     * set in main.ml, via -no-prelude option
     *)
let parse_prelude = ref true


    (* if true enable the java2 option in the java compiler
     * set in main.ml, via -java2 option
     *)
let java2_mode = ref false


    (* if true enable the jml option in the java compiler
     * set in main.ml, via -jml option
     *)
let jml_mode = ref false


    (* do things, a normal user should never see
     * set via -expert
     *)
let expert_mode = ref false


    (* generate method bodies without local variables
     * set via -hoare
     *)
let hoare_mode = ref false


    (* don't use the fixed point definition for method calls,
     * but the direct rewrite instead. This simplifies verification
     * when there are no (mutually) recursive method definitions.
     * set via -no-recursion
     *)
let no_java_recursion = ref false


    (* if <> "" generate all output in this file
     * set in main.ml, via -o option
     *)
let target_file = ref ""


    (* access functions for target_file
     *)
let set_target_file file =
  target_file := file


let get_target_file () =
  !target_file


    (* if <> "" create output in this directory
     * set in main.ml, via -d option
     *)
let target_directory = ref ""


    (* access functions for target_directory
     *)
let set_target_directory directory =
  target_directory := directory ^ "/"


let get_target_directory () =
  !target_directory


(***********************************************************************
 ***********************************************************************
 *
 * debug level section
 *
 *)

    (* debug_level_var: controls amount of status information
     * 1 : identical to -v (verbose)
     * 2 : print information about lexer
     * 4 : print information about parser
     * 8 : print information about resolution
     * 16 : print information about inheritance
     * 32 : print information about typecheck
     * 64 : print information about syntax tree
     * 128 : print information about symbol table
     * 256 : apply everything also to the prelude
     * 512 : print information about unification
     * 1024 : print assertions directly after parsing
     * 2048 : variance pass
     * 4096 : feature pass
     * set in main.ml, via -D [int] option
     *)
let _VERBOSE = 1
let _DEBUG_LEXER = 2
let _DEBUG_PARSER = 4
let _DEBUG_RESOLUTION = 8
let _DEBUG_INHERITANCE = 16
let _DEBUG_TYPECHECK = 32
let _DEBUG_DUMP_SYNTAX_TREE = 64
let _DEBUG_DUMP_SYMBOL_TABLE = 128
let _DEBUG_PRELUDE = 256
let _DEBUG_UNIFICATION = 512
let _DEBUG_PARSERCHECK = 1024
let _DEBUG_VARIANCE = 2048
let _DEBUG_FEATURES = 4096

let debug_level_var = ref 0

    (* access functions for debug_level_var
     *)
let add_debug_level level =
  if level > 0
  then
    debug_level_var := !debug_level_var lor level
  else
    debug_level_var := 0


let get_debug_level () =
  !debug_level_var


let debug_level pass =
  (!debug_level_var land pass) <> 0


    (* if false, do not generate any theories
     * unset in main.ml, via option -prf
     *)
let theory_generation = ref true


    (* if true, unparse the AST
     * set in main.ml, via option -unparse
     *)
let unparse = ref false


    (* if true, print the output on stdout
     * set in main.ml, via option -c
     *)
let filter_style = ref false


    (*
     * generate batch file for theorem prover
     * set via -batch
     *)
let generate_batch = ref false


    (*
     * list of proof names to for proof testing
     * set via -prooftest
     *)
let do_proofs = ref ([] : string list)


    (*
     * do not generate any theories or proof files,
     * only print proofs in do_proofs on stdout
     *
     * set via -prooftest
     *)
let proof_test = ref false


    (*
     * do theory generation only for selected classes
     *
     * set via -class
     *)

let selected_classes_only = ref false


    (*
     * list of classes to generate theories for
     *
     * set via -class
     *)
let selected_classes = ref ([] : string list)


    (*
     * don't do theory generation for selected classes
     *
     * set via -butclass
     *)

let deselected_classes_not = ref false

    (*
     * list of classes to exclude for theory generation
     *
     * set via -butclass
     *)
let deselected_classes = ref ([] : string list)


    (*
     * enable opt_formula and opt_expression
     *
     * unset via -no-opt
     *)
let optimize_expressions = ref true


let inline_liftings = ref true


(***********************************************************************
 ***********************************************************************
 *
 * other variables
 *
 *)


    (* remember the top level input file,
     * inluces are taken relative to the directory of this file
     * This variable is maintained in abstract_lexer
     *)

let current_top_level_input = ref(None : string option)


    (* collect input files from the command line *)
let input_files = ref ([] : string list)

let ccsl_input_files = ref ([] : string list)

let add_input_file file =
  input_files := !input_files @ [file]


let get_input_files () =
  !input_files


    (* add ccsl files, get ccsl files *)
let add_ccsl_input_file file =
  ccsl_input_files := file :: !ccsl_input_files

let get_ccsl_input_files () =
  List.rev !ccsl_input_files

    (* holds the type used for natural numbers 
     * val set_nat_type : string -> unit
     * 
     * val get_nat_type : unit -> string
     *)

let nat_type_ref = ref "nat"

let set_nat_type name = nat_type_ref := name

let get_nat_type () = !nat_type_ref


    (* the path for the fixedpoints library
     *)

let fixedpointlib = ref Config.fixedpointlib

let set_pvs_ccsl_lib_location loc = fixedpointlib := loc

let get_pvs_ccsl_lib_location () = !fixedpointlib


(*******************************************************************
 *
 * ccsl specialities
 *)

    (* true when parsing the prelude *)

let parsing_prelude = ref false

    (*
     * this flag modifies iface creation in the parser,
     * if set, creates special objects of special subclasses
     * this way it is possible, to change the iface generated for the
     * list adt in the prelude, such that the internal pvs lists are used
     * in the translation
     *)
let parsing_builtin_prelude = ref false
				

    (* generate theories for inductive invariants *)
let ccsl_generate_paths = ref false

(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)
