(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Sunday 19 May 02 13:34:40 tews@ithif56.inf.tu-dresden.de>
 *
 * predicate and relation lifting
 *
 * $Id: lifting.ml,v 1.19 2002/05/22 13:42:40 tews Exp $
 *
 *)

open Global
open Util
open Top_variant_types
open Top_variant_types_util
open Logic_util
open Names
open Types_util
open Name_space


(***********************************************************************
 *
 * type param_lifting_type is an assoc list that
 * associates with every type parameter id two 
 * optional expressions. The first is the lifting for 
 * negative occurences, the second for positive occurences.
 * If None, then no lifting is performed for this occurence.
 * 
 * The constraint ensure that the function can access the 
 * variances of components and compute their lifing.
 *)

type ('cl, 'mem) param_lifting_type =
    (('cl, 'mem) top_pre_identifier_record_type
     * (('cl, 'mem) top_pre_expressions option 
	* ('cl, 'mem) top_pre_expressions option )
    ) list
constraint 'cl = <get_name : string; 
                  get_parameters : ('cl,'mem) top_pre_parameter_type list;
		  get_adt_constructors : 'mem list;
		  get_all_sig_actions : 'mem list;
		  get_self_variance : variance_type;
		  get_model_coalgebra : ('cl, 'mem) top_pre_expressions;
                  .. > 
constraint 'mem = <get_domain : ('cl,'mem) top_pre_types; 
		   get_codomain : ('cl,'mem) top_pre_types; 
		   get_name : string;
		   get_accessors : 'mem list;
		   get_recognizer : 'mem;
		   get_sort : 'mem top_pre_member_sort;
		   ..>


(***********************************************************************
 *
 * type self_lifting_type contains the negative (first) and the 
 * positive lifting for self. 
 *)

type ('cl, 'mem) self_lifting_type = 
    ('cl, 'mem) top_pre_expressions option 
    * ('cl, 'mem) top_pre_expressions option
constraint 'cl = <get_name : string; 
                  get_parameters : ('cl,'mem) top_pre_parameter_type list;
		  get_adt_constructors : 'mem list;
		  get_all_sig_actions : 'mem list;
		  get_self_variance : variance_type;
		  get_model_coalgebra : ('cl, 'mem) top_pre_expressions;
                  .. > 
constraint 'mem = <get_domain : ('cl,'mem) top_pre_types; 
		   get_codomain : ('cl,'mem) top_pre_types; 
		   get_name : string;
		   get_accessors : 'mem list;
		   get_recognizer : 'mem;
		   get_sort : 'mem top_pre_member_sort;
		   ..>

(* 
 * four utility functions extend the lifting when passing into 
 * an unused parameter position. 
 *)

let extend_param_pred_lifting param_list =
  List.map (fun (id, _) -> (id, (None, None))) param_list

let extend_self_pred_lifting = function
  | _ -> (None, None)

let extend_param_rel_lifting param_list =
  List.map (function
	      | (id, (Some _, Some _)) as x -> x
	      | (id, (None, Some p)) -> (id, (Some p, Some p))
	      | (id, (Some p, None)) -> (id, (Some p, Some p))
	      | _ -> assert(false)
	   )
    param_list

let extend_self_rel_lifting = function
  | (Some _, Some _) as x -> x 
  | (None, Some p) -> (Some p, Some p)
  | (Some p, None) -> (Some p, Some p)
  | None,None -> assert(false)



(***********************************************************************
 *
 * pre_predicate lifting -- the working horse for predicate lifting
 * This implements predicate lifting for 
 * full higher order polynomial functors
 * 
 * for classes and adt's a symbolic Every is produced
 * 
 * Arguments
 *     eq_types		equality on types (needed for inlining liftings)
 *     self_subst	substitution for Self
 *     param_lifting	lifting for typeparameters
 *     self_lifting	lifting for self
 *     top_name_space	name space for argument name creation
 *     variance		current variance
 *     typ		type to lift
 *     expression 	expression of type typ, to which the lifting 
 *         		is applied
 *
 * The algorithm is driven by structural induction on typ, and may 
 * produce very verbose output. Therefore the exported versions applies
 * opt_formula to the result.
 *)

let rec pre_predlift 
  eq_types
  self_subst
  (param_lifting : ('cl,'mem) param_lifting_type)
  (self_lifting : ('cl,'mem) self_lifting_type)
  top_name_space 
  variance typ expression =

			(* collect some utility function in this closure *)
  let subst =  List.map (fun (v,typ) -> (v, self_subst typ)) 
  in

					(* this is the real lifting *)
  let rec pred_lift_type name_space variance typ expr =
      match typ with 
        | Groundtype(id, args)   -> 
	    if is_type_def id
	    then			(* expand definition on the fly *)
	      pred_lift_type name_space variance 
		  (expand_type_def eq_types typ) expr

	    else
			(* For full predicate lifting we have to pass 
			 * the predicate for the typeparameters
			 * Therefore I generate an every here.
			 *)
	      let preds = pre_argument_list_predlift eq_types self_subst
			    param_lifting self_lifting 
			    name_space variance 
			    (get_ground_type_parameters id) args 
	      in
		Formula(
	  	  Application(
		    Every(typ, preds),
		    expr))
	  
        | Class(cl, args) -> 
	    if !Global.inline_liftings &&
	      (make_simple cl#get_self_variance = Unused)
	    then 
	      pre_inline_class_pred_lifting eq_types self_subst
		param_lifting self_lifting 
		name_space variance cl args expr
	    else
	      let preds = pre_argument_list_predlift eq_types self_subst
			    param_lifting self_lifting 
			    name_space 
			    variance cl#get_parameters args 
	      in
		Formula(
          	  Application(
		    Every(typ, preds),
		    expr))
        	
        | Adt(adt,flag,args)  -> 
	    if !Global.inline_liftings &&
	      (make_simple adt#get_self_variance = Unused)
	    then 
	      Formula(
		pre_inline_adt_pred_lifting eq_types self_subst
			   param_lifting self_lifting 
			   name_space variance adt args expr
	      )
	    else
	      let preds = pre_argument_list_predlift eq_types self_subst
			    param_lifting self_lifting 
			    name_space 
			    variance adt#get_parameters args 
	      in
		Formula(
          	  Application(
		    Every( typ, preds), 
		    expr))
		
        | BoundTypeVariable id   -> 
	    (try
	       let contra,co = List.assq id param_lifting
	       in (match variance,contra,co with
		     | true,_,Some pex -> Formula(Application(pex, expr))
		     | false,Some pex,_ -> Formula(Application(pex, expr))
		     | _ -> True
		  )
	     with
	       | Not_found -> True
	    )
        | Self
        | Carrier ->
	    let contra,co = self_lifting
	    in (match variance,contra,co with
		  | true,_,Some pex ->
		      Formula(Application(pex, expr))
		  | false,Some pex,_ ->
		      Formula(Application(pex, expr))
		  | _ -> True
	       )
	| Bool -> True
        | Function(dom,codom) -> 
            (* do currying here *)
	    let arglist = match dom with
              | Product tl -> tl
              | t -> [t] in
              (* make new name space for quantification *)
	    let sub_ns = sub_space name_space in 
	    let typed_args = create_ids sub_ns arglist in
	    let args =  (* we do not want to mix up one tuples *)
              match dom with
          	| Product _ -> Tuple(List.map (fun (v,_) -> Term(v,Always,[])) 
				       typed_args)
          	| _ -> 
		    Term(fst (List.hd typed_args), Always,[]) 
	    in
              Forall(subst typed_args, 
		     Implies( 
		       And( List.map 
			      (fun (v,arg_typ) ->
				 pred_lift_type sub_ns (not variance)
				   arg_typ (Term(v,Always,[])))
			      typed_args),
		       (pred_lift_type sub_ns variance codom
			  (Application(expr,args)))))
        | Product(type_list)-> 
	    let tuple_len = List.length type_list in
	    (* count the arguments to get the projection numbers *)
	    let i = ref 0 in
              And(    
          	List.map (fun p_type  -> 
			    incr(i);
			    pred_lift_type name_space variance p_type
			      (Application( 
				 Projection(!i, tuple_len),expr))) 
			type_list)
					(* do records as an exeption *)
        | Record label_list ->
	    And(    
              List.map (fun (label_name, label_type) -> 
			  pred_lift_type name_space variance
			    label_type (RecordSelection(label_name,expr)))
		      label_list)
					(* may not occur *)
	| TypeConstant _
        | Predtype _
	| FreeTypeVariable _
        | IFace _
        | Array _         
    	| SmartFunction _ -> assert(false)

  in
    pred_lift_type top_name_space variance typ expression



		(* make predicates for arguments of class or adt components *)
and pre_argument_list_predlift
  eq_types
  self_subst
  parameter_lifting 
  full_self_lifting 
  top_name_space
  variance
  parameter_list
  argument_list 
  = 
			(* lifting as expression for non constant types *)
  let non_trivial_pred_lift param_lifting self_lifting variance typ =
    let name_space = sub_space top_name_space in
    let var = (create_one_id name_space typ) 
    in
      Abstraction ([var, self_subst typ],
		   Expression(pre_predlift eq_types self_subst
				param_lifting self_lifting 
				name_space variance
				typ (Term(var,Always,[]))))
  in
    List.fold_right 
      (fun (typ,var) accu -> match var with
	 | Unused -> 
	     (non_trivial_pred_lift 
		(extend_param_pred_lifting parameter_lifting)
		(extend_self_pred_lifting full_self_lifting)
		variance typ
	     ) :: accu
	 | Pos -> 
	     (non_trivial_pred_lift parameter_lifting full_self_lifting 
		variance typ
	     ) :: accu
	 | Neg -> 
	     (non_trivial_pred_lift parameter_lifting full_self_lifting  
		(not variance) typ
	     ) :: accu
	 | Mixed ->
	     (non_trivial_pred_lift parameter_lifting full_self_lifting 
		(not variance) typ) ::
	     (non_trivial_pred_lift parameter_lifting full_self_lifting 
		variance typ) 
	     :: accu
	 | _ -> assert(false)
      )
				(* produce a list of type * variance pairs *)
      (List.combine
	 (types_from_arguments argument_list)
	 (List.map (function TypeParameter id ->
		      Logic_util.make_simple id.id_variance)
	    parameter_list)
      )
      []

and pre_inline_adt_pred_lifting 
  eq_types
  self_subst
  param_lifting 
  self_lifting 
  top_name_space variance adt args expr 
  =
  let adt_subst = 
    substitute_types_only eq_types 
      (make_substitution adt#get_parameters args) in
  let do_constr constr =
    let sub_ns = sub_space top_name_space in
    let dom = constr#get_domain in
    let arglist = List.map adt_subst (member_arg_list constr) in
    let typed_args = (create_ids sub_ns arglist) in
    let args =			(* we do not want to mix up one tuples *)
      match dom with
	| Product _ -> List.map (fun (v,_) -> Term(v,Always,[])) typed_args
	| _ -> [Term( fst(List.hd(typed_args)), Always,[])] in
    let left = match dom with
      | Product [] -> Term(constr#get_name, Always, [])
      | _ -> SmartApplication(Term(constr#get_name, Always, []), args) in
    let right = 
      Expression(
	And(
	  List.map 
	      (fun (v,typ) -> 
		 pre_predlift
		   eq_types
		   self_subst
		   param_lifting
		   self_lifting
		   sub_ns variance typ (Term(v,Always,[])))
	      typed_args)) 
    in
      left, right
  in
    Case(expr,
	 List.map do_constr adt#get_adt_constructors
	)


and pre_inline_class_pred_lifting 
  eq_types
  self_subst
  param_lifting 
  self_lifting 
  top_name_space variance cl args expr 
  =
  let class_subst = 
    substitute_types_only eq_types 
      (make_substitution cl#get_parameters args) in
  let do_method meth =
    let sub_ns = sub_space top_name_space in
    let dom = meth#get_domain in
    let arglist = List.map class_subst (method_arg_list meth) in
    let typed_args = (create_ids sub_ns arglist) in
    let args =				(* merge method arguments *)
      match dom with
	| Product _ -> 
	    Tuple( expr ::
		     List.map (fun (v,_) -> Term(v,Always,[])) typed_args)
	| _ -> expr in
    let body = 
      Forall(
	typed_args,
	Implies(
	  And(
	    List.map
	      (fun (v,typ) -> 
		 pre_predlift
		   eq_types
		   self_subst
		   param_lifting
		   self_lifting
		   sub_ns 
		   (not variance)
		   typ (Term(v,Always,[])))
		typed_args), 
	  pre_predlift
	    eq_types
	    self_subst
	    param_lifting
	    self_lifting
	    sub_ns variance 
	    (class_subst meth#get_codomain)
	    (Application(			(* arguments *)
	       Application(			(* coalgebra *)
		 Term(meth#get_name,Always,[]),
		 cl#get_model_coalgebra),
	       args))))
    in
      body
  in
    And(List.map do_method cl#get_all_sig_actions)



(***********************************************************************
 *
 * fullpredlift -- the exported version with optimization
 * 
 * Arguments
 *     eq_types		equality on types (needed for inlining liftings)
 *     self_subst	substitution for Self
 *     param_lifting	lifting for typeparameters
 *     self_lifting	lifting for self
 *     top_name_space	name space for argument name creation
 *     typ		type to lift
 *     expression 	expression of type typ, to which the lifting 
 *         		is applied
 *)
      
      
let fullpredlift eq_types self_subst param_lifting self_lifting 
  top_name_space typ expression =
    opt_formula(
      pre_predlift eq_types self_subst param_lifting self_lifting 
		  top_name_space true typ expression
    )


(***********************************************************************
 *
 * predlift -- version of fullpredlift without parameter lifting 
 *             and substitution
 *
 * Arguments
 *     eq_types		equality on types (needed for inlining liftings)
 *     self_lifting	lifting for self
 *     top_name_space	name space for argument name creation
 *     typ		type to lift
 *     expression 	expression of type typ, to which the lifting 
 *         		is applied
 *)

let predlift eq_types self_lifting top_name_space typ expression =
					(* no lifting for parameters *)
  let param_lifting = [] in
  let self_subst = 
    (* the following is equivalent with 
     *    ccsl_substitute_types [] 
     * but does not spoil the types
     *)
    (fun a -> a)
  in
    fullpredlift eq_types self_subst param_lifting self_lifting 
      top_name_space typ expression



(***********************************************************************
 *
 * argument_list_fullpredlift -- predicate lifting for argument lists
 * 
 * Arguments
 *     self_subst	    substitution for Self
 *     param_lifting	    lifting for typeparameters
 *     self_lifting	    lifting for self
 *     top_name_space	    name space for argument name creation
 *     parameter_list       original parameter list of the 
 *			    ancestor or component (needed to get variances)
 *     argument_list        argument type list that drives the lifting
 *)


let argument_list_fullpredlift eq_types self_subst param_lifting self_lifting
  top_name_space parameter_list argument_list 
  =
  List.map opt_expression
    (pre_argument_list_predlift eq_types
       self_subst param_lifting self_lifting top_name_space
       true parameter_list argument_list)


(*************************************************************************
 *
 * inline_adt_pred_lifting -- predicate lifting of adt as case expression
 * 
 * Arguments
 * 
 *     eq_types		equality on types (for substitution)
 *     self_subst		substitution for carrier
 *     param_lifting	lifting for parameters
 *     self_lifting	lifting for self
 *     top_name_space	the name space
 *     adt			the adt
 *     args		its arguments 
 *     expr 		the expression to lift
 *)


let inline_adt_pred_lifting eq_types self_subst
  param_lifting self_lifting name_space adt args expr 
  =
  opt_expression(
    pre_inline_adt_pred_lifting eq_types self_subst
		   param_lifting self_lifting 
		   name_space true adt args expr
  )

(******************************
 * Same as predlift, with the following differences:
 * - the lifting wrt. to a one method instead of a type
 * - applications of this method occurs in standard form: m(c)( args )
 * 
 * In class_method_lifting anc_lifting name_space predicate coalgebra 
 * 	      ancestor_list method_list selfvar self_type_argument
 * the arguments are
 *     eq_types		  -> equality on types (needed for inlining liftings)
 *     param_lifting	  -> assoc list of param_lifting_type that gives 
 *			     the liftings form typeparameters
 *     self_lifting	  -> lifting for Self
 *     name_space         -> name space for name creation
 *     coalgebra          -> the coalgebra for getting the methods right
 *     meth		  -> the method
 *     selfvar            -> the variable of Self, to which we apply
 * 				the lifting
 *)


let class_pred_lifting eq_types param_lifting self_lifting
  name_space 
  coalgebra meth selfvar =

					(* substitution not needed here *)
  let self_subst = 
    (* the following is equivalent with 
     *    ccsl_substitute_types [] 
     * but does not spoil the types
     *)
    (fun a -> a)
  in

(* 	     (* lifting for ancestors *)
 *   let do_ancestor (anc, anc_arguments) =
 *     Formula(
 * 	 Application(
 * 	   Application(
 * 	     Application(
 * 	       Term(anc_lifting anc, Always,
 * 		    self_type_argument :: anc_arguments), 
 * 	       Application(Term(super_access_method anc, Always,[]), coalgebra)),
 * 	     (remove_option (snd self_lifting))
 * 	     ),
 * 	   Term(selfvar,Always,[]))) in
 * 	     (* for methods *)
 *)
  let dom = meth#get_domain in
  let codom = meth#get_codomain in 
  let name = meth#get_name in
					(* do currying here *)
  let arglist = method_arg_list meth in
				(* make new name space on quantification *)
  let sub_ns = sub_space name_space in
  let typed_args = (create_ids sub_ns arglist) in
  let args =			(* we do not want to mix up one tuples *)
      match dom with
	| Product _ -> Tuple(Term(selfvar,Always,[]) :: 
			       (List.map (fun (v,_) -> Term(v,Always,[])) 
				  typed_args))
	| _ -> Term(selfvar, Always,[]) 
  in let form =
      Forall(typed_args, 
	     Implies( 
               And( List.map (fun (v,arg_typ) ->
				pre_predlift eq_types self_subst param_lifting 
				  self_lifting sub_ns false
				  arg_typ (Term(v,Always,[])))
		      typed_args),
               (pre_predlift
		  eq_types
		  self_subst
		  param_lifting self_lifting
		  sub_ns 
		  true
		  codom
		  (Application(
		     Application(Term(name,Always,[]), coalgebra),
		     args)))))
  in
    opt_formula(form)
      


(* funtions for method specific predicate lifting,
 * 
 * class_method_pred_definition:
 * ns                  -> namespace
 * rec_type            -> give the type of the iface-functor
 * m_pred              -> predicate-type for method lifting
 * m                   -> method for wich we generate the lifting
 *)

let class_method_pred_definition ns m_pred c_name c_type m =
  let sub_ns = sub_space ns in
  let p = create_one_id sub_ns (Function(Self,Bool)) in
  let self_var = create_one_id sub_ns Self in
  let self_lifting = 
    (None, 
     Some(Expression(
	    ConstantPredicate(p)
	  )))
  in
  let param_lifting = [] 
  in

    Defn(
      name_of_method_predicate m,
      [[Declared(c_name,c_type)]],
      m_pred, 
      Abstraction(
        [p, Function(Self,Bool)],
        Abstraction(
          [self_var, Self],
	  Expression(
	    class_pred_lifting 
		       eq_ccsl_types
		       param_lifting
		       self_lifting		       
		       sub_ns 
		       (Term(c_name,Always,[]))
		       m 
		       self_var 
	  )
        )
      )
    )


(*
 * class_real_method_pred_lifting:
 *
 * DO NOT RELY ON SUPERCLASSES:
 *
 * Do not use the methodinv theory of one of the superclasses!
 * This would yield completely wrong liftings, if one
 * of the ancestors is instanciated with self
 *)



let class_real_method_pred_lifting cl class_args name_space list
  predicate coalgebra method_id method_enum_type
  selfvar =
    
  let do_method m =
    (Term( name_of_method_tag m,Always,[]),
     Application(
       Application(
	 Application(
	   Term(name_of_method_predicate m, Always, []),
	   coalgebra),
	 Expression(predicate)
       ),
       Term(selfvar, Always, [])
     ))
  in
  let method_term = Term(method_id,Always,[]) in
  let case_list = 
    List.map do_method
      (List.filter
         (fun m -> m#needs_lifting)
         cl#get_all_methods
      )
  in
    if case_list = [] then
      True
    else
      Forall(
	[method_id,method_enum_type],
	Implies(
	  Formula(
	    SmartApplication(
	      Term(name_of_list_member (), Always, []),
	      [method_term;
	       list]
	    )
	  ),
	  Formula(
	    Case(
	      method_term,
	      case_list
	    )
	  )
	)
      )

(* legacy code
 * (************************************************************************
 *  * full constructor wise adt predicate lifting, needed for Isabelle
 *  *
 *  * parameters
 *  *     self_subst	-> substitution for self
 *  *     param_lifting 	-> lifting for type parameters
 *  *     name_space       -> name space for name creation
 *  *     predicate        -> the predicate, that we lift 
 *  *     constr     	-> constructor
 *  *)
 * 
 * let adt_constr_predicate_lifting self_subst param_lifting 
 *   name_space predicate constr =
 * 					   (* Predicates for typeparameters *)
 *   let sub_ns_cons = sub_space name_space in
 * 	     (* lifting for constructors *)
 *   let self_lifting = 
 *     (Some (Expression(predicate)), Some(Expression(predicate))) in
 *   let dom = constr#get_domain in
 *   let codom = constr#get_codomain in 
 *   let name = constr#get_name in
 *   let arglist = member_arg_list constr
 * 	    (* make new name space on quantification *)
 *   let typed_args = (create_ids sub_ns_cons arglist) in
 *   let args =  (* we do not want to mix up one tuples *)
 *     match dom with
 * 	 | Product _ -> List.map (fun (v,_) -> Term(v,Always,[])) typed_args
 * 	 | _ -> [Term( fst(List.hd(typed_args)), Always,[])] in
 *   let left = match dom with
 *     | Product [] -> Term(constr#get_name, Always, [])
 *     | _ -> SmartApplication(Term(constr#get_name, Always, []), args) in
 *   let right = 
 *     Expression(
 * 	 And(
 * 	   List.map 
 * 	       (fun (v,typ) -> 
 * 		  pre_predlift 
 * 		    self_subst
 * 		    param_lifting
 * 		    self_lifting sub_ns_cons true typ (Term(v,Always,[])))
 * 	       typed_args)) 
 *   in
 *     left, (opt_expression right)
 *     
 *)

(***********************************************************************
 ***********************************************************************
 *
 * RELATION LIFTING
 *
 ***********************************************************************
 ***********************************************************************
 *
 * This implements relation lifting for 
 * full higher order polynomial functors
 * 
 * This function is used to create both the relation lifting for adt's 
 * and classes. For adt's we use an inductively defined function 
 * (reduce). For this to work an important condition must be 
 * fullfilled: The substitution for self1, self1_subst must 
 * substitute Self with a function self2 -> bool. The clause 
 * for the case ``Carrier'' makes only sense, if the previous 
 * statement ist true. Note that this is different, from predicate 
 * lifting, because there we don't generate an reduce, but equations 
 * for the ``Isabelle primrec'' feature. 
 * 
 * for classes and adt's a symbolic RelEvery is produced
 * 
 * Arguments
 *     eq_types		    equality on types (needed for inlining liftings)
 *     self1_subst          substitution function for Self for expr1
 *     self2_subst          substitution function for Self for expr2
 *     param_rel_lifting    the lifting for typeparameters
 *     self_lifting	    the lifting for Self/Carrier
 *     top_name_space       name space for argument name creation
 *     variance		    true for positive, false for negative
 *     typ                  type to lift
 *     expr1, expr2         expressions of type typ, to which the lifting 
 *         		    is applied
 *
 * The algorithm is driven by structural induction on typ, and may 
 * produce very verbose output. Therefore the exported versions applies
 * opt_formula to the result.
 *)

let rec pre_rellift 
  eq_types
  self1_subst self2_subst
  (param_rel_lifting : ('cl,'mem) param_lifting_type)
  (self_lifting : ('cl,'mem) self_lifting_type) 
  top_name_space 
  variance
  typ expr1 expr2 =
			(* collect some utility function in this closure *)
 (* we need substitutions for self1 and self2, for 
  * convenience we construct the following utility functions 
  * such that they can be applied to a (string * top_types) list,
  * where the substitution is only applied to the types
  *)
  let subst1 = 
    List.map (fun (v,typ) -> 
    (v, self1_subst typ)) in
  let subst2 = 
    List.map (fun (v,typ) -> 
    (v, self2_subst typ)) in

(* 	 (* predicate on parameters, returns false, 
 * 	  * if the lifting for this parameter is not Equality
 * 	  *)
 *   let param_p id = 
 *     let x = Term("x",Always,[]) in
 * 	 (param_rel_lifting id x x) = Equal(x,x) 
 *   in
 *)


					(* this is the real lifting *)
  let rec rel_lift_type name_space variance typ expr1 expr2 =
    match typ with 
      | Groundtype(id, args)   -> 
	  if is_type_def id
	  then				(* expand definition on the fly *)
	    rel_lift_type name_space variance
	      (expand_type_def eq_types typ) expr1 expr2

	  else
	        (* !!! same discussion as for predicate lifting !!! 
		 * 
		 * For full relation lifting we have to pass 
		 * the predicate for the typeparameters
		 * Therefore I generate an every here.
		 *)
	    let rels = pre_argument_list_rellift eq_types
			 self1_subst self2_subst
			 param_rel_lifting self_lifting 
			 name_space variance
			 (get_ground_type_parameters id)
			 args 
	    in
	      Formula(
		Application(
		  RelEvery(typ, rels),
		  Tuple([expr1;expr2])))

      | Class(cl, args) -> 
	  if !Global.inline_liftings && 
	    (make_simple cl#get_self_variance = Unused)
	  then 
	    pre_inline_class_rel_lifting eq_types self1_subst self2_subst
		param_rel_lifting self_lifting 
		name_space variance cl args expr1 expr2
	  else
	    let rels = pre_argument_list_rellift eq_types 
			 self1_subst self2_subst
			 param_rel_lifting self_lifting 
			 name_space variance cl#get_parameters args in
	      Formula(
		Application(
		  RelEvery(typ, rels),
		  Tuple([expr1;expr2])))
		    
      | Adt(adt,flag,args) 	-> 
	  if !Global.inline_liftings && 
	    (make_simple adt#get_self_variance = Unused)
	  then 
	    Formula(
	      pre_inline_adt_rel_lifting eq_types self1_subst self2_subst
		      param_rel_lifting self_lifting 
		      name_space variance adt args expr1 expr2
	    )
	  else
	    let rels = pre_argument_list_rellift eq_types
			 self1_subst self2_subst
			 param_rel_lifting self_lifting 
			 name_space variance adt#get_parameters args in
	      Formula(
		Application(
		  RelEvery( typ, rels), 
		  Tuple([expr1;expr2])))

      | BoundTypeVariable id    -> 
	    (try
	       let contra,co = List.assq id param_rel_lifting
	       in (match variance,contra,co with
		     | true,_,Some pex ->
			 Formula(Application(pex, Tuple([expr1; expr2])))
		     | false,Some pex,_ ->
			 Formula(Application(pex, Tuple([expr1; expr2])))
		     | _ -> Equal(expr1, expr2)
		  )
	     with
	       | Not_found -> Equal(expr1, expr2)
	    )
      | Self
      | Carrier -> 
	    let contra,co = self_lifting
	    in (match variance,contra,co with
		  | true,_,Some pex ->
		      Formula(Application(pex, Tuple([expr1; expr2])))
		  | false,Some pex,_ ->
		      Formula(Application(pex, Tuple([expr1; expr2])))
				(* self1 and self2 have different types 
				 * so equality does not work here
				 *)
		  | _ -> assert(false)
	       )

      | Bool -> Equal(expr1, expr2)

      | Function(dom,codom) -> 
					(* do currying here *)
	  let arglist = match dom with
	    | Product tl -> tl
	    | t -> [t] in
	    (* make new name space for quantification *)
	  let sub_ns = sub_space name_space in 
	  let (typed_args1, typed_args2) = create_id_pairs sub_ns arglist in
	  let args1 =	(* we do not want to mix up one tuples *)
	    match dom with
	      | Product _ -> Tuple(List.map (fun (v,_) -> Term(v,Always,[])) 
				     typed_args1)
	      | _ -> Term(fst (List.hd typed_args1), Always,[]) in
	  let args2 =	
	    match dom with
	      | Product _ -> Tuple(List.map (fun (v,_) -> Term(v,Always,[])) 
				     typed_args2)
	      | _ -> Term(fst (List.hd typed_args2), Always,[]) 
	  in
	    Forall(
	      (subst1 typed_args1) @ (subst2 typed_args2), 
	      Implies( 
		And( List.map2 
		       (fun (v1,arg_typ1) (v2,arg_typ2) ->
			    (* arg_typ1 = arg_typ2 is ensured by
			     * create_id_pairs
			     *)
			  rel_lift_type sub_ns (not variance)
			    arg_typ1 
			    (Term(v1,Always,[])) 
			    (Term(v2,Always,[])))
		       typed_args1 typed_args2),
		(rel_lift_type sub_ns variance codom
		   (Application(expr1,args1)) 
		   (Application(expr2,args2)))))
      | Product(type_list)-> 
	  let tuple_len = List.length type_list in
			(* count the arguments to get the projection numbers *)
	  let i = ref 0 in
	      And(		
	      	List.map (fun p_type  -> 
			    incr(i);
			    rel_lift_type name_space variance
			      p_type 
			      (Application(Projection(!i, tuple_len),expr1)) 
			      (Application(Projection(!i, tuple_len),expr2)))
				  type_list)
		(* do records as an exeption *)
      | Record label_list ->
	    And(		
	      List.map (fun (label_name, label_type) -> 
			  rel_lift_type name_space variance
			    label_type (RecordSelection(label_name,expr1))
			    (RecordSelection(label_name,expr2)))
				  label_list)

					(* not in ccsl input types *)
      | TypeConstant _
      | FreeTypeVariable _ 
      | Predtype _
      | IFace _
      | Array _ 	      
      | SmartFunction _ -> assert(false)
  in
    rel_lift_type top_name_space variance typ expr1 expr2
  

and pre_argument_list_rellift
  eq_types
  self1_subst self2_subst
  (parameter_rel_lifting : ('cl,'mem) param_lifting_type)
  (full_self_lifting : ('cl,'mem) self_lifting_type) 
  top_name_space 
  variance
  parameter_list argument_list = 

  let subst1 = List.map (fun (v,typ) -> (v, self1_subst typ)) in
  let subst2 = List.map (fun (v,typ) -> (v, self2_subst typ)) in

  let non_trivial_rel_lift param_rel_lifting self_lifting variance typ =
    let sub_ns = sub_space top_name_space in
    let (var1,var2) = (create_id_pairs sub_ns [typ] ) 
    in
      Abstraction(
	(subst1 var1) @ (subst2 var2),
	Expression
	  (pre_rellift
	     eq_types
	     self1_subst
	     self2_subst
	     param_rel_lifting
	     self_lifting
	     sub_ns variance typ 
	     (Term( fst(List.hd var1), Always,[]))
	     (Term( fst(List.hd var2), Always,[]))
	  ))
  in
              (* make predicates for arguments of class or adt components *)
    List.fold_right
      (fun (typ,var) accu -> match var with
         | Unused ->
             (non_trivial_rel_lift 
		(extend_param_rel_lifting parameter_rel_lifting)
		(extend_self_rel_lifting full_self_lifting)
		variance typ
             ) :: accu
         | Pos -> 
             (non_trivial_rel_lift parameter_rel_lifting full_self_lifting 
		variance typ
             ) :: accu
         | Neg -> 
             (non_trivial_rel_lift parameter_rel_lifting full_self_lifting 
		(not variance) typ
             ) :: accu
         | Mixed ->
             (non_trivial_rel_lift parameter_rel_lifting full_self_lifting 
		(not variance) typ) ::
             (non_trivial_rel_lift parameter_rel_lifting full_self_lifting 
		variance typ) 
             :: accu
         | _ -> assert(false)
      )
                                (* produce a list of type * variance pairs *)
      (List.combine
         (types_from_arguments argument_list)
         (List.map (function TypeParameter id ->
                      Logic_util.make_simple id.id_variance)
            parameter_list)
      )
      []


and pre_inline_class_rel_lifting eq_types self1_subst self2_subst
  param_rel_lifting self_lifting top_name_space variance cl args 
  expr1 expr2
  = 
  let class_subst = 
    substitute_types_only eq_types 
      (make_substitution cl#get_parameters args) in

  let subst1 = List.map (fun (v,typ) -> (v, self1_subst typ)) in
  let subst2 = List.map (fun (v,typ) -> (v, self2_subst typ)) in

  let do_method meth =
    let dom = meth#get_domain in
    let name = meth#get_name in
					(* do currying here *)
    let arglist = List.map class_subst (method_arg_list meth) in
				(* make new name space on quantification *)
    let sub_ns = sub_space top_name_space in
    let (typed_args1,typed_args2) = create_id_pairs sub_ns arglist in
    let args1 =			(* we do not want to mix up one tuples *)
      match dom with
	| Product _ -> Tuple( expr1 ::
			     (List.map (fun (v,_) -> Term(v,Always,[])) 
				typed_args1))
	| _ -> expr1 in
    let args2 = 
      match dom with
	| Product _ -> Tuple(expr2 :: 
			       (List.map (fun (v,_) -> Term(v,Always,[])) 
				  typed_args2))
	| _ -> expr2 
    in let form =
	Forall(
	  (subst1 typed_args1) @ (subst2 typed_args2),
	  Implies( 
	    And( List.map2 
		   (fun (v1,arg_typ1) (v2,arg_type2) ->
				(* arg_typ1 = arg_typ2 via create_id_pairs ! *)
		      pre_rellift 
			eq_types
			self1_subst
			self2_subst
			param_rel_lifting 
			self_lifting
			sub_ns
			(not variance)
			arg_typ1 
			(Term(v1,Always,[])) 
			(Term(v2,Always,[]))
		   )
		   typed_args1 typed_args2),
	    (pre_rellift
	       eq_types
	       self1_subst 
	       self2_subst
	       param_rel_lifting
	       self_lifting
	       sub_ns
	       variance
	       (class_subst meth#get_codomain)
	       (Application(
		  Application(Term(name,Always,[]), 
			      cl#get_model_coalgebra),
		  args1))
	       (Application(
		  Application(Term(name,Always,[]), 
			      cl#get_model_coalgebra),
		  args2))
	    )))
    in
      form
  in
    And(List.map do_method cl#get_all_sig_actions)


and pre_inline_adt_rel_lifting eq_types self1_subst self2_subst
  param_rel_lifting self_lifting top_name_space variance adt args 
  expr1 expr2
  = 
  if constant_arg_list (fun id -> List.mem_assoc id param_rel_lifting) args
  then 
    Expression(Equal(expr1, expr2))
  else
    let adt_subst = 
      substitute_types_only eq_types 
	(make_substitution adt#get_parameters args) in
    let do_constructor constr =
      let sub_ns = sub_space top_name_space in
      let arglist = List.map adt_subst (member_arg_list constr) in
			  (* variables for constructor arguments left side *)
      let vars_left = create_ids sub_ns arglist in
					  (* variable for whole right side *)
      let body =
	Expression
	  (And
	     (Formula(
		Application(
		  Term(constr#get_recognizer#get_name,Always,[]),
		  expr2))
	      ::
		List.map2
		(fun (l_var,l_typ) r_acc -> 
		   pre_rellift
		     eq_types
		     self1_subst 
		     self2_subst
		     param_rel_lifting 
		     self_lifting
		     sub_ns
		     variance
		     l_typ 
		     (Term(l_var,Always,[]))
		     (Application(
			Term(r_acc#get_name,Always,[]),
			expr2))
		)
		vars_left
		constr#get_accessors
	     ))
      in let case = 
	  if constr#get_sort = Adt_Const_Constructor 
	  then (Term(constr#get_name,Always,[]), body)
	  else 
	    (SmartApplication(
	       Term(constr#get_name,Always,[]),
	       (List.map (fun (v,t) -> Term(v,Always,[])) vars_left)) ,
	     body)
      in
	case
    in 
      Case(expr1, List.map do_constructor adt#get_adt_constructors)



(* the exported version : 
 *   optimize the output of relation lifting 
 *)
let fullrellift eq_types self1_subst self2_subst param_rel_lifting 
  self_lifting top_name_space typ expr1 expr2 
  =
  opt_formula(
    pre_rellift eq_types 
		self1_subst self2_subst param_rel_lifting self_lifting
		top_name_space true typ expr1 expr2
  )


let argument_list_fullrellift eq_types 
  self1_subst self2_subst param_rel_lifting 
  self_lifting top_name_space parameter_list argument_list
  =
  List.map opt_expression
    (pre_argument_list_rellift
       eq_types
       self1_subst self2_subst
       param_rel_lifting
       self_lifting
       top_name_space 
       true
       parameter_list 
       argument_list)



(**************************************************************************
 *
 * exported version of inlined adt relation lifting 
 * 
 *     eq_types           -> equality on types
 *     self1_subst        -> substitution function for Self for selfvar1
 *     self2_subst        -> substitution function for Self for selfvar2
 *     param_lifting	  -> lifting for type parameters
 *     self_lifting	  -> lifting for self
 *     top_name_space     -> name space for name creation
 *     adt		  -> the adt to lift
 *     expr1, expr2       -> the expressions
 *)

let inline_adt_rel_lifting eq_types self1_subst self2_subst
  param_rel_lifting self_lifting top_name_space adt 
  expr1 expr2 
  = 
  opt_expression(
    pre_inline_adt_rel_lifting eq_types self1_subst self2_subst
		   param_rel_lifting self_lifting top_name_space true adt 
		   (arguments_from_parameters adt#get_parameters)
		   expr1 expr2)




(* 
 *   hide the lifting for parameters
 *)
let rellift eq_types self1_subst self2_subst self_lifting top_name_space 
  typ expr1 expr2 =
    (* define this here as long as we don't have full liftings *)
  let param_rel_lifting = []
  in
    fullrellift eq_types self1_subst self2_subst param_rel_lifting self_lifting
      top_name_space typ expr1 expr2



(* special relation lifting for whole methods (full lifting)
 * parameters
 *     param_lifting	  -> lifting for type parameters
 *     self_lifting	  -> lifting for self
 *     name_space         -> name space for name creation
 *     self1_subst        -> substitution function for Self for selfvar1
 *     self2_subst        -> substitution function for Self for selfvar2
 *     coalgebra1
 *     coalgebra2         -> the coalgebras for getting the methods right
 *     meth		  -> the method
 *     selfvar1
 *     selfvar2           -> the variables of Self, to which we apply
 *           		     the lifting
 * 
 * This function contains a slightly altered version of the 
 * Function case of the general lifting obove. The special thing 
 * is that we massage domain and codomain of each method such that 
 * the coalgebra and the selfvar argument get merged, and the method 
 * is in the standard form m(c)(args).
 *)


let class_method_full_rel_lifting
  eq_types
  self1_subst self2_subst
  param_rel_lifting 
  self_lifting name_space 
  coalgebra1 coalgebra2 
  meth 
  selfvar1 selfvar2 =

  (* we need substitutions for self1 and self2, for 
   * convenience we construct the following utility functions 
   * such that they can be applied to a (string * top_types) list,
   * where the substitution is only applied to the types
   *)
  let subst1 = List.map (fun (v,typ) -> (v, self1_subst typ)) in
  let subst2 = List.map (fun (v,typ) -> (v, self2_subst typ)) in

  let dom = meth#get_domain in
  let codom = meth#get_codomain in 
  let name = meth#get_name in
					(* do currying here *)
  let arglist = method_arg_list meth in
				(* make new name space on quantification *)
  let sub_ns = sub_space name_space in
  let (typed_args1,typed_args2) = create_id_pairs sub_ns arglist in
  let args1 = (* we do not want to mix up one tuples *)
    match dom with
      | Product _ -> Tuple(Term(selfvar1,Always,[]) :: 
			     (List.map (fun (v,_) -> Term(v,Always,[])) 
				typed_args1))
      | _ -> Term(selfvar1, Always,[]) in
  let args2 = 
    match dom with
      | Product _ -> Tuple(Term(selfvar2,Always,[]) :: 
			     (List.map (fun (v,_) -> Term(v,Always,[])) 
				typed_args2))
      | _ -> Term(selfvar2, Always,[]) 
  in let form =
      Forall(
	(subst1 typed_args1) @ (subst2 typed_args2),
	Implies( 
	  And( List.map2 
		 (fun (v1,arg_typ1) (v2,arg_type2) ->
				(* arg_typ1 = arg_typ2 via create_id_pairs ! *)
		    pre_rellift 
		      eq_types
		      self1_subst
		      self2_subst
		      param_rel_lifting 
		      self_lifting
		      sub_ns
		      false
		      arg_typ1 
		      (Term(v1,Always,[])) 
		      (Term(v2,Always,[]))
		 )
		 typed_args1 typed_args2),
	  (pre_rellift 
	     eq_types
	     self1_subst 
	     self2_subst
	     param_rel_lifting
	     self_lifting
	     sub_ns
	     true
	     codom
	     (Application(
		Application(Term(name,Always,[]), coalgebra1),
		args1))
	     (Application(
		Application(Term(name,Always,[]), coalgebra2),
		args2))
	  )))
  in
    opt_formula form



(***************************************************************** 
 * relation lifting for adt's, we do the whole adt here and 
 * generate the arguments for reduce
 * 
 * Because of this higher order definition of the lifting, 
 * substitutions play a important role, and are quite difficult.
 * To avoid to much code duplication, this fuction computes the 
 * substitution for the adt relation lifting. It is exported, 
 * because the substitutions are needed, to generate precise 
 * importings.
 * 
 * arguments
 * 	   adt		the adt
 * 	   typed_args1  	parameter list 1
 * 	   typed_args2  	parameter list 2
 * result 
 * 	   a pair of substitution functions
 *)

let adt_rel_lifting_substitutions adt type_param1 type_param2 =
  let adt_type_parameters = adt#get_parameters in
  let _ = assert((List.length type_param1) = 
		   (List.length adt_type_parameters)) in
  let _ = assert((List.length type_param2) = 
		   (List.length adt_type_parameters)) in
  let self2_pre_subst = 
    make_substitution_param_param adt_type_parameters type_param2 in
  let adt2_type = 
    ccsl_substitute_types self2_pre_subst 
                 (* the following is similar to this_adt_type, but 
		  * this method exists only for theories, not for ifaces 
		  *)
      (Adt(adt, Always, 
	  arguments_from_parameters adt_type_parameters)) in
  let self2_subst = 
    ((Carrier, adt2_type) :: self2_pre_subst) in
  let self1_subst = 
    ((Carrier, Function(adt2_type, Bool))
       :: (make_substitution_param_param adt_type_parameters type_param1))
  in
    self1_subst, self2_subst


(* here comes the real lifting
 * 
 * arguments
 *   ns 			name space
 *   type_param1		first set of type parameters
 *   type_param2		second set of type parameters
 *   adt			the adt for which the lifting is done
 *   param_rel_assoc	an assoc list for parameter relations,
 * 			   this should associate original parameters 
 * 			   from the adt with relations. Equal is taken
 * 			   for missing associations.
 * 
 * The result is a list of expressions, which can be applied to a 
 * reduce.
 * 
 *)

let adt_rel_lifting type_param1 type_param2 param_rel_lifting ns adt  =
    (* all of type_param, type_param2, and param_rel_assoc should have 
     * the same length as the type parameter list of the adt. Here we 
     * check, that param_rel_assoc has the same length as type_param1, 
     * the rest is checked in the subroutine 
     * adt_rel_lifting_substitutions.
     *)
  let _ = assert((List.length param_rel_lifting) =
		   (List.length type_param1)) in
					(* this is never really used *)
  let (self1_subst, self2_subst) = 
    match adt_rel_lifting_substitutions adt type_param1 type_param2 with
	l,r -> ccsl_substitute_types l, ccsl_substitute_types r in
  let adt1_type = self1_subst Carrier in
  let adt2_type = self2_subst Carrier in
    (* the lifting Carrier for relation lifting for Adt's is 
     * function application, see my thesis
     *)
  let eval = 
    Abstraction(
      [("f", adt1_type);
       ("x", adt2_type)],
      Application(Term("f",Always,[]), Term("x", Always,[]))
    ) in
  let self_lifting = (None, Some eval)
  in
				(* return the list of reduce arguments *)
    List.map
      (fun m ->
	 let sub_ns = sub_space ns in
	 let arglist = member_arg_list m in
			(* variables for constructor arguments left side *)
	 let vars_left = create_ids sub_ns arglist in
					(* variable for whole right side *)
	 let var_right = create_one_id sub_ns adt2_type in
	 let body =
	   Abstraction(
			(* adt2_type has undergone substitution already *)
	     [(var_right, adt2_type)],
		(* Uli used cases first, then we took if's, 
		 * I take a simple conjunction now 
		 *)
	     Expression(
	       And(
		 Formula(
		   Application(
		     Term(m#get_recognizer#get_name,Always,[]),
		     Term(var_right,Always,[])))
		   ::
		   List.map2
		     (fun (l_var,l_typ) r_acc -> 
			pre_rellift
			  eq_ccsl_types
			  self1_subst 
			  self2_subst
			  param_rel_lifting 
			  self_lifting
			  sub_ns
			  true
			  l_typ 
			  (Term(l_var,Always,[]))
			  (Application(
			     Term(r_acc#get_name,Always,[]),
			     Term(var_right,Always,[])))
		     )
		     vars_left
		     m#get_accessors
	       )))
	 in let red_fun = 
	     if m#get_sort = Adt_Const_Constructor 
	     then body
	     else 
	       SmartAbstraction(
			(* remember, that we use the higher order approach *)
		 (List.map (fun (v,t) -> 
			      (v, self1_subst t))
	      	    vars_left),
		 body)
	 in
	   opt_expression red_fun
      )
      adt#get_adt_constructors


(***********************************************************************
 ***********************************************************************
 *
 *  Here, we generate the parameter information and the naming 
 *  information for the single_step_transitions needed for inductive
 *  Diamond definitions
 *
 *  maybe this function should be shifted into another file
 *
 *  Arguments:
 *  single_step_type2parameter 
 *     typ --- the type of a method that has to be considered
 *
 *  Result is a list of ordered pairs where the first component is
 *  a type (a product, usually), the second one is a string containing
 *  the appropriate name for the type 
 *  names will be generated as follows: if type is
 *     a constant --> no name ( empty list returned   )
 *     Self       --> no name but list returned 
 *                    contains a pair of an empty product and empty string
 *     Product(t1,..,tn) 
 *                --> for each of the t1,...,tn the result lists appended
 *                    and the strings augmented with 'p1' ... 'pn' in the
 *                    first position
 *     Function(t1,t2)
 *                --> result of t2 augmented with an additional element t1
 *                    in the product of the first component
 *     All others: ignored, i.e. treated as constant   
 *)

let single_step_type2parameter dom_type codom_type acc_fn =

  let rec pre_single_step_type2parameter ctyp =
    match ctyp with
      | Self -> [([], "",[],(fun x -> x))]
      | Bool -> []
      | Function(t1,t2) ->
	  let l1 = pre_single_step_type2parameter t2 in
	  let adjust_list l =
	    if List.length(l) = 1 then
	      match l with
		| [([],s,[],succ_fun)] -> [([],"",[],succ_fun)]
		| _ -> l
	    else
	      l
	  in
	    List.map 
	      (fun (t,name,accessors,succ_fun) -> 
		 match t1 with
		   | Product(pl) -> 
		       let new_accessors = ref (acc_fn pl) in
			 ( pl@t,
			   name,
			   (!new_accessors)@accessors,
			   (fun x ->
			      Application(
				succ_fun x,
				Tuple(!new_accessors)
			      )
			   )
			 )
		   | tn ->
		       let new_accessors = ref (acc_fn [tn]) in
			 ( tn::t, name,
			   (!new_accessors)@accessors,
			   (fun x ->
			      Application(
				succ_fun x,
				Tuple(!new_accessors)
			      )
			   )
			 )			     
	      )
	      (adjust_list l1)
      | Product(tl) -> 
	  let augment l n = 
	    List.fold_right
 	      (fun (t ,name, acc, succ_fun) l ->
		 (
		   ( t, 
		     name ^ "_p" ^ (string_of_int n),
		     acc,
		     (
		       fun x ->
			 Application(
(* Jan: can you insert the length of the tuple here ?
*)
			   Projection(n,-1),
			   succ_fun x
			 )
		     )
		   )::l))
 	      l
 	      []
	      
	  in
	  let augment_and_flatten list_for_tl = 
	    snd(
 	      List.fold_right
 		  (fun l1 (n,l2) ->
 		     (n-1,(augment l1 n) @ l2))
 		  (list_for_tl)
 	      (List.length(list_for_tl),[])
	    )
	  in
	    augment_and_flatten
	      (List.map pre_single_step_type2parameter tl) 
      | TypeConstant _
      | BoundTypeVariable _ 
      | Class _
      | IFace _
      | Adt _ 
      | Groundtype _ -> []
					(* the rest barfs *)
      | Carrier 
      | SmartFunction _
      | FreeTypeVariable _
      | Record _
      | Array _
      | Predtype _ -> assert(false)
  in 
  let augment_with_domain (types,name,accessors,succ_fun) =
    match dom_type with
      | Product(tl) -> 
	  let new_accessors = ref (acc_fn tl) in
	    ( 
	      tl@types,
	      name,
	      (!new_accessors)@accessors,
	      (fun mem_expr state_expr ->
		 (succ_fun
		    ( Application(
			mem_expr,
			Tuple(
			  state_expr::(!new_accessors)
			)
		      )
		    )
		 )
	      )
	    )
      | _ -> assert(false)
  in
    List.map
      augment_with_domain
      (pre_single_step_type2parameter codom_type)

(***********************************************************************
 ***********************************************************************
 *
 *  Here, we generate the constructor declarations
 *
 *  step_constructor_declarations ns l
 *  takes arguments
 *    ns  --> the namespace
 *    cl  --> the class description for wich the adt is generated
 *
 * delivers a list of constructor declarations
 *)

let step_constructor_declarations ns cl =
  let sub_ns = sub_space ns in
  let accessors tl = List.map 
		       (fun (s,t) -> Term(s,Never,[]))
		       (create_ids sub_ns tl) 
  in 
  let recognizer na =  
    match !output_mode with 
      | Pvs_mode -> na ^ "?"
      | Isa_mode -> "is_" ^ na
  in
  let terms2strings tl = 
    List.map 
      (fun t -> match t with
	 | Term(s,_,_) -> s
	 | _ -> assert(false)
      )
      tl
  in 
  let triple_for_path mname (t,n,accl,_)=
    (mname ^ n, List.combine (terms2strings accl) t, mname ^ (recognizer n))
  in
  let triples_for_method m =
    let dom = match m#get_domain with
      | Self -> Product([])
      | Product(Self::tl) -> Product(tl)
      | _ -> assert(false)
    in
      List.map (triple_for_path m#get_name) 
	(single_step_type2parameter dom m#get_codomain accessors)
  in
  let triples_for_methods ml =
    List.fold_right
      (fun m l ->
	 (triples_for_method m)@l)
      ml
      []

  in
    match !output_mode with
      | Pvs_mode ->
	  Datatypedecl("",[], triples_for_methods cl#get_all_methods)
      | Isa_mode ->
	  Datatypedecl(
	    ccsl_step_theory_name cl,
	    cl#get_parameters,
	    triples_for_methods cl#get_all_methods)

let admissible_cases_for_method ns mem check_list_term =
  let sub_ns = sub_space ns in
  let accessors tl = List.map 
		       (fun (s,t) -> Term(s,Never,[])) 
		       (create_ids sub_ns tl) 
  in
  let case_computation = 
    Application(
      Term("member",Always,[]),
      Tuple([
	      Term(mem#get_name,Always,[]);
	      check_list_term])
    )
  in
  let case_for_path (tl,pname,accl,_) = 
    match tl with
      | [] -> (Term(mem#get_name ^ pname,Never,[]),case_computation)
      | l -> (Application(
		Term(mem#get_name ^ pname,Never,[]),
		Tuple(accl)
	      ),
	      case_computation)
  in
  let dom =
    match mem#get_domain with
      | Self -> Product([])
      | Product(Self::tl) -> Product(tl)
      | _ -> assert(false)
  in
    List.map
      case_for_path
      (single_step_type2parameter dom mem#get_codomain accessors)

(********************************************************************
 *
 ********************************************************************) 
    
let transition_cases_for_method ns mem coalg_term x_term =
  let sub_ns = sub_space ns in
  let accessors tl = List.map 
		       (fun (s,t) -> Term(s,Never,[])) 
		       (create_ids sub_ns tl) 
  in
  let accessors2terms l =
    List.map 
      (fun s -> Term(s,Never,[]))
  in
  let mem_expr = 
    Application(
      Term(mem#get_name,Never,[]),
      coalg_term
    )
  in
  let dom = match mem#get_domain with
    | Self -> Product([])
    | Product(Self::tl) -> Product(tl)
    | _ -> assert(false)
  in
    List.map
      (fun (ty,st,accl,fn) ->
	 ( Application(
	     Term(mem#get_name^st,Never,[]),
	     Tuple(
	       accl
	     )
	   ),
	   fn mem_expr x_term
	 )
      )
      (single_step_type2parameter dom mem#get_codomain accessors)
      


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)
