(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 3.11.99 by Hendrik
 *
 * Time-stamp: <Tuesday 28 May 02 17:33:18 tews@ithif51>
 *
 * Semantics
 *
 * $Id: semantics_theory.ml,v 1.14 2002/05/30 13:10:31 tews Exp $
 *
 *)

open Util
open Global
open Top_variant_types
open Name_space 
open Names
open Classtypes
open Types_util
open Pvs_proof_util
open Theory_class 
open Pre_printing;;


(***********************************************************************
 ***********************************************************************
 *
 * Semantic theory
 *
 *)

class ['class_type, 'member_type] ccsl_pre_semantics_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_class_theory_class cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
      ([
	 name_of_coalgebra;
	 name_of_assert cl;
	 name_of_creation cl;
	 name_of_model cl
       ]
       @
       (List.map 
	  (fun ass -> name_of_assertion_predicate 
	       ass.assertion_name.token_name )
	  cl#get_assertions)
       @
       (List.map
	  (fun crea ->  name_of_creation_predicate
	       crea.assertion_name.token_name )
	  cl#get_creations)
      )
     
    method get_name = ccsl_semantics_theory_name cl

    method get_parameters = self#simple_parameters 

    initializer top_theory#override_file_name (ccsl_class_file_name cl)

    method private imports =
      Import 
	(
	  (if cl#has_feature HasBisimulationFeature
	   then 
	     if cl#has_feature HasGreatestBisimFeature
	     then
	       [(ccsl_private_bisim_eq_rewrite_theory_name cl, 
		 self#simple_arguments)]
	     else
	       [(ccsl_bisim_eq_theory_name cl, 
		 self#simple_arguments)]
	   else [])
	  @
	  (if cl#has_feature HasMorphismFeature
	   then
	     [ccsl_morphism_rewrite_theory_name cl, []]
	   else
	     []
	  )
	  @
	  [(ccsl_box_theory_name cl,
	    self#simple_arguments)] 
	  @
	  (if cl#get_rel_lifting_requests = [] then [] else
	     [(ccsl_req_bisim_theory_name cl, self#simple_arguments)])
	  @
	  (cl#get_assertion_imports)
	)

	
    (* print an assertion or a creation condition as predicate 
     * over base_decl, which is a pair of string and type for Abstraction
     * let_trick_list is a list of triples (string, typ, term),
     * if the string occurs in the assertion, a let is generated, which 
     * applies term to string
     * nameing_function is a function, that gives the name for 
     * the declaration
     * ass is the assertion itself
     * previous_option is an assertion option, it makes the assertion 
     * dependent on the previous one (for -dependent-assertions)
     *)
    method private do_ass_or_crea nameing_function 
                             let_trick_list ass base_decl previous_option = 
      let base_typ = snd(base_decl) in
      let decl_from_rec id = 
	id.id_token.token_name, id.id_type  in
      let string_occur name s =
	let length_name = String.length name in
	let length_s = (String.length s) - length_name +1 in
	let res = ref false in
	let i = ref 0 in
	  while (!i < length_s) && (not !res) do
	    try
	      begin
	      	i := String.index_from s !i name.[0];
		if !i < length_s then 
	      	  res := (name = String.sub s !i length_name);
	      	incr(i)
	      end
	    with
	      | Not_found -> i := length_s
	      | ex -> raise ex
	  done;
	  !res
      in

      (* 	     begin
       * 	       for i = 0 to (String.length s) - length_m
       * 	       do
       * 		 res := !res or (name = String.sub s i length_m)
       * 		 done;
       * 	       !res
       * 	     end in
       *)
      let filter_let_trick_list text_formula =
	option_filter
	  (List.map
	     (fun ((name,_,_) as x) -> 
		if string_occur name text_formula
	      	then Some x else None
	     )
	     let_trick_list) in
      let let_trick text_formula = 
	Formula(
	  Let( (List.map 
	       	  (fun (name,typ,term) ->
		     (id_record_from_string name, 
		      Some(typ), 
		      Application(Term(name,Always,[]),
			       	  term)))
	       	  (filter_let_trick_list text_formula)),
	       Expression(ConstantPredicate(text_formula))))
      in let inner_ex =
		  Forall(
		    List.map decl_from_rec ass.free_variables,
		    match ass.assertion_formula with
		      | Symbolic f -> 
			  ccsl_pre_pretty_formula cl
			    self#get_member_fun self#get_iface_fun
			    f
		      | Pvs_String f -> 
			  assert(!output_mode = Pvs_mode); let_trick f
		      | Isa_String f -> 
			  assert(!output_mode = Isa_mode); let_trick f
		  )
      in
      	Defn( nameing_function ass.assertion_name.token_name,
	      [[Declared(name_of_coalgebra, self#coalgebra_type)]],
	      Function(base_typ, Bool),
	      Abstraction(
		[base_decl],
		Expression(
		  match previous_option with 
		    | None -> inner_ex
		    | Some pre_ass ->
			And[
			  Formula(
			    Application(
			      Application(
				Term(nameing_function 
				       pre_ass.assertion_name.token_name,
				       Always, []),
				coalgebra_term),
			      Term(fst base_decl, Always, [])));
			  inner_ex]
		)))
	      
    method private do_all_assertions =
      let let_trick_list =
	[
	  (name_of_private_bisim_eq cl,
	   Function(Product[Self;Self], Bool),
	   coalgebra_term);
	  (name_of_public_bisim_eq cl,
	   Function(Product[Self;Self], Bool),
	   coalgebra_term);
	  ("Box?",
	   Function(
	     Function(Self, Bool),
	     Function(Self, Bool)),
	   coalgebra_term);
	  ("Diamond?",
	   Function(
	     Function(Self, Bool),
	     Function(Self, Bool)),
	  coalgebra_term)
	]
	@
	(List.map
	   (fun m -> (m#get_name, 
		      Function(m#get_domain,m#get_codomain),
		      coalgebra_term))
	   cl#get_all_actions)
      in
      let previous_ass = ref None 
      in
      	(List.map 
	   (fun ass -> 
	      let base_decl = 
	      	match ass.self_variable with
	      	  | None -> assert(false)
	      	  | Some id -> (id.id_token.token_name,
			      	id.id_type)
	      in let decl = self#do_ass_or_crea name_of_assertion_predicate
			      let_trick_list ass base_decl !previous_ass;
	      in
		if !dependent_assertions then
		  previous_ass := Some ass;
		decl
	   )
	   cl#get_assertions)

    method private do_all_creations =
      let let_trick_list =
	[
	  (name_of_private_bisim_eq cl,
	   Function(Product[Self;Self], Bool),
	   coalgebra_term);
	  (name_of_public_bisim_eq cl,
	   Function(Product[Self;Self], Bool),
	   coalgebra_term);
	  ("Box?",
	   Function(
	     Function(Self, Bool),
	     Function(Self, Bool)),
	   coalgebra_term);
	  ("Diamond?",
	   Function(
	     Function(Self, Bool),
	     Function(Self, Bool)),
	  coalgebra_term)
	]
	@
	(List.map
	   (fun m -> 
	      if m#is_action 
	      then  (m#get_name, 
		     Function(m#get_domain,m#get_codomain),
		     coalgebra_term)
	      else match m#get_sort with
		| Defined_Method ->
		    (m#get_name, 
		     m#get_full_type,
		     coalgebra_term)
		| Var_Constructor ->
		    (m#get_name, 
		     m#get_full_type,
		     algebra_term)
		| Const_Constructor -> 
		    (m#get_name, 
		     m#get_full_type,
		     algebra_term)

		| Proper_Attribute _
		| Normal_Method
		| Update_Method
		| Adt_Const_Constructor
		| Adt_Var_Constructor 
		| Adt_Accessor 
		| Adt_Recognizer | Adt_Reduce 
		| Class_Coreduce | Class_Special
		| GroundTerm | InfixGroundTerm
		    -> assert(false)
	   )
	   (cl#get_all_actions @ cl#get_constructors))
      in
      	(List.map 
	   (fun crea -> 
	      let base_decl = (name_of_algebra, self#algebra_type)
	      in
	      	self#do_ass_or_crea name_of_creation_predicate
		  let_trick_list crea base_decl None
	   )
	   cl#get_creations)
	
    method private do_assert =
      let sub_ns = sub_space ns in
      let x = create_one_id sub_ns Self in
      let body =
	(* special isabelle hack: If there are no assertions 
	 * the definition
	 *       SeqAssert c == ((ALL (x :: 'Self) .  (True)))
	 * yields an 
	 *   *** Extra type variables on rhs: 'Self
	 * error. Isabelle has no empty types, so the quantification
	 * is not necessary.
	 *)
	if (!output_mode = Isa_mode) && (cl#get_assertions = [])
	then True
	else
	  Forall(
	    [x,Self],
	    And(
	      List.map
		  (fun ass ->
		     Formula(
		       Application(
			 Application(
			   Term(name_of_assertion_predicate
				  ass.assertion_name.token_name, 
				  Always, []),
			   coalgebra_term),
			 Term(x,Always,[]))))
		  cl#get_assertions
	    ))
      in		
      	Defn(name_of_assert cl, 
	   [[Declared(name_of_coalgebra, self#coalgebra_type)]],
	   Bool,
	   Expression(	   
	     And(
	       List.map
		   (fun ianc ->
		      Formula(
			Application(
			  Term(name_of_assert ianc, Always, []),
			  Application(Term(super_access_method(ianc),
					   Always,[]), 
				      coalgebra_term)
			)))
		   cl#get_resolved_ancestors
	       @			(* own assertions *)
	       [body]
	     )))


    method private do_method_wise_assert =
      let sub_ns = sub_space ns in 
      let (decl_list, coalgebra_from_tuple ) = self#coalgebra_as_tuple sub_ns
      in
	Defn(name_of_method_assert cl,
	     [decl_list],
	     Bool,
	     Application(
	       Term(name_of_assert cl, Always, []),
	       coalgebra_from_tuple
	     ))


    method private do_create =
      Defn(name_of_creation cl, 
	   [[Declared(name_of_coalgebra, self#coalgebra_type)]],
	   Function(self#algebra_type, Bool),
	   Abstraction(
	     [name_of_algebra,self#algebra_type],
	     Expression(
	       And(List.map
		     (fun crea ->
		      	Formula(
			  Application(
			    Application(
			      Term(name_of_assertion_predicate
				     crea.assertion_name.token_name,
				     Always, []),
			      coalgebra_term),
			    algebra_term)))
			  cl#get_creations
		     ))))


    method private do_model =
      let sub_ns = sub_space ns in
	Defn(name_of_model cl,
	     [Undeclared(name_of_coalgebra, self#coalgebra_type) ::
	      if cl#has_constructors 
	      then [Undeclared(name_of_algebra, self#algebra_type)]
	      else []],
	     Bool,
	       Expression(
		 And(
		   Formula(
		     Application(Term(name_of_assert cl,Always,[]),
				 coalgebra_term))::
		   if cl#has_constructors 
		   then [
		     Formula(
		       Application(
			 Application(Term(name_of_creation cl,Always,[]),
				     coalgebra_term),
			 algebra_term))]
		   else [])))

    method make_body =
      [
	self#imports;
	self#coalgebra_decl
      ]
      @
      self#do_all_assertions
      @
      self#do_assert :: 
	self#do_method_wise_assert :: 
	self#do_all_creations
      @
      (if cl#has_constructors then [self#do_create] else [])
      @
      [self#do_model] 

    method get_proofs = []
end

class ccsl_semantics_theory cl = 
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_semantics_theory 
  cl eq_ccsl_types


(***********************************************************************
 ***********************************************************************
 *
 * Basic theory
 *
 *)

class ['class_type, 'member_type] ccsl_pre_basic_theory 
  (cl : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)

  : ['class_type, 'member_type] ccsl_pre_class_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_class_theory_class cl eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns []
           
    method get_name = ccsl_basic_theory_name cl

    method get_parameters = self#simple_parameters 

    initializer top_theory#override_file_name (ccsl_class_file_name cl)

    method private imports =
      Import [ccsl_semantics_theory_name cl, self#simple_arguments]

    method private do_inherited_assert_lemmas = 
         (* First built the proof. The proof is identical for all
          * super classes.
	  *)
      let proof = 
	make_simple_proof	(* make a simple proof without branching *)
	  [
	    skosimp_star;			(* (SKOSIMP* ) *)
	    expand (name_of_assert cl);		(* (EXPAND "<class>Assert?") *)
	    pvs_assert;				(* (ASSERT) *)
	  ]
      in
	 (* The following function lemma computes the inherit assert 
	  * lemma for its argument ianc, a super class of the current 
	  * class
	  *)
      let lemma ianc =
	Lemma(					(* its a lemma (not a type) *)
	  name_of_inherited_assert_lemma ianc,	(* get its name *)
	  Forall(				(* forall coalgebras c *)
	    [(name_of_coalgebra, self#coalgebra_type)],
	    Implies(
	      self#assert_coalgebra_hypothesis,	(* Assert?(c) IMPLIES *)
	      Formula(
		Application(			(* Assert?(super(c)) *)
		  Term(name_of_assert ianc, Always, []),
		  Application(Term(super_access_method(ianc),
				   Always,[]), 
			      coalgebra_term)
		)))))
      in
      List.map				(* iterate over all direct ancestors *)
	(fun ianc ->				(* iteration function *)
	   Proved(			(* Pack lemma together with proof *)
	     lemma ianc,
	     Anon_proof proof))
	cl#get_resolved_ancestors		(* get all direct ancestors *)


    method private assertion_lemma_proof ass number_of_ass previous_opt = 
      let ancestor_number = List.length cl#get_ancestors in
      let self_var = 
	match ass.self_variable with
	  | None -> assert(false)
	  | Some id -> id.id_token.token_name
      in let case_if_previous = match previous_opt with
	| None -> []
	| Some ass -> 
	    [case (Pvs_pretty.string_of_pvs_top_expression(
		     Application(
		       Application(
			 Term(name_of_assertion_predicate 
				ass.assertion_name.token_name, Always, []),
			 Term(pvs_skolem name_of_coalgebra, Always, [])),
		       Term(pvs_skolem self_var, Always, []))))
	    ]
      in let other_cases = match previous_opt with
	| None -> []
	| Some _ -> [make_simple_proof [
		       flatten_antecedent;
		       pvs_assert
		     ]]
      in let assert_if_previous = match previous_opt with
	| None -> []
	| Some _ -> [pvs_assert]
      in
	PTree( 
	  [ 
	    skolem_bang;
	    case (Pvs_pretty.string_of_pvs_top_expression
		    (Application(
		       Term(name_of_assert cl, Always, []),
		       Term(pvs_skolem name_of_coalgebra, Always, []))))
	  ],
	  [PTree
	     ([
		pvs_assert;
		expand (name_of_assert cl);
		flatten_antecedent;
		hide_all_but [-(ancestor_number +1); 1];
		inst_val (-1) (pvs_skolem self_var);
		flatten_antecedent;
		hide_all_but [- number_of_ass; 1];
		expand (name_of_assertion_predicate 
			  ass.assertion_name.token_name);
	      ]
	      @ (
		case_if_previous
	      ),
	      (make_simple_proof
		 (assert_if_previous
		  @
		  [
		    inst_question;
		    pvs_assert;
		  ]
		 )
	      ) :: (
		other_cases
	      )
	     );
	   make_simple_proof
	     [pvs_assert]
	  ])

    method private do_assertion_lemma ass proof =
      let form =
	match ass.assertion_formula with
	  | Pvs_String _ 
	  | Isa_String _ ->
	      let sub_ns = sub_space ns in
	      let x = create_one_id sub_ns Self in
		Forall( [name_of_coalgebra, self#assert_coalgebra_type;
			 x,Self],
			Formula(
			  Application(
			    Application(
			      Term(name_of_assertion_predicate 
				     ass.assertion_name.token_name,
				     Always,[]),
			      coalgebra_term),
			    Term(x,Always,[]))))
	  | Symbolic f -> 
	      let decl_from_rec id = 
		id.id_token.token_name, id.id_type  in
	      let x_decl = 
	      	match ass.self_variable with
	      	  | None -> assert(false)
	      	  | Some id -> (id.id_token.token_name,
			      	id.id_type) in
	      let pre_pretty_ass = 
		ccsl_pre_pretty_formula cl 
		  self#get_member_fun self#get_iface_fun
		  f
	      in 
		(* For PVS snip the first forall off the assertion 
		   and merge it with the quantifier for the coalgebra
		*)
	      let (all_decl, all_body) = match pre_pretty_ass with
		| Forall(decllist, body) -> (decllist, body)
		| x -> ([], x)
	      in
		match !output_mode with
		  | Pvs_mode -> 
		      Forall(
			(name_of_coalgebra, self#coalgebra_type) ::
			x_decl :: (List.map decl_from_rec ass.free_variables)
			@ all_decl,
			Implies(
			  self#assert_coalgebra_hypothesis,
			  all_body))
		  | Isa_mode -> 
		      MetaImplies(
			self#assert_coalgebra_hypothesis,
			pre_pretty_ass)
      in	
	Proved(
	  Lemma(
	    name_of_assertion_lemma ass.assertion_name.token_name,
	    form),
	  Anon_proof proof)

    method private do_assertion_lemmas = 
      let ass_count = ref 1 in
      let previous_ass = ref None 
      in
	List.map 
	  (fun ass -> 
	     let lemma = self#do_assertion_lemma ass 
			 (self#assertion_lemma_proof ass !ass_count
			    !previous_ass)
	     in
	       incr ass_count;
	       if !dependent_assertions then
		 previous_ass := Some ass;
	       lemma
	  )
	  cl#get_assertions


    method private creation_lemma_proof crea number_of_crea = 
      PTree( 
	[ 
	  skolem_bang;
	  case (Pvs_pretty.string_of_pvs_top_expression
		  (Application(
		     Term(name_of_assert cl, Always, []),
		     Term(pvs_skolem name_of_coalgebra, Always, []))))
	],
	[PTree(
	   [
	     case (Pvs_pretty.string_of_pvs_top_expression
		     (Application(
			Application(
			  Expression(ConstantPredicate(name_of_creation cl)),
			  Term(pvs_skolem name_of_coalgebra,Always,[])),
			Term(pvs_skolem name_of_algebra,Always,[]))))
	   ],
	   [make_simple_proof
	      [
		pvs_assert;
		expand (name_of_creation cl);
		flatten_antecedent;
		hide_all_but [-number_of_crea; 1];
		expand (name_of_creation_predicate 
			  crea.assertion_name.token_name);
		pvs_assert;
	      ];
	    make_simple_proof
	      [pvs_assert]
	   ]);
	 make_simple_proof
	   [pvs_assert]
	])


    method private do_creation_lemma crea proof =
      let form =
	     match crea.assertion_formula with
	       | Pvs_String _ 
	       | Isa_String _ ->
		   Forall( [name_of_coalgebra, self#assert_coalgebra_type;
			    name_of_algebra, self#assert_algebra_type],
			   Formula(
			     Application(
			       Application(
				 Term(name_of_creation_predicate 
				  	crea.assertion_name.token_name,
				  	Always,[]),
				 coalgebra_term),
			       algebra_term)))
	       | Symbolic f -> 
		   let decl_from_rec id = 
		     id.id_token.token_name, id.id_type  in
		   let pre_pretty_crea = 
		     ccsl_pre_pretty_formula cl
		       self#get_member_fun self#get_iface_fun
		       f
		   in 
		     match !output_mode with
		       | Pvs_mode -> 
			   Forall(
			     (name_of_coalgebra, self#coalgebra_type) ::
			     (name_of_algebra, self#algebra_type) ::
			     (List.map decl_from_rec crea.free_variables),
			     Implies(
			       And[self#assert_coalgebra_hypothesis;
				   self#creation_algebra_hypothesis],
			       pre_pretty_crea))
		       | Isa_mode -> 
			     MetaImplies(
			       self#assert_coalgebra_hypothesis,
			       MetaImplies(
				 self#creation_algebra_hypothesis,
				 pre_pretty_crea))
      in
	Proved(
	  Lemma(
	    name_of_assertion_lemma crea.assertion_name.token_name,
	    form),
	  Anon_proof proof)


    method private do_creation_lemmas =
      let crea_count = ref 1 
      in
	List.map 
	  (fun crea -> 
	     let lemma = self#do_creation_lemma crea 
			 (self#creation_lemma_proof crea !crea_count)
	     in
	       incr crea_count;
	       lemma
	  )
	  cl#get_creations


    method make_body =
      [
	self#imports;
      ]
      @
      self#do_inherited_assert_lemmas
      @
      self#do_assertion_lemmas
      @
      self#do_creation_lemmas

    method get_proofs = []
end

class ccsl_basic_theory cl = 
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_basic_theory cl eq_ccsl_types




(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

