(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 17.6.99 by Hendrik
 *
 * Time-stamp: <Monday 20 May 02 23:40:13 tews@ithif56.inf.tu-dresden.de>
 *
 * the theories for ground signatures
 *
 * $Id: sig_theory.ml,v 1.3 2002/05/22 13:42:43 tews Exp $
 *
 *)

open Global
open Name_space
open Top_variant_types
open Names
open Logic_util
open Classtypes 
open Types_util
open Theory_class
open Pre_printing



type ('cl,'mem) type_or_def = 
  | DefExt of ('cl,'mem) top_pre_definition_record
  | TypeDef of ('cl,'mem) top_pre_identifier_record_type
  
let get_sequence_number x = 
  let ret = match x with
    | DefExt def -> def.def_sequence
    | TypeDef id -> id.id_sequence
  in
    assert(ret >= 0);
    ret

let get_params = function
    | DefExt def -> get_definition_parameters def
    | TypeDef id -> get_ground_type_parameters id

let get_local_params = function
    | DefExt def -> def.defined_method#get_local_parameters
    | TypeDef id -> id.id_parameters


(********************************************************************** 
 *
 * definitions in ground signatures
 * 
 *)

class ['class_type, 'member_type] ccsl_pre_sig_def_theory 
  (si : 'class_type) 
  (* the equality relation for initializing the name space *)
  (eq_types : ('class_type, 'member_type) top_pre_types
     -> ('class_type, 'member_type) top_pre_types -> bool)
  (thname : string)
  (items : ('class_type, 'member_type) type_or_def list)
  (previous : string option)

  : ['class_type, 'member_type] ccsl_pre_sig_theory_type =
  object (self : 'self)

    inherit 
      ['class_type, 'member_type] ccsl_pre_sig_theory_class si eq_types
      as top_theory

       (* reserve all names, that we want to declare *)
    initializer reserve ns 
      [
	 (* definitions are reserved because all members are reserved *)
      ]

    method get_name = thname

    method get_parameters = get_params (List.hd items)

    initializer top_theory#override_file_name (ccsl_sig_file_name si)

    method get_proofs = []

    method private user_imports = si#get_iface_imports 

    method private component_imports = 
      List.fold_right
	(fun (v,c,args) accu -> 
	   (match c#get_kind with
	      | Spec_class -> 
		  if c#has_feature FinalSemanticsFeature
		  then (ccsl_final_theory_name c, args) :: accu
		  else (ccsl_loose_theory_name c, args) :: accu
	      | Spec_adt -> 
		  (match !output_mode with
		     | Isa_mode -> (ccsl_adtutil_theory_name c, args) :: accu
		     | Pvs_mode -> (ccsl_adt_theory_name c, args) :: accu
		  )
	      | Spec_sig -> 
		  if c#has_own_output
		  then			(* does have an own theory *)
		    (ccsl_sig_def_theory_name c, args) :: accu
		  else			(* no own theory for c *)
		    (if c#instanciate_importings && 
		       si#instanciate_importings
		     then
		       let subst = make_substitution c#get_parameters args 
		       in
			 (List.map 
			    (fun (name,args) ->
			       (name, 
				substitute_arguments_types_only
				  eq_types subst args)
			    )
			    c#get_iface_imports
			 )
		     else
		       List.map (fun (name,_) -> (name, [])) 
			 c#get_iface_imports
		    )
		    @ accu
	      | Spec_Spec -> assert(false)
	   ))
	si#get_components []
			  
    method private imports =
      match previous with
	| None -> Import(self#component_imports @ self#user_imports)
	| Some previous -> Import([previous, []])


    method private do_type_def idrec =
      Typedecl(get_ground_type_parameters idrec, 
	       idrec.id_token.token_name,
	       idrec.id_type)

    method private do_definition def =
      let rec do_abstraction innerex = function
	| [] -> innerex
	| vars :: rest ->
	    Abstraction(
	      List.map 
		(fun id -> (id.id_token.token_name,
			    id.id_type))
		vars,
	      do_abstraction innerex rest)
      in
      let mem = def.defined_method 
      in
	match def.definition with
	  | Symbolic eq ->
	      let pretty_eq =
		ccsl_pre_pretty_formula si
		  self#get_member_fun self#get_iface_fun eq
	      in 
		Defn(mem#get_name,
		     [],
		     mem#get_full_type,
		     do_abstraction
		       (Expression pretty_eq)
		       def.variables
		    )
	  | _ -> assert false


    method make_body =
      self#imports
      ::
	(List.map (function
		     | DefExt def -> self#do_definition def
		     | TypeDef id -> self#do_type_def id
		  )
	   items)

end

class ccsl_sig_def_theory si thname items = 
  [ccsl_iface_type, ccsl_member_type] ccsl_pre_sig_def_theory 
  si eq_ccsl_types thname items


let ccsl_sig_def_theories si =
(* *)					(* debugging support *)
  let signame = si#get_name in
(* *)
  let items = 
    (List.map (fun def -> DefExt def) si#get_definitions) @ 
    (List.map (fun td -> TypeDef td)
       (List.filter is_type_def si#get_all_ground_types))
  in let sorted_items = 
      List.sort (fun x y -> 
		   compare (get_sequence_number x) 
		     (get_sequence_number y)) 
	items
  in let (last_accu, rev_separated_items_tail) = 
					(* process from the head of the list *)
      List.fold_left
	(fun (cur_accu, res_accu) def ->
	   match (get_local_params def), cur_accu with
	     | [], _ -> (def :: cur_accu, res_accu)
	     | _::_, [] -> ([], [def] :: res_accu)
	     | _::_, _::_ -> ([], [def] :: (List.rev cur_accu) :: res_accu)
	)
	([],[]) sorted_items
  in 
  let rev_separated_items = match last_accu with
    | [] -> rev_separated_items_tail
    | _::_ -> (List.rev last_accu) :: rev_separated_items_tail 
  in
  let base_name = ccsl_sig_def_theory_name si in
  let name_counter = ref (List.length rev_separated_items) in
  let next_name () = 
    let n = base_name ^ (string_of_int !name_counter)
    in
      decr name_counter;
      n
  in
  let rev_named_items = match rev_separated_items with
    | [] -> assert(false);
    | [x] -> [(base_name, x)]
    | top :: rest ->
	(base_name, top) ::
	(List.map (fun items -> (next_name(), items)) rest)
  in
  let named_items = List.rev rev_named_items in
  let previous_name = ref None
  in
    List.map (fun (thname, items) -> 
		let th = new ccsl_sig_def_theory si thname 
			   items !previous_name
		in
		  previous_name := Some thname;
		  th
	     )
      named_items



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

