(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 11.11.99 by Hendrik
 *
 * Time-stamp: <Tuesday 16 July 02 21:28:04 tews@ithif56.inf.tu-dresden.de>
 *
 * substitution and optimization for top_types
 *
 * $Id: logic_util.ml,v 1.22 2002/07/18 13:43:27 tews Exp $
 *
 *)


open Util
open Top_variant_types
open Top_variant_types_util		(* for resolution_of *)
open Top_names
;;

(***********************************************************************
 ***********************************************************************
 *
 * substitution
 *
 * the substitution is given as two assoc lists, 
 * the first gives a substitution on types, 
 * with elements top_types * top_types 
 * the second gives a substitution on values.
 * 
 * Substitution on values is important because of value
 * parameters (and arguments). Currently, only substitution on values 
 * works only on very restricted set of expression (Term, Termvar)
 *)


type ('cl, 'mem) substitution_type = 
  (('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_types) list  *
  (('cl, 'mem) top_pre_expressions * ('cl, 'mem) top_pre_expressions) list


	(* Utility function for binders: this function takes a 
	 * substitution and a list of identifiers (coming from the 
	 * declaration list of Forall for instance). It deletes all 
	 * pairs (Term(id,_,_),_) where id is in the list of bound
	 * identifiers from the expression substitution.
	 *)
let bind_vars (typ_sub,ex_sub) id_list =
  (typ_sub,
   List.fold_right
     (fun sub res ->
	match sub with
	  | (Term(v,_,_),_) -> (if List.mem v id_list
				then res
				else sub::res)
	  | (BasicExpr(TermVar idc), _) -> 
	      (if List.mem (resolution_of idc).id_token.token_name id_list
	       then res
	       else sub :: res)
	  | _ -> sub :: res
     ) ex_sub []
  )

	(* Utility function to extract identifiers which 
	 * are bound in an case pattern. This works only for 
	 * case pattern of the form 
	 * Application(f,Tuple([Term(v,_,_); ... ]))
	 *)
let extract_case_binders pat =
  let args = 
    match pat with
      | Application(_,Tuple(ts)) -> ts
      | Application(_, t) -> [t]
      | SmartApplication(_, ts) -> ts
				(* for other cases do not bind anything *)
      | _ -> [] in
		   (* order is not important, so use efficient fold_left *)
  let ids = List.fold_left 
	      (fun res t -> match t with 
		 | Term(v,_,_) -> v :: res
		 | BasicExpr(TermVar idc) -> 
		     (resolution_of idc).id_token.token_name :: res
		 | _ -> res
	      ) [] args
  in
    ids


	(* make first an oparator, which works poymorphically on 
	 * argument and type lists
	 * Basically this is the same as List.map, the only difference is,
	 * that substitute_list preserves sharing, if possible
	 *
	 * Later (for the sequential binding of Let) I need also
	 * an equvalent to fold_right, but instead of implementing a 
	 * fold_right which preserves sharing, I will use 
	 * substitute_list from below with some side effects. For
	 * this to work it is essential, that sub_fun is applied to 
	 * the elements of the list in left-to-right order.
	 *
	 * val substitute_list : ('a -> 'a) -> 'a list -> 'a list
	 *
	 *)  
let rec substitute_list sub_fun arg_list = match arg_list with
  | [] -> arg_list
  | arg :: rest_list -> 
      let narg = sub_fun arg in
      let narg_list = substitute_list sub_fun rest_list in
	if (arg == narg) && (rest_list == narg_list) 
	then arg_list
	else narg :: narg_list


	(* substitution operator on option type *)

let substitute_option sub_fun opt = match opt with
  | None -> opt
  | Some x -> 
      let nx = sub_fun x in
	if x == nx 
	then opt
	else Some nx

    (**************************
     * here is the real substitution function
     * Every node of the abstract representation can be substituted 
     * with anything else. 
     *
     * Two equality functions are needed (one on types 
     * and one on expressions)
     *)

let rec substitute_types ((eq_typ,_) as eqs) ((typ_sub,_) as subst) typ = 
					   (* first try to substitute *)
  try Util.assoc eq_typ typ typ_sub
  with					(* now do the recursion *)
      Not_found -> 
	let recurse_type = substitute_types eqs subst in
	let recurse_exp = substitute_expression eqs subst in
	let recurse_form = substitute_formula eqs subst in
	let recurse_args = substitute_list (substitute_argument eqs subst) in
 	  (match typ with
	       
	   | Groundtype(id,args) -> 
	       let nargs = recurse_args args in
		 if nargs == args 
		 then typ 
		 else Groundtype(id,nargs)

	   | TypeConstant(name,flag,args) -> 
	       let nargs = recurse_args args in
		 if nargs == args 
		 then typ 
		 else TypeConstant(name,flag,nargs)
	   | BoundTypeVariable id -> typ
	   | FreeTypeVariable t -> typ
	   | Bool -> typ
	   | Self -> typ
	   | Carrier -> typ
					       (* we are in substitute_types *)
	   | Function(dom,codom) -> 
	       let ndom = recurse_type dom in
	       let ncodom = recurse_type codom in
		 if (dom == ndom) && (codom == ncodom) 
		 then typ
		 else Function(ndom,ncodom)
	   | SmartFunction(doml,codom) -> 
	       let ndoml = substitute_list recurse_type doml in
	       let ncodom = recurse_type codom in
		 if (doml == ndoml) && (codom == ncodom) 
		 then typ
		 else SmartFunction(ndoml,ncodom)
	   | Product(type_list) 	-> 
	       let ntype_list = substitute_list recurse_type type_list in
		 if ntype_list == type_list 
		 then typ 
		 else Product(ntype_list)
	   | Class(cl, args) -> 
	       let nargs = recurse_args args in
		 if nargs == args 
		 then typ 
		 else Class(cl,nargs)
					       (* we are in substitute_types *)
	   | Adt(adt,flag,args) -> 
	       let nargs = recurse_args args in
		 if nargs == args 
		 then typ 
		 else Adt(adt,flag,nargs)
	   | Predtype(formula) -> 
	       let formula' = recurse_form formula in
		 if formula == formula'
		 then typ
		 else Predtype(formula')
	   | Record(labels) -> 
	       let nlabels = 
		 substitute_list 
		   (fun ((l,t) as cell) -> 
		      let t' = recurse_type t in
			if t == t' 
			then cell
			else (l,t'))
		   labels 
	       in
		 if labels == nlabels 
		 then typ
		 else Record(nlabels)
					       (* we are in substitute_types *)
	   | IFace(cl,arg_flag,args) ->
	       let nargs = recurse_args args in
		 if nargs == args 
		 then typ 
		 else IFace(cl,arg_flag, nargs)		 
	   | Array(cl,elemtyp,size) ->
	       let nelemtyp = recurse_type elemtyp in
		 if nelemtyp == elemtyp
		 then typ
		 else Array(cl, nelemtyp, size)		     

	   )

and substitute_argument eqs subst argument = match argument with
  | TypeArgument(typ) -> 
      let ntyp = substitute_types eqs subst typ in
	if ntyp == typ then argument else TypeArgument(ntyp)

	     
and substitute_expression ((_,eq_ex) as eqs) ((_,ex_sub) as subst) expr = 
					   (* first try to substitute *)
  try Util.assoc eq_ex expr ex_sub
  with					(* now do the recursion *)
      Not_found -> 
	let recurse_type = substitute_types eqs subst in
	let recurse_exp = substitute_expression eqs subst in
	let recurse_form = substitute_formula eqs subst in
	let recurse_args = substitute_list (substitute_argument eqs subst) in
	(match expr with
	   | ExprLoc(ex,loc) ->
	       let nex = recurse_exp ex in
		 if ex == nex 
		 then expr
		 else ExprLoc(nex,loc)
	   | BasicExpr bexp -> expr
	   | Term(name,flag,args) -> 
	       let nargs = recurse_args args in
		 if nargs == args 
		 then expr
		 else Term(name,flag,nargs)
	   | TypedTerm(ex, typ) ->
	       let ex' = recurse_exp ex in
	       let typ' = recurse_type typ in
		 if ex==ex' && typ==typ' 
		 then expr
		 else TypedTerm(ex', typ')
					  (* we are in substitute_expression *)
	   | TypeAnnotation(ex, typ) ->
	       let ex' = recurse_exp ex in
	       let typ' = recurse_type typ in
		 if ex==ex' && typ==typ' 
		 then expr
		 else TypeAnnotation(ex', typ')
		 (* not in ccsl input types *)
	   | QualifiedTerm(th,arg_flag,args,term) ->
	       let nargs = recurse_args args in
		 if nargs == args 
		 then expr
		 else QualifiedTerm(th,arg_flag,nargs,term)
	   | MethodSelection(ex, instiface, m) ->
		  let ex' = recurse_exp ex in
		    if ex == ex' (* && typeopt==typeopt' *)
		    then expr
		    else MethodSelection(ex', instiface, m)

	   | Tuple(ex_list) -> 
	       let nex_list = substitute_list recurse_exp ex_list in 
		 if ex_list == nex_list 
		 then expr
		 else Tuple nex_list		 
	   | Projection(i,n) -> expr

					  (* we are in substitute_expression *)
	   | RecordTuple(fields) -> 
	       let nfields = 
		 substitute_list
		   (fun ((l,ex) as cell) -> 
		      let ex' = recurse_exp ex in
			if ex == ex' 
			then cell
			else (l,ex'))
		   fields 
	       in
		 if fields == nfields 
		 then expr
		 else RecordTuple(nfields)
	   | RecordSelection(field,ex) -> 	
	       let nex = recurse_exp ex in
		 if nex == ex 
		 then expr
		 else RecordSelection(field,nex)
	   | RecordUpdate(ex, changes) -> 
	       let nex = recurse_exp ex in
	       let nchanges = 
		 substitute_list
		   (fun ((l,ex) as cell) -> 
		      let ex' = recurse_exp ex in
			if ex == ex' 
			then cell
			else (l,ex'))
		   changes
	       in 
		 if ex == nex && changes == nchanges 
		 then expr
		 else RecordUpdate(nex,nchanges)
					  (* we are in substitute_expression *)
	   | List(ex_list) ->
	       let nex_list = substitute_list recurse_exp ex_list in 
		 if ex_list == nex_list 
		 then expr
		 else List nex_list		 
	   | Abstraction(decl_list,ex) ->
	       let ids = List.map fst decl_list in
	       let nsubst = bind_vars subst ids in
		 if nsubst = ([],[])
		 then expr
		 else 
		   let nex = substitute_expression eqs nsubst ex in
		     if nex == ex 
		     then expr
		     else Abstraction(decl_list,nex)
	   | SmartAbstraction(decl_list,ex) ->
	       let ids = List.map fst decl_list in
	       let nsubst = bind_vars subst ids in
		 if nsubst = ([],[])
		 then expr
		 else 
		   let nex = substitute_expression eqs nsubst ex in
		     if nex == ex 
		     then expr
		     else SmartAbstraction(decl_list,nex)
					  (* we are in substitute_expression *)
	   | Application(ex1,ex2) -> 
	       let nex1 = recurse_exp ex1 in
	       let nex2 = recurse_exp ex2 in
		 if ex1 == nex1 && ex2 == nex2 
		 then expr
		 else Application(nex1,nex2)
	   | InfixApplication(ex1,instiface,memcontainer,ex2) ->
	       let nex1 = recurse_exp ex1 in
	       let nex2 = recurse_exp ex2 in
		 if ex1 == nex1 && ex2 == nex2 
		 then expr
		 else InfixApplication(nex1,instiface,memcontainer,nex2)
	   | SmartApplication(ex1,ex2l) ->
	       let nex1 = recurse_exp ex1 in
	       let nex2l = substitute_list recurse_exp ex2l in
		 if ex1 == nex1 && ex2l == nex2l 
		 then expr
		 else SmartApplication(nex1,nex2l)
	   | FunUpdate(fex, changes) ->
	       let nfex = recurse_exp fex in
	       let nchanges = 
		 substitute_list
		   (fun ((ex1,ex2) as cell) -> 
		      let nex1 = recurse_exp ex1 in
		      let nex2 = recurse_exp ex2 in
			if ex1 == nex1 && ex2 == nex2 
			then cell
			else (nex1,nex2)
		   )
		   changes
	       in 
		 if fex == nfex && changes == nchanges 
		 then expr
		 else FunUpdate(nfex,nchanges)
					  (* we are in substitute_expression *)
	   | Let(decl_list, ex) ->
		   (* Let binds sequentially, so I use substitute_list
                    * to mimic a fold_right, that collects the
		    * identifiers bound so far (or better applies
		    * bind_vars as I walk through the decl_list).
		    *)
	       let nsubst = ref subst in
	       let ndecl_list = 
		 substitute_list
		   (fun ((id_rec,typ_opt,def) as cell) -> 
		      if !nsubst = ([],[]) 
		      then cell
		      else 			  
			let ntyp_opt = substitute_option 
				(substitute_types eqs !nsubst) typ_opt in

			let ndef = substitute_expression 
				     eqs !nsubst def in
			let _ = nsubst := bind_vars !nsubst 
				  [id_rec.id_token.token_name] 
			in
			  if typ_opt == ntyp_opt && def == ndef 
			  then cell
			  else (id_rec,ntyp_opt,ndef)
		   ) decl_list in
	       let nex = substitute_expression eqs !nsubst ex in
		 if decl_list == ndecl_list && ex == nex 
		 then expr
		 else Let(ndecl_list,nex)
					  (* we are in substitute_expression *)
	   | If(conds,ex) ->
	       let nconds = 
		 substitute_list
		   (fun ((form,case) as cell) -> 
		      let nform = recurse_form form in
		      let ncase = recurse_exp case in
			if form == nform && case == ncase 
			then cell
			else (nform,ncase)
		   ) conds in
	       let nex = recurse_exp ex in
		 if conds == nconds && ex == nex 
		 then expr
		 else If(nconds,nex)
	   | Case(ex,variants) -> 
	       let nex = recurse_exp ex in
	       let nvariants = 
		 substitute_list
		   (fun ((pat,case) as cell) -> 
					(* extract binders from pat *)
		      let ids = extract_case_binders pat in
		      let nsubst = bind_vars subst ids in
			if nsubst = ([],[]) 
			then cell
			else 
			  let ncase = substitute_expression 
					eqs nsubst case in
			    if case == ncase 
			    then cell
			    else (pat,ncase)
		   ) variants in
		 if ex == nex && variants == nvariants 
		 then expr
		 else Case(nex, nvariants)
					  (* we are in substitute_expression *)
	   | CCSL_Case(ex,variants) -> 
	       let nex = recurse_exp ex in
	       let nvariants = 
		 substitute_list
		   (fun ((mem,var_list,case) as cell) -> 
					(* extract binders from pat *)
		      let ids = List.map (fun id -> id.id_token.token_name)
				  var_list in
		      let nsubst = bind_vars subst ids in
			if nsubst = ([],[]) 
			then cell
			else 
			  let ncase = substitute_expression 
					eqs nsubst case in
			    if case == ncase 
			    then cell
			    else (mem,var_list,ncase)
		   ) variants in
		 if ex == nex && variants == nvariants 
		 then expr
		 else CCSL_Case(nex, nvariants)
	   | Box(typ,pred,tlist) ->
	       let n_typ =  recurse_type typ in
	       let n_pred = recurse_exp pred in
		 if n_typ == typ && n_pred == pred then
		   expr
		 else Box(typ,n_pred,tlist)
	   | Diamond(typ,pred,tlist) ->
	       let n_typ =  recurse_type typ in
	       let n_pred = recurse_exp pred in
		 if n_typ == typ && n_pred == pred then
		   expr
		 else Diamond(typ,n_pred,tlist)
					  (* we are in substitute_expression *)
	   | Every(typ, form_list) -> 
	       let ntyp = recurse_type typ in
	       let nform_list = substitute_list recurse_exp form_list in
		 if typ == ntyp && form_list == nform_list
		 then expr
		 else Every(ntyp,nform_list)
	   | RelEvery(typ, form_list) -> 
	       let ntyp = recurse_type typ in
	       let nform_list = substitute_list recurse_exp form_list in
		 if typ == ntyp && form_list == nform_list
		 then expr
		 else RelEvery(ntyp,nform_list)
	   | Map(typ, expr_list) ->
	       let ntyp = recurse_type typ in
	       let nexpr_list = substitute_list recurse_exp expr_list in
		 if typ == ntyp && expr_list == nexpr_list
		 then expr
		 else Map(ntyp, nexpr_list)
	   | Expression form ->
	       let nform = recurse_form form in
		 if form == nform 
		 then expr
		 else Expression nform
					  (* we are in substitute_expression *)
	   | Comprehension(s,typ,form) -> 
	       let ntyp = recurse_type typ in
	       let nsubst = bind_vars subst [s] in
	       let nform = 
		 if nsubst = ([],[]) 
		 then form
		 else substitute_formula eqs nsubst form
	       in
		 if typ == ntyp && form == nform 
		 then expr
		 else Comprehension(s,ntyp,nform)
	)
	
and substitute_formula eqs subst formula =
  let recurse_type = substitute_types eqs subst in
  let recurse_exp = substitute_expression eqs subst in
  let recurse_form = substitute_formula eqs subst in
  let recurse_args = substitute_list (substitute_argument eqs subst) 
  in
  (match formula with
     | FormLoc(f,loc) ->
	 let nf = recurse_form f in
	   if f == nf 
	   then formula
	   else FormLoc( nf, loc)
     | True -> formula
     | False -> formula
     | Not f -> 
	 let nf = recurse_form f in
	   if f == nf 
	   then formula
	   else Not nf
     | And f_list -> 
	 let nf_list = substitute_list recurse_form f_list in
	   if nf_list == f_list 
	   then formula
	   else And nf_list
     | Or f_list -> 
	 let nf_list = substitute_list recurse_form f_list in
	   if nf_list == f_list 
	   then formula
	   else Or nf_list
					     (* we are in substitute_formula *)
     | Implies(assum,concl) -> 
	 let nassum = recurse_form assum in
	 let nconcl = recurse_form concl in
	   if assum == nassum && concl == nconcl 
	   then formula
	   else Implies(nassum,nconcl)
     | Iff(assum,concl) -> 
	 let nassum = recurse_form assum in
	 let nconcl = recurse_form concl in
	   if assum == nassum && concl == nconcl 
	   then formula
	   else Iff(nassum,nconcl)
     | Equal(ex_a,ex_b) -> 
	 let nex_a = recurse_exp ex_a in
	 let nex_b = recurse_exp ex_b in
	   if ex_a == nex_a && ex_b == nex_b 
	   then formula
	   else Equal(nex_a, nex_b)
     | Forall(quant_list, f) -> 
	 let ids = List.map fst quant_list in
	 let nsubst = bind_vars subst ids in
	 let nf = recurse_form f in
	   if f == nf 
	   then formula
	   else Forall(quant_list,nf)
					     (* we are in substitute_formula *)
     | Exists(quant_list, f) -> 
	 let ids = List.map fst quant_list in
	 let nsubst = bind_vars subst ids in
	 let nf = recurse_form f in
	   if f == nf 
	   then formula
	   else Exists(quant_list,nf)
     | ConstantPredicate name -> formula
     | Formula ex -> 
	 let nex = recurse_exp ex in
	   if ex == nex 
	   then formula
	   else Formula nex
     | MetaImplies(assum,concl) -> 
	 let nassum = recurse_form assum in
	 let nconcl = recurse_form concl in
	   if assum == nassum && concl == nconcl 
	   then formula
	   else MetaImplies(nassum,nconcl)
     | MetaForall(quant_list, f) -> 
	 let ids = List.map fst quant_list in
	 let nsubst = bind_vars subst ids in
	 let nf = recurse_form f in
	   if f == nf 
	   then formula
	   else MetaForall(quant_list,nf)
					     (* we are in substitute_formula *)
     | Bisim(typ,ex1,ex2) -> 
	 let typ1 = recurse_type typ in
	 let nex1 = recurse_exp ex1 in
	 let nex2 = recurse_exp ex2 in
	   if typ == typ1 && ex1 == nex1 && ex2 == nex2 
	   then formula
	   else Bisim(typ,nex1,nex2)
     | Obseq(t,ex1,ex2) -> 
	 let t1 = substitute_option 
		    (function (name,typ) as arg -> 
		       let typ1 = recurse_type typ in
			 if typ1 == typ 
			 then arg
			 else (name, typ1)
		    ) t in
	 let nex1 = recurse_exp ex1 in
	 let nex2 = recurse_exp ex2 in
	   if t == t1 && ex1 == nex1 && ex2 == nex2 
	   then formula
	   else Obseq(t1,nex1,nex2)
  )	
  
  

					   (* export for argument lists *)
let substitute_arguments eqs subst =
  substitute_list (substitute_argument eqs subst)

(****************
 * specialize the general function to just substitute the types 
 * given in the assoc list
 *
 * val substitute_types_only :
 *   (('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types -> bool) ->
 *     (('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_types) list ->
 * 	 ('cl, 'mem) top_pre_types -> 
 * 	   ('cl, 'mem) top_pre_types 
 *)

let substitute_types_only eq_typ sub_typ typ =
  if sub_typ = [] 
  then typ 
  else substitute_types (eq_typ,(==)) (sub_typ,[]) typ
  

let substitute_arguments_types_only eq_type subst arg_list =
  substitute_arguments (eq_type,(==)) (subst,[]) arg_list


(***********************************************************************
 ***********************************************************************
 *
 * optimization of formulas and expressions 
 *
 *
 * rewrites:
 * Forall( ) : f -> f
 * Forall( v1 ) : Forall (v2 ) : f -> Forall( v1@v2 ) : f
 * Forall( v ) : true/false  -> true/false
 * Exists( v1 ) : Exists( v2 ) : f -> Exists( v1@v2 ) : f
 * Exists( v ) : false -> false
 * Every(t, True)(x) -> True
 * Proj_i ( ... ith ... ) -> ith
 * (Lambda (decl) : body1) = (Lambda (decl) : body2) ->
 *         Forall(decl) : body 1 = body2
 * true/false in And, OR, Implies
 * verious versions of eta 
 *)

(* utility function: checks for free occurences of an identifier.
 * takes care of bindings in Forall/Exists/Lambda. Variables bind 
 * in cases are taken care of if they are in the form supported by
 * extract_case_binders. 
 * An identifier is free, if it occurs as Term(id,...) or as 
 * BasicExpr(Termvar(... id ...))
 *)

let rec occurs_free_in_expression id expr = 
  let recurse_exp = occurs_free_in_expression id in
  let recurse_form = occurs_free_in_formula id 
  in
    (match expr with
       | ExprLoc(ex,loc) -> recurse_exp ex
       | BasicExpr bexp -> 
	   (match bexp with
	      | TermVar idc -> 
		  (resolution_of idc).id_token.token_name = id
	      | _ -> false
	   )
       | Term(name,flag,args) -> name = id
       | TypedTerm(ex, typ) -> recurse_exp ex
       | TypeAnnotation(ex, typ) -> recurse_exp ex
       | QualifiedTerm(th,arg_flag,args,term) -> false
       | MethodSelection(ex, instiface, m) -> recurse_exp ex
       | Tuple(ex_list) -> 
	   List.exists recurse_exp ex_list
       | Projection(i,n) -> false
       | RecordTuple(fields) -> 
	   List.exists (fun (label,ex) -> recurse_exp ex) fields
       | RecordSelection(field,ex) -> recurse_exp ex
       | RecordUpdate(ex, changes) -> 
	   (recurse_exp ex) or
	   (List.exists (fun (label,ex) -> recurse_exp ex) changes)
       | List(ex_list) -> List.exists recurse_exp ex_list
       | Abstraction(decl_list,ex) ->
	   if List.exists (fun (v,typ) -> v = id) decl_list
	   then				(* id is bound *)
	     false
	   else 
	     recurse_exp ex
       | SmartAbstraction(decl_list,ex) ->
	   if List.exists (fun (v,typ) -> v = id) decl_list
	   then				(* id is bound *)
	     false
	   else 
	     recurse_exp ex
       | Application(ex1,ex2) -> 
	   (recurse_exp ex1) or (recurse_exp ex2)
       | InfixApplication(ex1,instiface,memcontainer,ex2) ->
	   (recurse_exp ex1) or (recurse_exp ex2)
       | SmartApplication(ex1,ex2l) ->
	   (recurse_exp ex1) or (List.exists recurse_exp ex2l)
       | FunUpdate(fex, changes) ->
	   (recurse_exp fex) or
	   (List.exists (fun (ex1, ex2) ->
			   (recurse_exp ex1) or (recurse_exp ex2))
	      changes)
       | Let(decl_list, ex) ->
	   List.fold_right
	     (fun (lid,typopt,lex) res ->
		if lid.id_token.token_name = id 
		then			(* bound by this let *)
		  recurse_exp lex
		else			(* not bound *)
		  (recurse_exp lex) or res
	     )
	     decl_list (recurse_exp ex)
       | If(conds,ex) ->
	   (List.exists (fun (cond,ex) -> 
			   (recurse_form cond) or recurse_exp ex)
	      conds)
	   or 
	   (recurse_exp ex)
       | Case(ex,variants) -> 
	   (recurse_exp ex) or
	   (List.exists 
	      (fun (pat,case) ->
		 if List.exists (fun caseid -> caseid = id) 
		   (extract_case_binders pat)
		 then
		   false
		 else
		   recurse_exp case
	      )
	      variants)
       | CCSL_Case(ex,variants) -> 
	   (recurse_exp ex) or
	   (List.exists 
	      (fun (memc,idreclist,case) ->
		 if List.exists (fun idrec -> idrec.id_token.token_name = id)
		   idreclist
		 then
		   false
		 else
		   recurse_exp case
	      )
	      variants)
       | Box(typ,pred,tlist) -> recurse_exp pred
       | Diamond(typ,pred,tlist) -> recurse_exp pred
       | Every(typ, pred_list) -> 
	   List.exists recurse_exp pred_list
       | RelEvery(typ, pred_list) -> 
	   List.exists recurse_exp pred_list
       | Map(typ, expr_list) ->
	   List.exists recurse_exp expr_list
       | Expression form -> recurse_form form 
       | Comprehension(s,typ,form) -> 
	   if id = s then false
	   else recurse_form form
    )
	
and occurs_free_in_formula id formula =
  let recurse_exp = occurs_free_in_expression id in
  let recurse_form = occurs_free_in_formula id 
  in
    (match formula with
       | FormLoc(f,loc) -> recurse_form f
       | True -> false
       | False -> false
       | Not f -> recurse_form f
       | And f_list -> List.exists recurse_form f_list
       | Or f_list -> List.exists recurse_form f_list
       | Implies(assum,concl) -> 
	   (recurse_form assum) or (recurse_form concl)
       | Iff(assum,concl) -> 
	   (recurse_form assum) or (recurse_form concl)
       | Equal(ex_a,ex_b) -> 
	   (recurse_exp ex_a) or (recurse_exp ex_b)
       | Forall(quant_list, f) -> 
	   if List.exists (fun (v,typ) -> v = id) quant_list
	   then false
	   else recurse_form f
       | Exists(quant_list, f) -> 
	   if List.exists (fun (v,typ) -> v = id) quant_list
	   then false
	   else recurse_form f
       | ConstantPredicate name -> false
       | Formula ex -> recurse_exp ex 
       | MetaImplies(assum,concl) -> 
	   (recurse_form assum) or (recurse_form concl)
       | MetaForall(quant_list, f) -> 
	   if List.exists (fun (v,typ) -> v = id) quant_list
	   then false
	   else recurse_form f
       | Bisim(typ,ex1,ex2) -> 
	   (recurse_exp ex1) or (recurse_exp ex2)
       | Obseq(t,ex1,ex2) -> 
	   (recurse_exp ex1) or (recurse_exp ex2)
    )
  


(* utility function: decides wether an abstraction is of a special form
 * (more precisely if it is the identity permutation), that allows to 
 * optimize it with eta equality of functions
 *)

let rec identity_abs abs =
  match abs with
    | Abstraction([v,t],Term(v',_,_))
	when v=v' -> true
    | Abstraction(vl, Tuple(exl))->
	List.for_all2
	  (fun (v1,t1) ex ->
	     match ex with 
	       | Term(v2,_,_) when v1 = v2 -> true 
	       | _ -> false)
	  vl exl
    | Term(f,Always,_) -> f = name_of_identity_function()
    | Map(_, tl) -> List.for_all identity_abs tl
    | _ -> false


(* utility function: reorders a variable declaration list, such that 
 * the declaration of a given variable stands at front. Don't do 
 * anything, if this variable is not declared here. The order of the 
 * other declarations is not changed.
 *)

let push_decl_to_front name decl_list = 
  let rec doit left = function
    | [] -> (* scanned everything, name is not declared here *)
	decl_list
    | (v,_) as v_decl :: rest -> 
	if v = name then
	  v_decl :: (List.rev left) @ rest
	else doit (v_decl :: left) rest
  in
    doit [] decl_list


(* utility function: equality used in substitution of 
 * terms for variables
 *)

let term_subst_eq ex1 ex2 = match ex1,ex2 with
  | Term(v1,_,_),Term(v2,_,_) -> v1 = v2
  | Term(v1,_,_), BasicExpr(TermVar idc) -> 
      v1 = (resolution_of idc).id_token.token_name
  | _ -> false
	
let rec do_opt_formula = function
  | Forall([],f) -> do_opt_formula f
	   (* substitute equalities like 
	    *   Forall( a1 : A, a2 : A) : a1 = a2 IMPLIES F
	    *   ---> Forall( a1 : A ) : F[ a1 / a2 ]
	    *)
  | Forall( (v1, t_v1) as v1_decl :: var_decls,
	    Implies(
	      And(
		Equal( (Term(v1',_,_) as v1_term), 
		       (Term(v2',_,_) as v2_term)) :: assums ),
	      concl)) 
      when (v1 == v1') && 
	(match push_decl_to_front v2' var_decls with
	   | (v2, t_v2) :: var_decls' -> (v2 == v2') && (t_v1 == t_v2)
	   | _ -> false) ->
	  let reorder = push_decl_to_front v2' var_decls in
	  let ((v2, t_v2), var_decls') = (List.hd reorder, List.tl reorder) in
	    
	  let concl' = substitute_formula
			 ((==),term_subst_eq) 
			 ([], [(v2_term,v1_term)] ) concl in
	  let f' = 
	    do_opt_formula(
	      Forall( var_decls',
		      Implies(
			And( assums ),
			concl'))) 
	  in
	    (match f' with
	       | Forall( rest_decl, f'' ) -> 
		   Forall( v1_decl :: rest_decl, f'')
	       | f'' -> 
		   Forall( [v1_decl], f''))
  | Forall(var1,f) ->
      (match do_opt_formula f with
	 | True -> True
	 | Forall(var2,f')-> Forall(var1@var2, f')
	 | f' -> Forall(var1,f'))
  | Exists(var1, f) ->
      (match do_opt_formula f with
	 | False -> False
	 | Exists(var2,f')-> Exists(var1@var2, f')
	 | f' -> Exists(var1,f'))
  | And fl		->
      let fl' = List.filter (function
			       | True -> false
			       | _ -> true)
		  (List.map do_opt_formula fl) in
	if List.mem False fl' then False
	else
	  (match fl' with
	     | [] -> True
	     | [fl''] -> fl''
	     | fl'' -> And fl'')
  | Or fl		-> 
      let fl' = List.filter (function
			       | False -> false
			       | _ -> true)
		  (List.map do_opt_formula fl) in
	if List.mem True fl' then True
	else
	  (match fl' with
	     | [] -> False
	     | [fl''] -> fl''
	     | fl'' -> Or fl'')
  | Implies (f1,f2) ->
      (match (do_opt_formula f1, do_opt_formula f2) with
	 | (_, True) -> True
	 | (True,f) -> f
	 | (False,_) -> True
	 | (f1',f2') -> Implies(f1',f2'))
      
  | Formula(exp)	-> (match do_opt_expression exp with
				Expression f -> f
				    (* | Tuple [] -> True  *)
			      | x -> Formula x)
  | Equal(ex1, ex2) -> 
      (match (ex1, ex2) with
	 | Abstraction(decllist1, body1), Abstraction(decllist2, body2)
	     when decllist1 = decllist2 ->
	     do_opt_formula(
	       Forall(decllist1, 
		      Equal(body1,body2)))
	 | Tuple(exleft), Tuple(exright)
	     when (List.length exleft) = (List.length exright)
	       -> 
	     do_opt_formula
	       (And( List.map2 (fun exl exr -> Equal(exl, exr))
		       exleft exright
		   ))

	 | nex1,nex2 -> 
	     Equal( do_opt_expression nex1, 
		    do_opt_expression nex2)
      )
(* 	 (match (do_opt_expression ex1, do_opt_expression ex2) with
 * 	    | Abstraction(decllist1, body1), Abstraction(decllist2, body2)
 * 		when decllist1 = decllist2 ->
 * 		Forall(decllist1, Equal(body1,body2))
 * 	    | nex1,nex2 -> Equal(nex1, nex2)
 * 	 )
 *)

(*   | LessOrEqual(ex1, ex2) -> 
 * 	 LessOrEqual(do_opt_expression ex1, do_opt_expression ex2)
 *)
  | Bisim(cl,al,ex) -> Bisim(cl,al,do_opt_expression ex)
  | x 		-> x

and do_opt_expression = function
    (* eta equality for functions *)
  | Abstraction(
      [f1,Function(dom_f1,codom_f1);
       f2,Function(dom_f2,codom_f2)] as fun_decl,
      Expression(
	Forall(
	  [v1,t_v1; v2,t_v2],
	  Implies(
	    And [Equal(Term(v1',_,_),
		       Term(v2',_,_))],
	    Equal(
	      Application(
		Term(f1',_,_) as f1_term, Term(v1'',_,_)),
	      Application(
		Term(f2',_,_) as f2_term, Term(v2'',_,_)))))))
      when (dom_f1 == dom_f2) && (codom_f1 == codom_f2) 
	&& (t_v1 == dom_f1) && (t_v2 == dom_f2) 
	&& (v1' == v1) && (v2' == v2) && (v1'' == v1) && (v2'' == v2) 
	&& (f1' == f1) && (f2' == f2) -> 
	  do_opt_expression(
	    Abstraction(fun_decl, 
			Expression(Equal(f1_term, f2_term))))
  | Abstraction([],exp) -> do_opt_expression exp
(*   | Abstraction([v,t] as decl, ex) ->
 * 	 (match (do_opt_expression ex) with
 * 	    | Application(Term _ as ex',Term(v',Always,[]))
 * 		when v= v' 
 * 			-> ex'
 * 	    | Application(Expression(ConstantPredicate _) as ex',
 * 			  Term(v',Always,[]))
 * 		when v= v' 
 * 			-> ex'
 * 	    | Term(v',Always,[]) when v = v' -> 
 * 		Term(name_of_identity_function(), Always, [TypeArgument t])
 * 	    | ex' -> (Abstraction(decl,ex')))      
 *   | Abstraction(vl, ex)->
 * 	 (match do_opt_expression ex with
 * 	    | Application(Term _ as ex',Tuple tl)
 * 		when tl = (List.map (fun (v,t) -> Term(v,Always,[])) vl)
 * 			  -> ex'
 * 	    | Application(Expression(ConstantPredicate _) as ex',Tuple tl)
 * 		when tl = (List.map (fun (v,t) -> Term(v,Always,[])) vl)
 * 			  -> ex'
 * 	    | Tuple(tl) 
 * 		when tl = (List.map (fun (v,t) -> Term(v,Always,[])) vl)
 * 		  -> Term(name_of_identity_function(), Always,
 * 			  [TypeArgument(Product(List.map snd vl))])
 * 	    | ex' -> Abstraction(vl,ex'))
 *)
      
  | Abstraction([v,t] as decl, ex)->
      (match do_opt_expression ex with
	 | Application(ex', Term(v',Always,[]))
	     when v= v' 
	       && not (occurs_free_in_expression v ex')
	       -> ex'
	 | Term(v',Always,[]) when v = v' -> 
	     Term(name_of_identity_function(), Always, [TypeArgument t])
	 | ex' -> (Abstraction(decl,ex')))      

  | Abstraction(vl, ex)->
      (match do_opt_expression ex with
	 | Application(ex', Tuple tl)
	     when tl = (List.map (fun (v,t) -> Term(v,Always,[])) vl)
	       && not (List.for_all (fun (v,t) -> 
				       occurs_free_in_expression v ex')
			 vl)
	       -> ex'
	 | Tuple(tl) 
	     when tl = (List.map (fun (v,t) -> Term(v,Always,[])) vl)
	       -> Term(name_of_identity_function(), Always,
		       [TypeArgument(Product(List.map snd vl))])
	 | ex' -> Abstraction(vl,ex'))
      
  | Application(
      RelEvery(typ, forml), Tuple[tl;tr]) ->
      let forml' = List.map do_opt_expression forml in
	if
	  List.for_all
	    (function
		 Abstraction(
		   [(v1,_); (v2,_)],
		   Expression(Equal(Term(ex1,_,_), Term(ex2,_,_))))
		   when (v1 = ex1) && (v2 = ex2) -> true
	       | _ -> false)
	    forml'
	then
	  match typ with
	    | Class(_, _) -> do_opt_expression(Expression(Bisim(typ,tl,tr)))
	    | Adt(_,_,_) -> do_opt_expression(Expression(Equal(tl,tr)))
	    | Groundtype _ -> do_opt_expression(Expression(Equal(tl,tr)))
		  (* other things must not occur under RelEvery *)
	    | _ -> assert(false)
	else
	  Application(
	    RelEvery(typ, forml'),
	    Tuple[do_opt_expression tl; do_opt_expression tr])
	    
  | Application(ex1,ex2)->
      (match (do_opt_expression ex1, do_opt_expression ex2) with
	 | (Abstraction([("f", _ );("x", _)],
			Application(Term("f",Always,[]), 
				    Term("x", Always,[]))),
	    Tuple([fex;ex])) 
	   ->
	     Application(fex,ex)
	 | (Abstraction(vl, ex), Tuple tl) 
	     when tl = (List.map (fun (v,t) -> Term(v,Always,[])) vl)
		       -> ex
	 | (Abstraction([v,t],ex), Term(v',_,_))
	     when v=v' -> ex
	 | (Abstraction
	      (["z",Bool], 
	       Expression(Equal(Term("z",Always,[]), 
				Expression(True)))),
	      ex2) -> Expression(Equal(ex2, Expression(True)))
	 | (Every(typ, pred_list),ex2') 
	     when List.for_all
	       (function
		  | Abstraction(_,Expression(True)) -> true
		  | _ -> false)
	       pred_list -> Expression(True)
	 | (Projection (i,n)),(Tuple exl) ->
	     assert(List.length exl = n);
	     assert((i > 0) && (i <= n));
	     List.nth exl (i-1)
	 | (ex1',ex2') ->
	     if identity_abs ex1' then ex2'
	     else Application(ex1',ex2'))	     
  | Expression f	-> 
      (match do_opt_formula f with
	   Formula x -> x
	 | x -> Expression x)
  | Projection(i,n) -> Projection(i,n)
  | Tuple exl -> 
      (match List.map do_opt_expression exl with
	 | [] -> Tuple []
	 | (Application(Projection(1, n), tt)) :: oexl' as exl' ->
	     if (List.length exl') = n
	       && snd (List.fold_left 
			 (fun (i,res) -> function 
			    | Application(Projection(j,m) , tt')
				when j = i && m = n && tt == tt'
				  -> (i+1,res)
			    | _ -> (i+1, false)
			 )
			 (2, true) oexl'
		      )
	     then			(* (Proj_1 t, Proj_2 t, Proj_3 t) *)
	       tt
	     else
	       Tuple exl'
	 | exl' -> Tuple exl'
      )
(* 
 *   | Tuple exl -> Tuple (List.map do_opt_expression exl)
 *)
  | RecordTuple lexl -> 
      RecordTuple (List.map
		     (fun (l,ex) -> (l, do_opt_expression ex))
		     lexl)

  | Case (ex, exl) ->
      let nex = do_opt_expression ex in
      let nexl = List.map 
		   (fun (l,r) -> (l, do_opt_expression r)) exl
      in
	if List.for_all (fun (_,ex) -> ex = Expression(True)) nexl
	then Expression(True)
	else
	  Case( nex, nexl )
  | CCSL_Case (ex, match_list) ->
      CCSL_Case( do_opt_expression ex,
		 List.map
		   (fun (mem,var_list,e) -> (mem, var_list, 
					     do_opt_expression e))
		   match_list)
  | Let ([], ex) -> do_opt_expression ex
  | Let (ll, ex) ->
      Let (List.map (fun (s,t,e) -> (s,t, do_opt_expression e)) ll,
	   do_opt_expression ex)
  | Every(typ, pred_list) ->
      Every(typ, (List.map do_opt_expression pred_list))
  | Map(typ, expr_list) ->
      Map(typ, List.map do_opt_expression expr_list)
  | SmartApplication(ex, exl) ->
      SmartApplication(
	do_opt_expression ex,
	List.map do_opt_expression exl)
  | SmartAbstraction(decllist,ex) ->
      SmartAbstraction(decllist, do_opt_expression ex)
  | x 			-> x


let opt_formula f = 
  if !Global.optimize_expressions 
  then do_opt_formula f 
  else f

let opt_expression f = 
  if !Global.optimize_expressions 
  then do_opt_expression f 
  else f



(***********************************************************************
 * 
 * manipulate variances
 * 
 *)

let valid_variance = function
  | Pair(vn,vp) -> ( ((vn = -1) or (vn mod 2 = 1)) &&
		     ((vp = -1) or (vp mod 2 = 0)) 
		   )
  | _ -> true


let make_simple = function
  | Pair(-1,-1) -> Unused
  | Pair(-1,_) -> Pos
  | Pair(_,-1) -> Neg
  | Pair _ -> Mixed
  | v -> v


   (* make -1 a zero with respect to addition *)

let vadd i1 i2 = match i1,i2 with
  | -1, _  -> -1
  | _ , -1 -> -1
  | _,_ -> i1 + i2


let rec variance_subst v1 v2 = match v1,v2 with
  | Pair(n1,p1), Pair(n2,p2) -> Pair( max (vadd n1 p2) (vadd p1 n2),
				      max (vadd n1 n2) (vadd p1 p2))
  | Pair _ ,_ -> variance_subst (make_simple v1) v2
  | _, Pair _ -> variance_subst v1 (make_simple v2)

  | Unused, _ -> Unused
  | _, Unused -> Unused
  | Mixed, _ -> Mixed
  | _, Mixed -> Mixed
  | Pos,Pos -> Pos
  | Pos,Neg -> Neg
  | Neg,Pos -> Neg
  | Neg,Neg -> Pos
  | Unset,_
  | _, Unset -> assert(false)


let rec variance_join v1 v2 = match v1,v2 with
  | Pair(n1,p1), Pair(n2,p2) -> Pair( max n1 n2, max p1 p2 )
  | Pair _ ,_ -> variance_join (make_simple v1) v2
  | _, Pair _ -> variance_join v1 (make_simple v2)

  | Mixed, _ -> Mixed
  | _, Mixed -> Mixed
  | Unused, v -> v
  | v, Unused -> v
  | Pos,Pos -> Pos
  | Pos,Neg -> Mixed
  | Neg,Pos -> Mixed
  | Neg,Neg -> Neg
  | Unset,_
  | _, Unset -> assert(false)


let variance_flatten vlist = 
  List.fold_right
    (fun v accu -> match v with
	  | None, Some a -> a :: accu
	  | Some a, None -> a :: accu
	  | Some a, Some a' -> a :: a' :: accu
	  | _ -> assert(false)
    )
    vlist []


    (* convert a list of names with variance information, 
     * as procuced by create_ids_with_variance into a list of 
     * terms with variance information
     *)
let var_terms_from_var_ids =
  let build_term name = Some(Term(name,Always,[])) 
  in 
    fun type_param_ids var_ids ->
      List.map2
	(fun id -> function
	   | Some(name, _), None -> (id, (build_term name, None))
	   | None, Some(name, _) -> (id, (None, build_term name))
	   | Some (name1,_), Some(name2,_) ->
	       (id, (build_term name1, build_term name2))
	   | _ -> assert(false)
	)
	type_param_ids var_ids


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

