;;;   ccsl-mode.el --- major mode for editing Ccsl

;; Copyright (C) 1998 KUN.

;; Authors:	1998 Wim Janssen KUN (wim@cs.kun.nl)
;; Created:	1998/22/09
;; Version:	0.3
;; Modified:	1999/01/13
;; Keywords:	Unix languages

;; This file is part of LOOP.

;; LOOP is .....

;;;   Commentary:

;; This package provides a major mode for editing ccsl. It knows about
;; ccsl syntax, indentation, comments and fontifying.
;;
;; To setup `ccsl-mode' put this file in the directory `site-lisp'
;; or put it in some directory and extend the `load-path' by adding the
;; following lines to your `.emacs' file.
;; The directory `~/Emacs/' is used in the sample code.
;;
;;   ;; Extend `load-path' for `ccsl-mode' and `loop-compile' packages.
;;   (setq load-path
;;     (cons (concat (expand-file-name "~") "/Emacs/") load-path))
;;
;;   ;; Setup autoloading the `ccsl-mode'.
;;   ;; Extend emacs with the `M-x ccsl-mode' command.
;;   (setq auto-mode-alist
;;     (cons '("\\.beh\\'" . ccsl-mode) auto-mode-alist))
;;
;;   (autoload 'ccsl-mode "ccsl-mode"
;;     "Major mode for editing ccsl." t nil)
;;
;; If you want to use `font-lock' in `ccsl-mode' add the next lines.
;;
;;   ;; Use `font-lock' in `ccsl-mode', otherwise comment out the next line.
;;   (add-hook 'ccsl-mode-hook 'font-lock-mode)
;;
;; You can preset some user definable variables in your `.emacs' file.
;; (Currently just one variable for `ccsl-mode')
;;
;;   ;; Examples of user definable variables
;;
;;   ;; From `ccsl-mode.el'.
;;   ;;
;;   ;; The comment style used in `ccsl-mode' buffers.
;;   ;; Chooose between 'pvs or 'ml, default is 'pvs.
;;   (setq ccsl-comment-style 'ml)
;;
;; End of example code to insert in your `.emacs' file.


;;;   Code:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; user definable variables                                  ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar ccsl-comment-style 'pvs
  "*The comment style used in `ccsl-mode' buffers.

A choice between pvs comment style ( % ..... ) or
ml comment style ( (* ..... *) ) can be made with
argument 'pvs or 'ml. The initial value is 'pvs.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; no user definable variables beyond this point             ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;   Map
;;;
(defvar ccsl-mode-map nil
  "Keymap table used in `ccsl-mode' buffers.")

(if ccsl-mode-map
    ()
  (setq ccsl-mode-map (make-sparse-keymap))
  (define-key ccsl-mode-map [return] 'newline-and-indent)
  (define-key ccsl-mode-map [tab] 'indent-relative)
  (define-key ccsl-mode-map "\M-;" 'indent-for-comment)
  (define-key ccsl-mode-map "\C-ca" 'abbrev-mode)
  ;;(define-key ccsl-mode-map "\C-c;" 'ccsl-comment-region)
  (define-key ccsl-mode-map "\C-c;" 'comment-region)
  (define-key ccsl-mode-map "\C-c:" 'ccsl-uncomment-region)
  (define-key ccsl-mode-map "\C-c?" 'ccsl-mode-describe))

;;;   Syntax table
;;;
(defvar ccsl-mode-syntax-table nil
  "Syntax table used in `ccsl-mode' buffers.")

(if ccsl-mode-syntax-table
    ()
  (setq ccsl-mode-syntax-table (make-syntax-table))
  (modify-syntax-entry ??  "w    " ccsl-mode-syntax-table)
  (modify-syntax-entry ?_  "w    " ccsl-mode-syntax-table)
  (modify-syntax-entry ?\\ "\\   " ccsl-mode-syntax-table)
  (modify-syntax-entry ?+  ".    " ccsl-mode-syntax-table)
  (modify-syntax-entry ?-  ".    " ccsl-mode-syntax-table)
  (modify-syntax-entry ?*  ". 23b" ccsl-mode-syntax-table)
  (modify-syntax-entry ?/  ".    " ccsl-mode-syntax-table)
  (modify-syntax-entry ?<  ".    " ccsl-mode-syntax-table)
  (modify-syntax-entry ?>  ".    " ccsl-mode-syntax-table)
  (modify-syntax-entry ?&  ".    " ccsl-mode-syntax-table)
  (modify-syntax-entry ?|  ".    " ccsl-mode-syntax-table)
  (modify-syntax-entry ?=  ".    " ccsl-mode-syntax-table)
  (modify-syntax-entry ?%  "<    " ccsl-mode-syntax-table)
  (modify-syntax-entry ?[  "(]   " ccsl-mode-syntax-table)
  (modify-syntax-entry ?]  ")[   " ccsl-mode-syntax-table)
  (modify-syntax-entry ?(  "()1 b" ccsl-mode-syntax-table)
  (modify-syntax-entry ?)  ")(4 b" ccsl-mode-syntax-table)
  (modify-syntax-entry ?\n ">    " ccsl-mode-syntax-table)
  (modify-syntax-entry ?\' "\"   " ccsl-mode-syntax-table))

;;;   Abbrev table
;;;
(defvar ccsl-mode-abbrev-table nil
  "Abbrev table used in `ccsl-mode' buffers.")

(if ccsl-mode-abbrev-table
    ()
  (define-abbrev-table 'ccsl-mode-abbrev-table
    '()))

;;;   Menubar menu
;;;
(defvar ccsl-mode-menubar-menu nil
  "Menubar menu used in `ccsl-mode' buffers.")

(if ccsl-mode-menubar-menu
    ()
  (easy-menu-define
   ccsl-mode-menubar-menu ccsl-mode-map "CCSL"
   '("CCSL"
     ;;["Comment Region" ccsl-comment-region (mark)]
     ["Comment Region" comment-region (mark)]
     ["Uncomment Region" ccsl-uncomment-region (mark)]
     "-"
     ("CCSL Mode Options"
      ;;["Abbreviation Mode" abbrev-mode
      ;; :style toggle
      ;; :selected abbrev-mode]
      ;;["Read Abbreviations" read-abbrev-file t]
      ;;["Save Abbreviations" write-abbrev-file t]
      ;;"--"
      ["PVS Comment Style" (ccsl-set-comment-style 'pvs)
       :style radio
       :selected (eq ccsl-comment-style 'pvs)]
      ["ML Comment Style" (ccsl-set-comment-style 'ml)
       :style radio
       :selected (eq ccsl-comment-style 'ml)])
     ["Describe Mode" ccsl-mode-describe t])))

;;;   Font lock keywords
;;;
(defvar ccsl-mode-font-lock-keywords
  (let* ((keyword-regexp
	  (concat
	   "\\("
	   "\\<adt\\>" "\\|"
	   "\\<and\\>" "\\|"
	   "\\<as\\>" "\\|"
	   "\\<assertion\\>" "\\|"
	   "\\<attribute\\>" "\\|"
	   "\\<defining\\>" "\\|"
	   "\\<begin\\>" "\\|"
	   "\\<carrier\\>" "\\|"
	   "\\<classspec\\>" "\\|"
	   "\\<final\\>" "\\|"
	   "\\<constructor\\>" "\\|"
	   "\\<creation\\>" "\\|"
	   "\\<request\\>" "\\|"
	   "\\<end\\>" "\\|"
	   ;;"\\<endpvs\\>" "\\|"
	   "\\<from\\>" "\\|"
	   "\\<importing\\>" "\\|"
	   "\\<in\\>" "\\|"
	   "\\<inherit\\>" "\\|"
	   "\\<method\\>" "\\|"
	   "\\<private\\>" "\\|"
	   "\\<public\\>" "\\|"
	   ;;"\\<pvs\\>" "\\|"
	   "\\<renaming\\>" "\\|"
	   "\\<self\\>" "\\|"
	   "\\<selfvar\\>" "\\|"
	   "\\<type\\>" "\\|"
	   "\\<groundsignature\\>" "\\|"
	   "\\<constant\\>" "\\|"
	   "\\<bool\\>" "\\|"
	   "\\<var\\>" "\\|"
	   "\\<forall\\>" "\\|"
	   "\\<exists\\>" "\\|"
	   "\\<cases\\>" "\\|"
	   "\\<of\\>" "\\|"
	   "\\<endcases\\>" "\\|"
	   "\\<implies\\>" "\\|"
	   "\\<iff\\>" "\\|"
	   "\\<or\\>" "\\|"
	   "\\<let\\>" "\\|"
	   "\\<groundterm\\>" "\\|"
	   "\\<groundtype\\>" "\\|"
	   "\\<if\\>" "\\|"
	   "\\<then\\>" "\\|"
	   "\\<else\\>" "\\|"
	   "\\<always\\>" "\\|"
	   "\\<eventually\\>" "\\|"
	   "\\<lambda\\>" "\\|"
	   "\\<for\\>" "\\|"
	   "\\<pos\\>" "\\|"
	   "\\<neg\\>" "\\|"
	   "\\<mixed\\>" "\\|"
;	   "\\<\\>" "\\|"
	   "\\<not\\>" 
	   "\\)"))
	 ;;(comment-regexp "\\s<.*\\s>")
	 ;;(ccsl-string-regexp
	 ;; (concat
	 ;;  "\""
	 ;;  "\\(" ".*" "\\)"
	 ;;  "\""))
	 (pvs-string-regexp
	  (concat
	   "\\("
	   "\\<pvs\\>" "\\|"
	   "\\<endpvs\\>" "\\)")))
    (list
     ;; keywords
     (list keyword-regexp '(1 font-lock-keyword-face))
     ;; comments (Are already colored in syntactic phase.)
     ;;(list comment-regexp '(0 font-lock-comment-face t))
     ;; ccsl strings (Are already colored in syntactic phase.)
     ;;(list ccsl-string-regexp '(1 font-lock-string-face))
     ;; pvs strings
     (list pvs-string-regexp '(1 font-lock-string-face))))
  "Default expressions to highlight in Ccsl mode.")

;;;   Comment Style
;;;
(defun ccsl-set-comment-style (comment-style)
  "Comment style used in `ccsl-code' buffers.

A choice between pvs comment style ( % ..... ) or
ml comment style ( (* ..... *) ) can be made with
argument 'pvs or 'ml.
With no argument the default, pvs comment style, is chosen."
  (cond
   ((eq comment-style 'pvs)
    (setq ccsl-comment-style 'pvs
	  comment-start "% "
	  comment-end ""
	  comment-start-skip "%+ "
	  comment-column 4
	  comment-multi-line 1))
   ((eq comment-style 'ml)
    (setq ccsl-comment-style 'ml
	  comment-start "(* "
	  comment-end " *)"
	  comment-start-skip "([*]+ "
	  comment-column 4
	  comment-multi-line 1))
   (t
    (message "unknown comment style, pvs comment style is chosen")
    (setq ccsl-comment-style 'pvs
	  comment-start "% "
	  comment-end ""
	  comment-start-skip "%+ "
	  comment-column 4
	  comment-multi-line 1))))

(defun ccsl-comment-indent ()
  "Comment indentation used in `ccsl-mode' buffers.

This is used by `indent-for-comment' to decide how much to
indent a comment in ccsl code based on its context."
  (cond
   ((looking-at "%")
    (current-column))
   ((looking-at "([*]")
    (current-column))
   (t
    (skip-chars-backward " \t")
    (max (if (bolp) 0 (1+ (current-column))) comment-column))))

(defun ccsl-comment-region (beg end &optional arg)
  ""
  (interactive "r\nP")
  (comment-region beg end arg))

(defun ccsl-uncomment-region (beg end &optional arg)
  ""
  (interactive "r\nP")
  (comment-region beg end '(4)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  Error processing
;;

;; The error processing is taken and adopted from tuareg.el
;; from Albert Cohen.

(require 'compile)

;; Two regexp's to extract the range info.

(defconst ccsl-error-chars-same-line
  ".*(char \\([0-9]+\\)-\\([0-9]+\\))"
  "First regular expression extracting the character numbers
from an error message produced by the ccsl compiler.")

(defconst ccsl-error-chars-different-lines
  ".*(char \\([0-9]+\\)) - line \\([0-9]+\\) (char \\([0-9]+\\))"
  "Second regular expression extracting the character numbers
from an error message produced by the ccsl compiler.")

;; Wrapper around next-error to get the location.

(defadvice next-error (after ccsl-next-error activate)
 "Reads the extra positional information provided by the CCSL compiler.

Puts the point and the mark exactly around the erroneous program
fragment. The erroneous fragment is also temporarily highlighted if
possible."
 (if (eq major-mode 'ccsl-mode)
     (let ((pos (point)) (beg nil) (end nil) (oline nil))
       (save-excursion
	 (set-buffer compilation-last-buffer)
	 (save-excursion
	   (goto-char (window-point (get-buffer-window (current-buffer) t)))
	   (cond 
	    ((looking-at ccsl-error-chars-same-line)
	     (setq beg (+ (string-to-int (match-string 1)) pos)
		   end (+ (string-to-int (match-string 2)) pos)
		   ))
	    ((looking-at ccsl-error-chars-different-lines)
	     (setq beg (+ (string-to-int (match-string 1)) pos)
		   oline (string-to-int (match-string 2))
		   end (string-to-int (match-string 3)))
	     ))))
       (beginning-of-line)
       (if oline
	   (save-excursion
	     (goto-line oline)
	     (setq end (+ (point) end))))
       (if beg
	   (progn
	     (goto-char beg)
	     (push-mark end t t)))
       )))


;;;   Describe
;;;
(defun ccsl-mode-describe ()
  "Create a help buffer with a brief description of the ccsl-mode."
  (interactive)
  (with-output-to-temp-buffer "*Help*"
    (princ
     (format
      "Ccsl Mode:\nFor a complete description, type %s\n%s\n"
      (substitute-command-keys
       "\\<ccsl-mode-map>\\[describe-mode] from within a `ccsl-mode' buffer")
      (substitute-command-keys "\\{ccsl-mode-map}")))
    (save-excursion
      (set-buffer standard-output)
      (help-mode))
    (print-help-return-message)))

;;;###autoload
;;;
(defun ccsl-mode ()
  "Major mode used in `ccsl-mode' buffers.

This is a very simple major mode, providing mainly a little indentation
and the syntax of comments.  It uses its own keymap and has the some
variables for customizing indentation.  It has its own abbrev table
and its own syntax table.  Turning on `ccsl-mode' calls the value of
the variable `ccsl-mode-hook' with no args, if that value is non-nil."
  (interactive)
  (require 'easymenu)
  (require 'font-lock)
  (kill-all-local-variables)
  ;; mode stuff
  (setq major-mode 'ccsl-mode)
  (setq mode-name "ccsl")
  ;; mode map, abbrev table and  syntax table
  (use-local-map ccsl-mode-map)
  (setq local-abbrev-table ccsl-mode-abbrev-table)
  (set-syntax-table ccsl-mode-syntax-table)
  ;; font lock mode support.
  (make-local-variable 'font-lock-defaults)
  (setq font-lock-defaults '(ccsl-mode-font-lock-keywords nil t))
  ;; local variables
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'indent-relative-maybe)
  (make-local-variable 'comment-indent-function)
  (setq comment-indent-function 'ccsl-comment-indent)
  ;; comment
  (make-local-variable 'comment-start)
  (make-local-variable 'comment-end)
  (make-local-variable 'comment-column)
  (make-local-variable 'comment-start-skip)
  (make-local-variable 'comment-multi-line)
  (ccsl-set-comment-style ccsl-comment-style)
  ;; menubar-menu
  (if ccsl-mode-menubar-menu
      (easy-menu-add ccsl-mode-menubar-menu))
  ;; hooks
  (run-hooks 'ccsl-mode-hook))

;;;   ccsl-mode.el ends here
