(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: dep_lists.v,v 1.18 2013/04/10 11:17:14 tews Exp $
 *)

(** ** Dependent lists

      Dependent lists are a key datatype for the whole formalization,
      beause it ensures the well-formedness of proof trees. There is a
      partial nth function [dep_nth], predicate lifting
      [every_dep_nth] and several variants of map.
*)


Require Export lists.


Section Dep_lists.
  Variable A : Type.
  Variable T : A -> Type.

  (***************************************************************************)
  (** ***  dependent predicates  *)
  (***************************************************************************)

  Definition dep_prop : Type := forall(a : A), T a -> Prop.

  Definition subset_dep_prop(P Q : dep_prop) : Prop :=
    forall(a : A)(ta : T a), P a ta -> Q a ta.


  (***************************************************************************)
  (** ***  dependent lists  *)
  (***************************************************************************)

  Inductive dep_list : list A -> Type :=
    | dep_nil : dep_list []
    | dep_cons : forall(a : A)(al : list A), 
        T a -> dep_list al -> dep_list (a :: al).


  (***************************************************************************)
  (** *** Partial nth *)
  (***************************************************************************)

  Fixpoint dep_nth(al : list A)(tal : dep_list al)(n : nat)
                             (inside : n < length al) : T (nth al n inside) :=
    match tal 
      in (dep_list l)
      return (forall inside0 : n < length l, T (nth l n inside0))
    with
      | dep_nil => fun(inside0 : n < length []) => 
        nth_0_nil_tcc n inside0 _
      | dep_cons a al0 ta tal0 => fun(inside0 : n < length (a :: al0)) =>
          match n 
            return (forall(inside1 : n < length (a :: al0)), 
                                          T (nth (a :: al0) n inside1))
          with
            | 0 => fun _ => ta 
            | S n0 => fun(inside1 : S n0 < length (a :: al0)) =>
              dep_nth al0 tal0 n0 (nth_succ_tcc n0 a al0 inside1)
          end inside0
    end inside.

  Lemma nth_dep_nth_tcc_irr :
    forall(B : Type)(f : forall(a : A), T a -> B)
          (al : list A)(tal : dep_list al)
          (n : nat)(n_less_1 n_less_2 : n < length al),
      f (nth al n n_less_1) (dep_nth al tal n n_less_1) =
      f (nth al n n_less_2) (dep_nth al tal n n_less_2).     
  Proof.
    induction tal.
      intros n n_less_1 n_less_2.
      eapply nth_0_nil_tcc; eauto.
    intros n n_less_1 n_less_2.
    destruct n.
      trivial.
    simpl in *.
    apply IHtal.
  Qed.


  (***************************************************************************)
  (** *** Predicate lifting via [dep_nth] *)
  (***************************************************************************)

  Definition every_dep_nth(P : dep_prop)(al : list A)(tal : dep_list al) 
                                                                      : Prop :=
    forall(n : nat)(n_less : n < length al), 
      P (nth al n n_less) (dep_nth al tal n n_less).

  Lemma every_dep_nth_empty : forall(P : dep_prop),
    every_dep_nth P [] dep_nil.
  Proof.
    unfold every_dep_nth in *.
    intros P n n_less.
    eapply nth_0_nil_tcc; eauto.
  Qed.

  Lemma every_dep_nth_cons : 
    forall(P : dep_prop)(a : A)(ta : T a)(l : list A)(tl : dep_list l),
      P a ta -> 
      every_dep_nth P l tl -> 
        every_dep_nth P (a :: l) (dep_cons a l ta tl).
  Proof.
    unfold every_dep_nth in *.
    intros P a ta l tl H H0 n n_less.
    destruct n.
      trivial.
    simpl.
    apply H0.
  Qed.

  Lemma every_dep_nth_head : 
    forall(P : dep_prop)(a : A)(ta : T a)(l : list A)(tl : dep_list l),
      every_dep_nth P (a :: l) (dep_cons a l ta tl) -> 
        P a ta.
  Proof.
    intros P a ta l tl H.
    apply (H 0 (lt_0_Sn (length l))).
  Qed.

  Lemma every_dep_nth_tail :
    forall(P : dep_prop)(a : A)(ta : T a)(l : list A)(tl : dep_list l),
      every_dep_nth P (a :: l) (dep_cons a l ta tl) -> 
        every_dep_nth P l tl.
  Proof.
    intros P a ta l tl H i i_less.
    specialize (H (S i) (lt_n_S _ _ i_less)).
    simpl in H.
    erewrite nth_dep_nth_tcc_irr.
    eexact H.
  Qed.


  (***************************************************************************)
  (** *** Map [list -> dep_list] and [dep_list -> list] *)
  (***************************************************************************)

  Fixpoint dep_map_const_dep(f : forall(a : A), T a)(al : list A) 
                                                              : dep_list al :=
    match al with
      | [] => dep_nil 
      | a :: al => dep_cons a al (f a) (dep_map_const_dep f al)
    end.

  Fixpoint dep_map_dep_const(B : Type)(f : forall(a : A), T a -> B)
                                (al : list A)(tal : dep_list al) : list B :=
    match tal with
      | dep_nil => []
      | dep_cons a al ta tal =>
          (f a ta) :: (dep_map_dep_const B f al tal)
    end.

  Lemma length_dep_map_dep_const : 
    forall(B : Type)(f : forall(a : A), T a -> B)
          (al : list A)(tal : dep_list al),
      length (dep_map_dep_const B f al tal) = length al.
  Proof.
    induction tal.
      trivial.
    simpl.
    rewrite IHtal.
    trivial.
  Qed.

  Lemma nth_dep_map_dep_const_tcc :
    forall(B : Type)(f : forall(a : A), T a -> B)
          (al : list A)(tal : dep_list al)(n : nat),
      n < length (dep_map_dep_const B f al tal) ->
        n < length al.
  Proof.
    intros B f al tal n H.
    rewrite length_dep_map_dep_const in H.
    trivial.
  Qed.

  Lemma nth_dep_map_dep_const :
    forall(B : Type)(f : forall(a : A), T a -> B)
          (al : list A)(tal : dep_list al)
          (n : nat)(n_less : n < length (dep_map_dep_const B f al tal)),
      nth (dep_map_dep_const B f al tal) n n_less =
        f (nth al n (nth_dep_map_dep_const_tcc B f al tal n n_less)) 
          (dep_nth al tal n (nth_dep_map_dep_const_tcc B f al tal n n_less)).
  Proof.
    induction tal.
      intros n n_less.
      exfalso.
      simpl in *.
      omega.
    intros n n_less.
    destruct n.
      simpl in *.
      trivial.
    simpl in *.
    rewrite IHtal.
    apply nth_dep_nth_tcc_irr.
  Qed.

  Lemma nth_dep_map_dep_const_inv_tcc :
    forall(B : Type)(f : forall(a : A), T a -> B)
          (al : list A)(tal : dep_list al)(n : nat),
      n < length al ->
        n < length (dep_map_dep_const B f al tal).
  Proof.
    intros B f al tal n H.
    rewrite length_dep_map_dep_const.
    trivial.
  Qed.

  Lemma nth_dep_map_dep_const_inv :
    forall(B : Type)(f : forall(a : A), T a -> B)
          (al : list A)(tal : dep_list al)
          (n : nat)(n_less : n < length al),
      f (nth al n n_less) (dep_nth al tal n n_less) =
        nth (dep_map_dep_const B f al tal) n 
            (nth_dep_map_dep_const_inv_tcc B f al tal n n_less).
  Proof.
    intros B f al tal n n_less.
    rewrite nth_dep_map_dep_const.
    apply nth_dep_nth_tcc_irr.
  Qed.


  (***************************************************************************)
  (** *** Results for [every_dep_nth] *)
  (***************************************************************************)

  Lemma every_dep_nth_dep_map_const :
    forall(B : Type)(f : forall(a : A), T a -> B)(P : B -> Prop)
          (al : list A)(tal : dep_list al),
      every_nth P (dep_map_dep_const B f al tal) ->
        every_dep_nth (fun(a : A)(ta : T a) => P (f a ta)) al tal.
  Proof.
    intros B f P al tal H n n_less.
    assert (n < length (dep_map_dep_const B f al tal)).
      rewrite length_dep_map_dep_const.
      trivial.
    specialize (H n H0).
    rewrite nth_dep_map_dep_const in H.
    erewrite nth_dep_nth_tcc_irr.
    eexact H.
  Qed.


  Lemma every_nth_exists :
    forall(Q : dep_prop)(l : list A),
      every_nth (fun(a : A) => exists(b : T a), Q a b) l ->
        exists(dl : dep_list l), every_dep_nth Q l dl.
  Proof.
    induction l.
      intros H.
      exists dep_nil.
      apply every_dep_nth_empty.
    intros H.
    specialize (IHl (every_nth_tail _ _ _ H)).
    destruct IHl.
    assert (H1 := every_nth_head _ _ _ H).
    destruct H1.
    exists (dep_cons a l x0 x). 
    apply every_dep_nth_cons.
      trivial.
    trivial.
  Qed.

  Lemma every_nth_exists_apply :
    forall(P : A -> Prop)(Q : dep_prop)(l : list A),
      every_nth (fun(a : A) => P a -> exists(b : T a), Q a b) l ->
      every_nth P l ->
        exists(dl : dep_list l), every_dep_nth Q l dl.
  Proof.
    intros P Q l H H0.
    apply every_nth_exists.
    intros n n_less.
    specialize (H0 n n_less).
    specialize (H n n_less H0).
    trivial.
  Qed.


  Lemma every_nth_exists_inv :
    forall(Q : dep_prop)(l : list A)(dl : dep_list l),
      every_dep_nth Q l dl ->
        every_nth (fun(a : A) => exists(b : T a), Q a b) l.
  Proof.
    induction dl.
      intros H.
      apply every_nth_empty.
    intros H.
    apply every_nth_cons.
      apply every_dep_nth_head in H.
      exists t.
      trivial.
    apply IHdl.
    apply every_dep_nth_tail in H.
    trivial.
  Qed.

End Dep_lists.  

Implicit Arguments dep_nil [A T].
Implicit Arguments dep_cons [A T].
Implicit Arguments dep_nth [A T].
Implicit Arguments every_dep_nth [A T].
Implicit Arguments every_dep_nth_empty [A T].
Implicit Arguments every_dep_nth_cons [A T].
Implicit Arguments every_dep_nth_head [A T].
Implicit Arguments every_dep_nth_tail [A T].
Implicit Arguments dep_map_const_dep [A T].
Implicit Arguments dep_map_dep_const [A T B].
Implicit Arguments nth_dep_map_dep_const_tcc [A T B f al tal n].
Implicit Arguments every_nth_exists [A T].
Implicit Arguments every_nth_exists_apply [A T].


  (***************************************************************************)
  (** *** Map [dep_list -> dep_list] *)
  (***************************************************************************)

Fixpoint dep_map_dep_dep{A : Type}{T1 T2 : A -> Type}
                        (f : forall(a : A), T1 a -> T2 a)
                        (al : list A)(dal : dep_list A T1 al) 
                                                          : dep_list A T2 al :=
  match dal with
    | dep_nil => dep_nil
    | dep_cons a al da dal => dep_cons a al (f a da) (dep_map_dep_dep f al dal)
  end.



Fixpoint dep_list_proj_left{A : Type}{T : A -> Type}{B : Type}
           (al : list A)(tsal : dep_list A (fun(a : A) => sum (T a) B) al) :
                                                     (dep_list A T al) + B :=
  match tsal with
    | dep_nil => inl dep_nil
    | dep_cons a al (inr b) _ => inr b
    | dep_cons a al (inl ta) tsal =>
      match dep_list_proj_left al tsal with
        | inr b => inr b
        | inl tal => inl (dep_cons a al ta tal)
      end
  end.


Lemma dep_list_proj_left_dep_map : 
  forall(A : Type)(T : A -> Type)(B : Type)
        (f : forall(a : A), sum (T a) B)(al : list A)(b : B),
    dep_list_proj_left al (dep_map_const_dep f al) = inr b ->
      exists n : nat,
        n_less # n < length al /#\ f (nth al n n_less) = inr b.
Proof.
  induction al.
    intros b H.
    simpl in H.
    inversion H.
  intros b H.
  simpl in H.
  destruct (f a) eqn:?.
    destruct (dep_list_proj_left al (dep_map_const_dep f al)) eqn:?.
      inversion H.
    inversion H.
    subst b0.
    lapply (IHal b); clear IHal.
      intros H0.
      decompose [ex and or dep_and] H0; clear H0.
      exists (S x).
      assert (S x < length (a :: al)).
        simpl.
        omega.
      constructor 1 with (a := H0).
      simpl.
      erewrite nth_tcc_irr.
      exact b0.
    trivial.
  exists 0.
  assert (0 < length (a :: al)).
    simpl.
    omega.
  constructor 1 with (a := H0).
  simpl.
  inversion H.
  subst b0.
  trivial.
Qed.


Fixpoint list_of_dep_list{A B : Type}(l : list A)
                         (dl : dep_list A (fun _ => B) l) : list B :=
  match dl with
    | dep_nil => []
    | dep_cons _ l b dl => b :: (list_of_dep_list l dl)
  end.

Lemma list_of_dep_list_dep_map_dep_dep :
  forall(A : Type)(T : A -> Type)(B : Type)
        (f : forall(a : A), T a -> B)
        (al : list A)(tal : dep_list A T al),
    list_of_dep_list al (dep_map_dep_dep f al tal)
      = dep_map_dep_const f al tal.
Proof.
  induction tal.
    simpl.
    trivial.
  simpl.
  rewrite IHtal.
  trivial.
Qed.


  (***************************************************************************)
  (** *** More results on [every_dep_nth] *)
  (***************************************************************************)

Lemma every_nth_of_dep_list : 
  forall{A : Type}{P : A -> Prop}(l : list A),
    (exists pl : dep_list A P l, True) -> every_nth P l.
Proof.
  induction l.
    intros H.
    apply every_nth_empty.
  intros H.
  decompose [ex] H; clear H.
  remember (a :: l) as al.
  destruct x.
    discriminate.
  apply every_nth_cons.
    trivial.
  inversion Heqal.
  apply IHl.
  subst al.
  exists x.
  trivial.
Qed.


Fixpoint dep_list_of_every_nth{A : Type}{P : A -> Prop}(l : list A)
                                    (p_l : every_nth P l) : dep_list A P l :=
  match l return every_nth P l -> dep_list A P l with
    | [] => fun _ => dep_nil 
    | a :: l => fun(H : every_nth P (a :: l)) =>
      dep_cons a l (every_nth_head _ _ _ H)
        (dep_list_of_every_nth l (every_nth_tail _ _ _ H))
  end p_l.

(* 
 * Lemma dep_list_of_every_nth : 
 *   forall{A : Type}{P : A -> Prop}(l : list A),
 *     every_nth P l -> exists pl : dep_list A P l, True.
 * Proof.
 *   induction l.
 *     intros H.
 *     exists dep_nil.
 *     trivial.
 *   intros H.
 *   assert (H0 := every_nth_head _ _ _ H).
 *   assert (H1 := every_nth_tail _ _ _ H).
 *   specialize (IHl H1).
 *   decompose [ex] IHl; clear IHl.
 *   exists (dep_cons a l H0 x).
 *   trivial.
 * Qed.
 *)


