(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: factor_subst.v,v 1.13 2013/04/10 11:17:14 tews Exp $
 *)

(** ** Factorize substitutions

      The syntactic proof of contraction and cut elimination need to
      factorize a substituion into an injective substitution and a
      renaming. To stay constructive, I do something weaker. Given a
      sequent [s] and a substitutions [sigma], I construct an
      injective [sigma_i] and a renaming [sigma_r], such that 
      [[
    subst_sequent sigma s = subst_sequent (subst_compose sigma_i sigma_r) s
      ]]
      The main idea for the construction is simple. Starting from the
      identity substitution and the identity renaming, I consider all
      propositional variables [v] in [s]. If I find [sigma v] in the
      codomain of [sigma_i], then I add a suitable renaming in
      [sigma_r]. Otherwise the mapping [v -> sigma v] is added to
      sigma_i.

      In order to work with finite, searchable data structures, I
      first collect the mappings for [sigma_i] and [sigma_r] in lists.
      At the end, these lists are converted into functions.

      A problem arises if [sigma] maps [v] to [lf_prop v'] for a [v']
      that does not appear in [s]. Then the resulting substitution
      [sigma_i] would map both [v] and [v'] to [lf_prop v'], breaking
      injectivity.

      To fix this problem, I collect all such problematic [v'], which
      are called variable-chain end points, [var_chain_ends]. For each
      such [v'] I add a mapping [v' -> f] to [sigma_i], for an [f]
      that is unique in the codomain of [sigma_i]. To construct such
      an [f], I include [v'] in [f] and the conjuntion of all formulas
      in [s]. Thus, [f] is bigger then all formulas that were
      originally in the codomain of [sigma_i] and it is different from
      all mappings that are added to fix variable-chain ends.

      The contraction proof relies directly on the properties proved
      here. The cut-elimination proof needs something more
      complicated, see [factor_two_subst].
*)

Require Export assoc renaming sequent_support.

Section Factor_subst.

  Variable V : Type.
  Variable L : modal_operators.

  (** Need decidable equalitities for the factoring *)
  Variable op_eq : eq_type (operator L).
  Variable v_eq : eq_type V.


  (***************************************************************************)
  (** *** Functions definitions *)
  (***************************************************************************)

  (** [subst] is the substitution to factorize and [pv] are the
      variables occurring in [s] (see above). The first returned list
      are the mappings for an injective substitution, the second are
      the mappings for the renaming.
   *)
  Fixpoint divide_subst(subst : lambda_subst V L)
                       (pv : list V)
                             : list (V * lambda_formula V L) * list (V * V) :=
    match pv with 
      | [] => ([], [])
      | v :: pv =>
        let (smap, rmap) := divide_subst subst pv 
        in
          match rassoc (lambda_formula_eq op_eq v_eq) smap (subst v) with
            | Some v' => (smap, (v, v') :: rmap)
            | None => ((v, subst v) :: smap, rmap)
          end
    end.

  (** Collect the variable-chain end points in the substitution
      mapping
   *)
  Fixpoint collect_var_chain_ends(map orig : list (V * lambda_formula V L))
                                                                   : list V :=
    match map with
      | [] => []
      | (v, f) :: map => match f with
          | lf_prop v' => match assoc v_eq orig v' with
              | Some _ => collect_var_chain_ends map orig
              | None => v' :: (collect_var_chain_ends map orig)
            end
          | _ => collect_var_chain_ends map orig
        end
    end.

  (** [(v, f) :: l] are the mappings for the injective substitution.
      They are here split into head and tail, because this function
      works only for non-empty lists of mappings.
   *)
  Fixpoint make_unique_form(v : V)(f : lambda_formula V L)
                           (l : list (V * lambda_formula V L))
                                                        : lambda_formula V L :=
    match l with
      | [] => lf_and (lf_prop v) f
      | (v', f') :: l => lf_and f (make_unique_form v' f' l)
    end.

  Definition subst_of_map(map : list (V * lambda_formula V L)) 
                                                          : lambda_subst V L :=
    fun(v : V) => match assoc v_eq map v with
      | None => lf_prop v
      | Some f => f
    end.

  Definition rename_of_map(rmap : list (V * V)) : lambda_subst V L :=
    subst_of_map (map (fun vv => (fst vv, lf_prop (snd vv))) rmap).

  (** Fix the mappings in [smap] such that the result is an injective
      assoc list. 
    *)
  Definition fix_var_chain_ends(smap : list (V * lambda_formula V L))
                                             : list (V * lambda_formula V L) :=
    match smap with
      | [] => []
      | (v1, f1) :: mtl =>
        (map (fun v => (v, lf_and (lf_prop v) (make_unique_form v1 f1 mtl)))
             (collect_var_chain_ends smap smap))
        ++ smap
    end.

  Definition factor_subst(subst : lambda_subst V L)(s : sequent V L) :
                                     (lambda_subst V L) * (lambda_subst V L) :=
    let (smap, rmap) := divide_subst subst (prop_var_sequent s)
    in (rename_of_map rmap, subst_of_map (fix_var_chain_ends smap)).


  (***************************************************************************)
  (** *** rename_of_map properties  *)
  (***************************************************************************)

  Lemma rename_of_map_cons_split :
    forall(P : lambda_formula V L -> Prop)(rmap : list (V * V))(v1 v2 v3 : V),
      (v1 = v3 -> P (lf_prop v2)) ->
      (v1 <> v3 -> P (rename_of_map rmap v3)) ->
        P (rename_of_map ((v1, v2) :: rmap) v3).
  Proof.
    intros P rmap v1 v2 v3 H H0.
    unfold rename_of_map, subst_of_map in *.
    simpl map.
    apply assoc_cons_split.
      apply H.
    apply H0.
  Qed.

  Lemma rename_of_map_prop :
    forall(rmap : list (V * V))(v : V),
      (exists(v' : V), assoc v_eq rmap v = Some v' /\ 
                         rename_of_map rmap v = lf_prop v') \/
      (assoc v_eq rmap v = None /\ rename_of_map rmap v = lf_prop v).
  Proof.
    clear op_eq.
    intros rmap v.
    unfold rename_of_map, subst_of_map in *.
    rewrite assoc_map_pair.
    destruct (assoc v_eq rmap v) eqn:H.
      left.
      exists v0.
      auto.
    right.
    auto.
  Qed.


  (***************************************************************************)
  (** *** Utility predicates  *)
  (***************************************************************************)

  (***************************************************************************)
  (** **** map_subst_correct *)
  (***************************************************************************)

  (** map_subst_correct expresses that
       - (smap, rmap) agree with original subst
       - (smap, rmap) are defined for just the right variables
       - codomain values in rmap are mapped in smap
   *)
  Definition map_subst_correct(subst : lambda_subst V L)
                              (smap : list (V * lambda_formula V L))
                              (rmap : list (V * V))
                              (pv : list V) : Prop :=
    subst_eq_on_vars 
      subst 
      (subst_compose (subst_of_map smap) (rename_of_map rmap))
      pv /\
    (forall(v : V), 
       In v pv 
          <->
       (is_some (assoc v_eq smap v) \/ is_some (assoc v_eq rmap v)))
    /\
    (forall(v1 v2 : V),
       assoc v_eq rmap v1 = Some v2 -> is_some (assoc v_eq smap v2)).

  Lemma map_subst_correct_empty :
    forall(subst : lambda_subst V L),
      map_subst_correct subst [] [] [].
  Proof.
    unfold map_subst_correct.
    intros subst.
    repeat split.
          apply subst_eq_on_vars_empty.
        intros H.
        contradiction.
      intros H.
      simpl in *.
      tauto.
    intros v1 v2 H.
    simpl in *.
    discriminate.
  Qed.

  Lemma map_subst_correct_cons_rmap :
    forall(subst : lambda_subst V L)(smap : list (V * lambda_formula V L))
          (rmap : list (V * V))(pv : list V)(v1 v2 : V),
      map_subst_correct subst smap rmap pv ->
      assoc v_eq smap v2 = Some (subst v1) ->
        map_subst_correct subst smap ((v1, v2) :: rmap) (v1 :: pv).
  Proof.
    unfold map_subst_correct in *.
    intros subst smap rmap pv v1 v2 H H0.
    decompose [and] H; clear H.
    split.
      intros v3 H5.
      destruct H5.
        subst v3.
        unfold subst_compose.
        apply rename_of_map_cons_split.
          intros H2; clear H2.
          rewrite subst_form_char.
          unfold subst_of_map.
          rewrite H0.
          trivial.
        intros H2.
        exfalso.
        apply H2.
        trivial.
      unfold subst_compose.
      apply rename_of_map_cons_split.
        intros H2.
        subst v3.
        rewrite subst_form_char.
        unfold subst_of_map.
        rewrite H0.
        trivial.
      intros H2.
      apply H1.
      trivial.
    split.
      intros v3.
      split.
        intros H2.
        destruct H2.
          subst v3.
          right.
          rewrite assoc_cons_first.
            simpl.
            trivial.
          trivial.
        apply assoc_cons_split.
            trivial.
          right.
          simpl.
          trivial.
        intros H2; clear H2.
        apply H3.
        trivial.
      intros H2.
      destruct H2.
        apply in_cons.
        apply H3.
        auto.
      revert H.
      apply assoc_cons_split.
          trivial.
        intros H H2.
        left.
        trivial.
      intros H H2.
      apply in_cons.
      apply H3.
      auto.
    intros v3 v4.
    apply assoc_cons_split.
        trivial.
      intros H H2.
      inversion H2; clear H2.
      subst v4.
      rewrite H0.
      simpl.
      trivial.
    intros H.
    apply H4.
  Qed.

  Lemma map_subst_correct_cons_smap :
    forall(subst : lambda_subst V L)(smap : list (V * lambda_formula V L))
          (rmap : list (V * V))(pv : list V)(v : V),
      rassoc (lambda_formula_eq op_eq v_eq) smap (subst v) = None ->
      map_subst_correct subst smap rmap pv ->
        map_subst_correct subst ((v, subst v) :: smap) rmap (v :: pv).
  Proof.
    unfold map_subst_correct in *.
    intros subst smap rmap pv v1 H H0.
    decompose [and] H0; clear H0.
    split.
      intros v2 H5.
      destruct H5.
        subst v2.
        specialize (H1 v1).
        unfold subst_compose in *.
        decompose [ex and or] (rename_of_map_prop rmap v1).
          rename x into v2.
          rewrite H5 in *.
          rewrite subst_form_char in *.
          unfold subst_of_map.
          apply assoc_cons_split.
              trivial.
            trivial.
          intros H2.
          apply H1.
          apply H3.
          rewrite H0.
          simpl.
          auto.
        rewrite H5 in *.
        rewrite subst_form_char.
        unfold subst_of_map in *.
        rewrite assoc_cons_first.
          trivial.
        trivial.
      assert (H5 := H1 v2 H0).
      unfold subst_compose in H5 |-*.
      decompose [ex and or] (rename_of_map_prop rmap v2).
        rename x into v3.
        rewrite H7 in *.
        rewrite subst_form_char in *.
        unfold subst_of_map.
        apply assoc_cons_split.
            trivial.
          intros H6.
          subst v3.
          exfalso.
          apply H4 in H2.
          lapply (H1 v1); clear H1.
            intros H1.
            unfold subst_compose in *.
            decompose [ex and or] (rename_of_map_prop rmap v1).
              rename x into v4.
              rewrite H9 in *.
              rewrite subst_form_char in *.
              apply H4 in H6.
              unfold subst_of_map in H1.
              destruct (assoc v_eq smap v4) eqn:H10.
                subst l.
                apply rassoc_assoc_none with (aeq := v_eq)(a := v4) in H.
                auto.
              contradiction.
            rewrite H9 in *.
            rewrite subst_form_char in *.
            unfold subst_of_map in H1.
            destruct (assoc v_eq smap v1) eqn:H10.
              subst l.
              apply rassoc_assoc_none with (aeq := v_eq)(a := v1) in H.
              auto.
            contradiction.
          apply H3.
          auto.
        intros H6.
        apply H5.
      rewrite H7 in *.
      rewrite subst_form_char in *.
      unfold subst_of_map.
      apply assoc_cons_split.
          trivial.
        intros H2.
        subst v2.
        trivial.
      intros H2.
      apply H5.
    split.
      clear H1 H4.
      intros v2.
      split.
        intros H0.
        destruct H0.
          subst v2.
          left.
          rewrite assoc_cons_first.
            simpl.
            trivial.
          trivial.
        apply assoc_cons_split.
            trivial.
          intros H1.
          simpl.
          auto.
        intros H1.
        apply H3.
        trivial.
      apply assoc_cons_split.
          trivial.
        intros H0 H1.
        subst v2.
        left.
        trivial.
      intros H0 H1.
      right.
      apply H3.
      trivial.
    clear H1 H3.
    intros v2 v3 H0.
    apply assoc_cons_split.
        trivial.
      simpl.
      trivial.
    intros H1.
    eapply H4.
    eexact H0.
  Qed.


  (***************************************************************************)
  (** **** map_rank
          The substitution mapping has some rank.
   *)
  (***************************************************************************)
  Definition map_rank(n : nat)(map : list (V * lambda_formula V L)) : Prop :=
    forall(v : V), 
      match assoc v_eq map v with
        | None => True
        | Some f => rank_formula (S n) f
      end.

  Lemma map_rank_empty : forall(n : nat), map_rank n [].
  Proof.
    clear. 
    unfold map_rank in *.
    intros n v.
    simpl.
    trivial.
  Qed.

  Lemma map_rank_cons :
    forall(n : nat)(map : list (V * lambda_formula V L))
          (v : V)(f : lambda_formula V L),
      rank_formula (S n) f ->
      map_rank n map ->
        map_rank n ((v, f) :: map).
  Proof.
    unfold map_rank in *.
    intros n map v f H H0 v'.
    apply assoc_cons_split.
      intros H1.
      trivial.
    intros H1.
    apply H0.
  Qed.

  Lemma map_rank_head : 
    forall(n : nat)(v : V)(f : lambda_formula V L)
          (map : list (V * lambda_formula V L)),
      map_rank n ((v, f) :: map) ->
        rank_formula (S n) f.
  Proof.
    unfold map_rank in *.
    intros n v f map H.
    specialize (H v).
    rewrite assoc_cons_first in H; trivial.
  Qed.

  Lemma map_rank_tail :
    forall(n : nat)(v : V)(f : lambda_formula V L)
          (map : list (V * lambda_formula V L)),
      assoc_mapping v_eq ((v, f) :: map) ->
      map_rank n ((v, f) :: map) ->
        map_rank n map.
  Proof.
    unfold map_rank in *.
    intros n v f map H H0 v'.
    specialize (H0 v').
    revert H0.
    apply assoc_cons_split.
      intros H0 H2.
      subst v'.
      apply assoc_mapping_cons_first in H.
      destruct (assoc v_eq map v).
        contradiction.
      trivial.
    trivial.
  Qed.

  Lemma map_rank_In : 
    forall(n : nat)(v : V)(f : lambda_formula V L)
          (map : list (V * lambda_formula V L)),
      In (v, f) map ->
      map_rank n map ->
      assoc_mapping v_eq map ->
        rank_formula (S n) f.
  Proof.
    induction map.
      intros H H0 H1.
      contradiction.
    intros H H0 H1.
    destruct a as [v' f'].
    destruct H.
      inversion H; clear H.
      subst v' f'.
      apply map_rank_head in H0.
      trivial.
    apply IHmap.
        trivial.
      apply map_rank_tail in H0; trivial.
    apply assoc_mapping_tail in H1.
    trivial.
  Qed.

  Lemma map_rank_append :
    forall(n : nat)(map1 map2 : list (V * lambda_formula V L)),
      map_rank n map1 ->
      map_rank n map2 ->
      assoc_mapping v_eq map1 ->
        map_rank n (map1 ++ map2).
  Proof.
    induction map1.
      intros map2 H H0 H1.
      trivial.
    intros map2 H H0 H1.
    destruct a as [v f].
    simpl.
    apply map_rank_cons.
      apply map_rank_head in H.
      trivial.
    apply IHmap1.
      apply map_rank_tail in H; trivial.
      trivial.
    apply assoc_mapping_tail in H1.
    trivial.
  Qed.

  Lemma map_rank_incl :
    forall(n : nat)(map1 map2 : list (V * lambda_formula V L)),
      incl map1 map2 ->
      map_rank n map2 ->
      assoc_mapping v_eq map2 ->
        map_rank n map1.
  Proof.
    induction map1.
      intros map2 H H0 H1.
      apply map_rank_empty.
    intros map2 H H0 H1.
    destruct a as [v f].
    apply map_rank_cons.
      lapply (H (v, f)); clear H.
        intros H.
        eapply map_rank_In; eauto.
      left.
      trivial.
    apply IHmap1 with (map2 := map2); trivial.
    apply incl_left_tail in H.
    trivial.
  Qed.


  (***************************************************************************)
  (** **** var_chain_ends
          defining property of the variable-chain end points
   *)
  (***************************************************************************)
  Definition var_chain_ends(map : list (V * lambda_formula V L))
                           (l : list V) : Prop :=
    forall(v : V), 
      In v l <->
        (is_some (rassoc (lambda_formula_eq op_eq v_eq) map (lf_prop v)) /\
         is_none (assoc v_eq map v)).


  (***************************************************************************)
  (** **** divide_subst Properties  *)
  (***************************************************************************)

  Lemma divide_subst_prop :
    forall(subst : lambda_subst V L)(n : nat)(pv : list V)
          (smap : list (V * lambda_formula V L))
          (rmap : list (V * V)),
      divide_subst subst pv = (smap, rmap) ->
      rank_subst (S n) subst ->
        injective_assoc v_eq smap /\ 
        map_subst_correct subst smap rmap pv /\
        assoc_mapping v_eq smap /\
        map_rank n smap.
  Proof.
    induction pv.
      intros smap rmap H H0.
      simpl in *.
      inversion_clear H.
      split.
        apply injective_assoc_empty.
      split.
        apply map_subst_correct_empty.
      split.
        apply assoc_mapping_nil.
      apply map_rank_empty.
    rename a into v.
    intros smap rmap H H0.
    simpl in *.
    destruct (divide_subst subst pv) as [smap_tail rmap_tail] eqn:H1.
    specialize (IHpv _ _ eq_refl H0).
    decompose [and] IHpv; clear IHpv.
    destruct (rassoc (lambda_formula_eq op_eq v_eq) smap_tail (subst v))
             as [v' |] eqn:H7.
      inversion H; clear H.
      subst smap_tail rmap.
      split; trivial.
      split.
        apply map_subst_correct_cons_rmap.
          trivial.
        apply rassoc_assoc_some with (beq := lambda_formula_eq op_eq v_eq);
              trivial.
      split; trivial.
    inversion H; clear H.
    subst smap rmap_tail.
    split.
      apply injective_assoc_cons_rassoc
            with (beq := lambda_formula_eq op_eq v_eq); trivial.
    split.
      apply map_subst_correct_cons_smap; trivial.
    split.
      apply assoc_mapping_cons.
        trivial.
      unfold map_subst_correct in *.
      decompose [and] H4; clear H4.
      specialize (H8 v).
      apply neg_is_none.
      intros H10.
      lapply (iff_left H8).
        intros H4.
        apply H in H4; clear H.
        unfold subst_compose in *.
        decompose [ex and or] (rename_of_map_prop rmap v).
          rename x into v'.
          rewrite H11 in *.
          rewrite subst_form_char in *.
          apply H9 in H.
          unfold subst_of_map in *.
          destruct (assoc v_eq smap_tail v') eqn:H12.
            subst l.
            apply rassoc_assoc_none with (aeq := v_eq)(a := v') in H7.
            auto.
          contradiction.
        clear H9.
        rewrite H11 in *.
        rewrite subst_form_char in *.
        unfold subst_of_map in *.
        destruct (assoc v_eq smap_tail v) eqn:H12.
          subst l.
          apply rassoc_assoc_none with (aeq := v_eq)(a := v) in H7.
          contradiction.
        contradiction.
      auto.
    apply map_rank_cons.
      apply H0.
    trivial.
  Qed.


  (***************************************************************************)
  (** ****  var_chain_ends Properties  *)
  (***************************************************************************)

  Lemma collect_var_chain_ends_correct_ind :
   forall(map orig : list (V * lambda_formula V L))(v : V),
     In v (collect_var_chain_ends map orig) 
        <->
     (is_some (rassoc (lambda_formula_eq op_eq v_eq) map (lf_prop v)) /\
      is_none (assoc v_eq orig v)).
  Proof.
    induction map.
      intros orig v.
      simpl in *.
      tauto.
    intros orig v.
    destruct a as [v1 f1].
    simpl collect_var_chain_ends.
    destruct f1 as [v2 | f1 | f1 f2 | op args].
          destruct (assoc v_eq orig v2) as [ass_o_v2 | ] eqn:H.
            specialize (IHmap orig v).
            split.
              intros H0.
              apply IHmap in H0; clear IHmap.
              destruct H0.
              split.
                apply rassoc_cons_split.
                  intros H2.
                  inversion H2; clear H2.
                  subst v2.
                  rewrite H in H1.
                  contradiction.
                trivial.
              trivial.
            apply rassoc_cons_split.
              intros H0 H1.
              destruct H1.
              inversion H0; clear H0.
              subst v2.
              rewrite H in H2.
              contradiction.
            tauto.
          specialize (IHmap orig v).
          split.
            intros H0.
            destruct H0.
              subst v2.
              split.
                rewrite rassoc_cons_first.
                simpl.
                trivial.
              rewrite H.
              simpl.
              trivial.
            apply rassoc_cons_split.
              intros H1.
              split.
                simpl.
                trivial.
              inversion H1; clear H1.
              subst v2.
              rewrite H.
              simpl.
              trivial.
            intros H1.
            apply IHmap.
            trivial.
          apply rassoc_cons_split.
            intros H0 H1.
            inversion H0; clear H0.
            left.
            trivial.
          intros H0 H1.
          right.
          apply IHmap.
          trivial.
        (* remaining trivial cases, where f was not a propositional formula *)
        rewrite rassoc_cons_tail.
          apply IHmap.
        discriminate.
      rewrite rassoc_cons_tail.
        apply IHmap.
      discriminate.
    rewrite rassoc_cons_tail.
      apply IHmap.
    discriminate.
  Qed.

  Lemma collect_var_chain_ends_correct :
    forall(map : list (V * lambda_formula V L)),
      var_chain_ends map (collect_var_chain_ends map map).
  Proof.
    unfold var_chain_ends in *.
    intros map v.
    apply collect_var_chain_ends_correct_ind.
  Qed.


  (***************************************************************************)
  (** ****  unique form Properties  *)
  (***************************************************************************)

  Lemma unique_form_greater :
    forall(map : list (V * lambda_formula V L))
          (v1 v2 : V)(f1 f2 : lambda_formula V L),
      assoc v_eq ((v1, f1) :: map) v2 = Some f2 ->
        formula_measure f2 < formula_measure (make_unique_form v1 f1 map).
  Proof.
    induction map.
      intros v1 v2 f1 f2.
      apply assoc_cons_split.
          trivial.
        intros H H0.
        inversion_clear H0.
        simpl.
        rewrite (formula_measure_char (lf_and _ _)).
        omega.
      simpl.
      discriminate.
    destruct a as [v2 f2].
    intros v1 v3 f1 f3.
    apply assoc_cons_split.
        trivial.
      intros H H0.
      inversion_clear H0.
      simpl.
      rewrite (formula_measure_char (lf_and _ _)).
      omega.
    intros H H0.
    apply IHmap in H0; clear IHmap.
    simpl.
    rewrite (formula_measure_char (lf_and _ _)).
    omega.
  Qed.

  Lemma rank_unique_form :
    forall(n : nat)(map : list (V * lambda_formula V L))
          (v : V)(f : lambda_formula V L),
      assoc_mapping v_eq ((v, f) :: map) ->
      map_rank n ((v, f) :: map) ->
        rank_formula (S n) (make_unique_form v f map).
  Proof.
    induction map.
      intros v f H H0.
      simpl.
      apply rank_formula_lf_and.
        apply rank_formula_lf_prop.
        apply le_n_S.
        apply le_0_n.
      apply map_rank_head in H0.
      trivial.
    destruct a as [v' f'].
    intros v f H H0.
    simpl.
    apply rank_formula_lf_and.
      apply map_rank_head in H0.
      trivial.
    apply IHmap.
      apply assoc_mapping_tail in H.
      trivial.
    apply map_rank_tail in H0; trivial.
  Qed.


  (***************************************************************************)
  (** ****  rename_of_map Properties  *)
  (***************************************************************************)

  Lemma renaming_rename_of_map : forall(m : list (V * V)),
    renaming (rename_of_map m). 
  Proof.
    unfold renaming, rename_of_map, subst_of_map in *.
    intros m v.
    decompose [and or ex] 
         (assoc_map_split v_eq
               (fun(vv : V * V) => (fst vv, lf_prop (L:=L) (snd vv))) m v).
      rewrite H0.
      simpl.
      trivial.
    rewrite H2.
    simpl.
    trivial.
  Qed.


  (***************************************************************************)
  (** ****  fix_var_chain Properties  *)
  (***************************************************************************)

  Lemma injective_assoc_fix_var_chain_ends :
    forall(m : list (V * lambda_formula V L)),
      injective_assoc v_eq m -> 
        injective_assoc v_eq (fix_var_chain_ends m).
  Proof.
    intros m H.
    unfold fix_var_chain_ends in *.
    destruct m.
      apply injective_assoc_empty.
    destruct p as [v1 f2].
    set (mapfun := fun v : V => (v, lf_and (lf_prop v)
                                    (make_unique_form v1 f2 m))).
    set (ve := (collect_var_chain_ends ((v1, f2) :: m) ((v1, f2) :: m))).
    apply injective_assoc_append.
          trivial.
        unfold injective_assoc in *.
        intros v2 v3 H0 H1.
        decompose [ex and or] (assoc_map_split v_eq mapfun ve v2).
          rewrite H3 in H1.
          contradiction.
        simpl in H3.
        subst x.
        rewrite H5 in *.
        clear H1 H2 H5.
        decompose [ex and or] (assoc_map_split v_eq mapfun ve v3).
          rewrite H2 in H0.
          discriminate.
        simpl in H2.
        subst x.
        rewrite H4 in *.
        clear H1 H4.
        simpl in H0.
        clear - H0.
        inversion_clear H0.
        trivial.
      trivial.
    intros v2 v3 H0 H1.
    decompose [ex and or] (assoc_map_split v_eq mapfun ve v2).
      rewrite H3 in *.
      contradiction.
    simpl in H3.
    subst x.
    rewrite H5 in *.
    clear H0 H2 H5.
    apply eq_sym in H1.
    apply unique_form_greater in H1.
    simpl in H1.
    rewrite formula_measure_char in H1.
    omega.
  Qed.

  Lemma fix_var_chain_ends_no_ends :
    forall(m : list (V * lambda_formula V L))(v1 v2 : V),
      assoc v_eq (fix_var_chain_ends m) v1 = Some (lf_prop v2) ->
      assoc v_eq (fix_var_chain_ends m) v2 = None ->
        False.
  Proof.
    intros m v1 v2.
    unfold fix_var_chain_ends in *.
    destruct m.
      simpl.
      discriminate.
    destruct p as [v3 f3].
    set (vf3m := (v3, f3) :: m).
    set (mapfun := fun v : V => (v, lf_and (lf_prop v)
                                    (make_unique_form v3 f3 m))).
    set (ve := (collect_var_chain_ends vf3m vf3m)).
    apply assoc_append_split.
        trivial.
      intros H H0 H1.
      clear H1.
      decompose [ex and or] (assoc_map_split v_eq mapfun ve v1).
        rewrite H2 in *.
        contradiction.
      rewrite H0 in *.
      clear - H4.
      simpl in *.
      discriminate.
    intros H H0.
    apply assoc_append_split.
        trivial.
      intros H1 H2.
      rewrite H2 in *.
      contradiction.
    intros H1 H2.
    decompose [ex and or] (assoc_map_split v_eq mapfun ve v2).
      apply H5 with (c := v2); trivial; clear H5.
      apply collect_var_chain_ends_correct.
      split.
        eapply assoc_rassoc_some.
        eexact H0.
      rewrite H2.
      simpl.
      trivial.
    rewrite H6 in *.
    contradiction.
  Qed.

  Lemma injective_subst_of_fixed_map : 
    forall(map : list (V * lambda_formula V L)),
      injective_assoc v_eq map ->
        injective (subst_of_map (fix_var_chain_ends map)).
  Proof.
    unfold injective in *.
    intros map H v1 v2 H0.
    unfold subst_of_map in *.
    destruct (assoc v_eq (fix_var_chain_ends map) v1) as [f1|] eqn:?.
      destruct (assoc v_eq (fix_var_chain_ends map) v2) as [f2|] eqn:?.
        apply injective_assoc_fix_var_chain_ends with (m := map).
            trivial.
          rewrite Heqo.
          rewrite Heqo0.
          subst f1.
          trivial.
        rewrite Heqo.
        simpl.
        trivial.
      subst f1.
      apply fix_var_chain_ends_no_ends in Heqo.
        contradiction.
      trivial.
    destruct (assoc v_eq (fix_var_chain_ends map) v2) as [f2|] eqn:?.
      subst f2.
      apply fix_var_chain_ends_no_ends in Heqo0.
        contradiction.
      trivial.
    inversion_clear H0.
    trivial.
  Qed.

  Lemma subst_subst_of_fixed_map :
    forall(subst : lambda_subst V L)(smap : list (V * lambda_formula V L))
          (v1 v2 : V),
      subst v1 = subst_of_map smap v2 ->
      is_some (assoc v_eq smap v2) ->
        subst v1 = subst_of_map (fix_var_chain_ends smap) v2.
  Proof.
    intros subst smap v1 v2 H H0.
    destruct smap as [| vf smap].
      trivial.
    destruct vf as [v3 f3].
    unfold subst_of_map, fix_var_chain_ends.
    set (vf3_smap := (v3, f3) :: smap) in *.
    set (mapfun := fun v : V => (v, lf_and (lf_prop v)
                                           (make_unique_form v3 f3 smap))).
    set (ve := (collect_var_chain_ends vf3_smap vf3_smap)).
    apply assoc_append_split.
        trivial.
      intros H1.
      exfalso.
      decompose [ex and or] (assoc_map_split v_eq mapfun ve v2).
        rewrite H3 in H1.
        contradiction.
      simpl in H3.
      subst x.
      apply collect_var_chain_ends_correct in H2.
      destruct H2.
      eapply option_contradiction.
        eexact H0.
      trivial.
    intros H1.
    apply H.
  Qed.


  Lemma weak_map_subst_correct_fix_var_chain_ends :
    forall(subst rsubst : lambda_subst V L)
          (smap : list (V * lambda_formula V L))(pv : list V),
      subst_eq_on_vars subst (subst_compose (subst_of_map smap) rsubst) pv ->
      renaming rsubst ->
      (forall(v1 v2 : V), 
             In v1 pv -> 
             rsubst v1 = lf_prop v2 -> 
               is_some (assoc v_eq smap v2)) ->
        subst_eq_on_vars 
          subst 
          (subst_compose (subst_of_map (fix_var_chain_ends smap)) rsubst)
          pv.
  Proof.
    unfold subst_eq_on_vars, subst_compose in *.
    intros subst rsubst smap pv H H0 H1 v H2.
    specialize (H0 v).
    specialize (H _ H2).
    specialize (H1 v).
    destruct (rsubst v) as [v1 | | |] eqn:H3; try contradiction.
    rewrite subst_form_char in *.
    apply subst_subst_of_fixed_map.
      trivial.
    apply H1; trivial.
  Qed.

  Lemma map_subst_correct_fix_var_chain_ends :
    forall(subst : lambda_subst V L)(smap : list (V * lambda_formula V L))
          (rmap : list (V * V))(pv : list V),
      map_subst_correct subst smap rmap pv ->
        subst_eq_on_vars 
          subst 
          (subst_compose (subst_of_map (fix_var_chain_ends smap)) 
                         (rename_of_map rmap))
          pv.
  Proof.
    intros subst smap rmap pv H.
    apply weak_map_subst_correct_fix_var_chain_ends.
        apply H.
      apply renaming_rename_of_map.
    unfold map_subst_correct in *.
    decompose [and] H; clear H.
    clear H0.
    intros v1 v2 H4 H5.
    decompose [ex and or] (rename_of_map_prop rmap v1).
      rename x into v3.
      rewrite H5 in *.
      inversion H1; clear H1.
      subst v3.
      apply H3 in H.
      trivial.
    rewrite H5 in *.
    inversion H1; clear H1.
    subst v2.
    apply (iff_right (H2 v1)) in H4.
    rewrite H0 in *.
    destruct H4.
      trivial.
    contradiction.
  Qed.

  Lemma rank_subst_subst_of_fixed_map :
    forall(n : nat)(m : list (V * lambda_formula V L)),
      assoc_mapping v_eq m ->
      map_rank n m ->
        rank_subst (S n) (subst_of_map (fix_var_chain_ends m)).
  Proof.
    intros n m H H0.
    destruct m.
      simpl.
      unfold rank_subst, subst_of_map in *.
      simpl.
      intros v.
      apply rank_formula_lf_prop.
      apply le_n_S.
      apply le_0_n.
    destruct p as [v1 f1].
    unfold rank_subst, subst_of_map, fix_var_chain_ends in *.
    set (vf1m := (v1, f1) :: m).
    set (mapfun := fun v0 : V => (v0, lf_and (lf_prop v0)
                                    (make_unique_form v1 f1 m))).
    set (ve := (collect_var_chain_ends vf1m vf1m)).
    intros v.
    apply assoc_append_split.
        trivial.
      intros H1.
      decompose [ex and or] (assoc_map_split v_eq mapfun ve v).
        rewrite H3 in *.
        contradiction.
      simpl in H3.
      subst x.
      clear H1.
      rewrite H5 in *.
      simpl.
      apply rank_formula_lf_and.
        apply rank_formula_lf_prop.
        apply le_n_S.
        apply le_0_n.
      apply rank_unique_form; trivial.
    intros H1.
    clear ve mapfun H1.
    subst vf1m.
    specialize (H0 v).
    destruct (assoc v_eq ((v1, f1) :: m) v).
      trivial.
    apply rank_formula_lf_prop.
    apply le_n_S.
    apply le_0_n.
  Qed.


  (***************************************************************************)
  (** *** Final Properties  *)
  (***************************************************************************)

  Lemma factor_subst_property :
    forall(subst : lambda_subst V L)(s : sequent V L)(n : nat),
      rank_subst (S n) subst ->
        exists(rsubst inj_subst : lambda_subst V L),
          renaming rsubst /\
          injective inj_subst /\
          rank_subst (S n) inj_subst /\
          subst_eq_on_vars subst (subst_compose inj_subst rsubst)
                           (prop_var_sequent s).
  Proof.
    intros subst s n H.
    exists (fst (factor_subst subst s)), (snd (factor_subst subst s)).
    unfold factor_subst in *.
    destruct (divide_subst subst (prop_var_sequent s)) as [smap rmap] eqn:H0.
    simpl.
    apply divide_subst_prop with (n := n) in H0; trivial.
    decompose [and] H0; clear H0.
    repeat split.
          apply renaming_rename_of_map.
        apply injective_subst_of_fixed_map; trivial.
      apply rank_subst_subst_of_fixed_map; trivial.
    apply map_subst_correct_fix_var_chain_ends; trivial.
  Qed.

End Factor_subst.

Implicit Arguments divide_subst [V L].
Implicit Arguments subst_of_map [V L].
Implicit Arguments rename_of_map [V L].
Implicit Arguments fix_var_chain_ends [V L].
Implicit Arguments map_subst_correct [V L].
Implicit Arguments map_rank [V L].
Implicit Arguments divide_subst_prop [V L].
Implicit Arguments factor_subst_property [V L].
