(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: misc.v,v 1.41 2013/04/10 11:17:15 tews Exp $
 *)


(* 
 * ascii symbols 
 *  ! " # $ % & ' * + , - . / : ; < = > ?  \  ^ _ ` | ~  () {} []
 * terminate the comment " to make coqc happy
 *)


(** * Library enhancements *)

(** ** Misc stuff

      This module does the importing of the Coq library and collects
      many results that don't really fit anywhere else. Some parts
      should probably put into separate files, such as the
      [counted_list]. There are also a lot of lemmas where I try to
      prove something just for better understanding of the
      intuitionistic logic of Coq.

*)

Require Export Coq.Bool.Bool.
Require Export Arith.
Require Omega.
Require Export Coq.Lists.List. 
Export ListNotations.

Definition non_empty_type(A : Type) : Prop := exists(a : A), True.


Definition injective{X Y : Type}(f : X -> Y) : Prop :=
  forall(x y : X), f x = f y -> x = y.

Definition enumerator(X : Type) : Type := {f : nat -> X | injective f}.

Definition enum_elem{X : Type}(f : enumerator X) : X := proj1_sig f 0.

Definition countably_infinite(X : Type) : Prop :=
  exists(f : enumerator X), True.

Definition enum_infinite{X : Type}(f : enumerator X) : countably_infinite X :=
  ex_intro _ f I.

Lemma countably_infinite_non_empty :
  forall(X : Type)(P : Prop),
    (forall(non_empty : X), P) ->
    countably_infinite X ->
      P.
Proof.
  intros X P H H0.
  destruct H0.
  clear H0.
  destruct x as [f H0].
  assert (x := f 0).
  apply H.
  exact x.
Qed.


  (***************************************************************************)
  (** *** Recognizers for the library *)
  (***************************************************************************)

Definition is_some{A : Type}(opt : option A) : Prop :=
  match opt with
    | Some _ => True
    | None => False
  end.

Definition is_none{A : Type}(opt : option A) : Prop :=
  match opt with
    | Some _ => False
    | None => True
  end.

Lemma neg_is_none : forall(A : Type)(o : option A),
  ~ (is_some o) -> is_none o.
Proof.
  intros A o H.
  destruct o.
    simpl in *.
    apply H.
    trivial.
  simpl.
  trivial.
Qed.

Lemma option_contradiction : forall(A : Type)(o : option A),
  is_some o -> is_none o -> False.
Proof.
  intros A o H H0.
  destruct o.
    contradiction.
  contradiction.
Qed.


Definition is_inl{A B : Type}(ab : A + B) : Prop :=
  match ab with
    | inl _ => True
    | inr _ => False
  end.

Definition is_inr{A B : Type}(ab : A + B) : Prop :=
  match ab with
    | inl _ => False
    | inr _ => True
  end.


Definition swap_pair{A B : Type}(ab : A * B) : B * A := (snd ab, fst ab).


Lemma split_nat_case_lt : forall(n m : nat)(P : Prop),
  (n < m -> P) -> (n >= m -> P) -> P.
Proof.
  intros n m P H H0.
  assert (H1 := le_or_lt m n).
  tauto.
Qed.

Lemma split_nat_case_le : forall(n m : nat)(P : Prop),
  (n <= m -> P) -> (n > m -> P) -> P.
Proof.
  intros n m P H H0.
  assert (H1 := le_or_lt n m).
  tauto.
Qed.


Inductive dep_and (A : Type) (B : A -> Type) : Type :=
  dep_conj : forall(a : A), B a -> dep_and A B.

Notation " p # P /#\ Q " := (dep_and P (fun(p : P) => Q))
    (at level 80, right associativity) : type_scope.

(* 
 * Definition dep_and_proj{A : Type}{B : A -> Type}(pair : dep_and A B) : A :=
 *   match pair with
 *     | dep_conj x _ => x
 *   end.
 *)


Lemma iff_right : forall{P Q : Prop}, (P <-> Q) -> P -> Q.
Proof.
  intros P Q H H0.
  tauto.
Qed.

Lemma iff_left : forall{P Q : Prop}, (P <-> Q) -> Q -> P.
Proof.
  intros P Q H H0.
  tauto.
Qed.

Definition eq_type(A : Type) : Type := forall(x y : A), {x = y}+{x <> y}.

Lemma eq_equal : forall(A B : Type)(f : eq_type A)(a : A)(b1 b2 : B),
  (if f a a then b1 else b2) = b1.
Proof.
  intros A B f a b1 b2.
  destruct (f a a).
    trivial.
  exfalso.
  apply n.
  trivial.
Qed.

Lemma eq_unequal : forall(A B : Type)(f : eq_type A)(a1 a2 : A)(b1 b2 : B),
  a1 <> a2 ->
    (if f a1 a2 then b1 else b2) = b2.
Proof.
  intros A B f a1 a2 b1 b2 H.
  destruct (f a1 a2).
    contradiction.
  trivial.
Qed.


  (***************************************************************************)
  (** *** function update *)
  (***************************************************************************)

Definition function_update{A B : Type}(a_eq : eq_type A)
                          (f : A -> B)(a0 : A)(b0 : B) : A -> B :=
   fun(a : A) =>
     if a_eq a a0 then b0
     else f a.

Lemma function_update_eq :
  forall{A B : Type}(a_eq : eq_type A)(f : A -> B)(a : A)(b : B),
    function_update a_eq f a b a = b.
Proof.
  intros A B a_eq f a b.
  unfold function_update in *.
  rewrite eq_equal.
  trivial.
Qed.

Lemma function_update_unequal :
  forall{A B : Type}(a_eq : eq_type A)(f : A -> B)(a1 a2 : A)(b : B),
    a1 <> a2 ->
      function_update a_eq f a1 b a2 = f a2.
Proof.
  intros A B a_eq f a1 a2 b H.
  unfold function_update in *.
  rewrite eq_unequal.
    trivial.
  intros H0.
  apply eq_sym in H0.
  contradiction.
Qed.

Lemma function_update_twice :
  forall{A B : Type}(a_eq : eq_type A)(f : A -> B)(a0 : A)(b0 : B),
    function_update a_eq (function_update a_eq f a0 b0) a0 b0 a0 =
    function_update a_eq f a0 b0 a0.
Proof.
  intros A B a_eq f a0 b0.
  unfold function_update in *.
  destruct (a_eq a0 a0).
    trivial.
  trivial.
Qed.

Lemma function_update_split :
  forall{A B : Type}(P : B -> Prop)
        (a_eq : eq_type A)(f : A -> B)(a a' : A)(b : B),
    (a = a' -> P b) ->
    (a <> a' -> P (f a')) ->
      P (function_update a_eq f a b a').
Proof.
  unfold function_update in *.
  intros A B P a_eq f a a' b H H0.
  destruct (a_eq a' a).
    apply H.
    apply eq_sym.
    trivial.
  apply H0.
  intros H1.
  apply eq_sym in H1.
  contradiction.
Qed.

Lemma function_update_split_result :
  forall{A B : Type}(a_eq : eq_type A)(f : A -> B)(a1 a2 : A)(b1 b2 : B),
    b1 = b2 ->
    f a2 = b2 ->
      function_update a_eq f a1 b1 a2 = b2.
Proof.
  intros A B a_eq f a1 a2 b1 b2 H H0.
  unfold function_update in *.
  destruct (a_eq a2 a1).
    trivial.
  trivial.
Qed.


Definition fcompose{X Y Z : Type}(f : Y -> Z)(g : X -> Y)(x : X) : Z :=
  f(g(x)).

(* 
 * Hint Unfold fcompose.
 * 
 * Notation " f ∘ g " := (compose f g)
 *   (at level 40, left associativity) : feq_scope.
 *)


Lemma ge_refl : forall(n : nat), n >= n.
Proof.
  intros n.
  unfold ">=" in *.
  apply le_refl.
Qed.

Lemma add_minus_diag : forall(a b : nat), a + b - b = a.
Proof.
  intros a b.
  omega.
Qed.

Lemma plus_minus_assoc : forall(a b c : nat),
  b >= c -> a + b - c = a + (b - c).
  intros a b c H.
  omega.
Qed.


  (***************************************************************************)
  (** *** Intuitionistic exercises *)
  (***************************************************************************)

Lemma bcontrapositive : forall{P Q : bool}, 
  (P = true -> Q = true) ->
    (negb Q = true -> negb P = true).
Proof.
  intros P Q H H0.
  destruct P.
    rewrite H in H0.
      discriminate.
    trivial.
  trivial.
Qed.


Lemma contrapositive : forall{P Q : Prop}, (P -> Q) -> (~Q -> ~P).   
Proof.
  intros P Q H H0 H1.
  apply H0.
  apply H.
  trivial.
Qed.

Lemma neg_implies : forall(P Q : Prop), ~(P -> Q) -> (~P -> ~Q).
Proof.
  intros P Q H H0 H1.
  apply H.
  intros H2.
  trivial.
Qed.

Lemma neg_neg_implies : forall(P Q : Prop), ~~(P -> Q) -> (~~P -> ~~Q).
Proof.
  intros P Q H H0 H1.
  apply H; clear H; intros H.
  apply H0; clear H0; intros H0.
  apply H1.
  apply H.
  trivial.
Qed.


Lemma iff_neg : forall(P Q : Prop), (P <-> Q) -> (~P <-> ~Q).
Proof.
  intros P Q.
  tauto.
Qed.

Lemma iff_and : forall(P1 P2 Q1 Q2 : Prop),
  (P1 <-> Q1) -> (P2 <-> Q2) -> ((P1 /\ P2) <-> (Q1 /\ Q2)).
Proof.
  intros P1 P2 Q1 Q2 H H0.
  tauto.
Qed.


(* not needed *)
Lemma neg_double_neg : forall{P : Prop}, P -> ~ ~ P.
Proof.
  intros P H H0.
  contradiction.
Qed.


(* not needed *)
Lemma exists_not_not_implies : forall{A : Type}{P : A -> Prop},
  (exists a : A, ~ P a) -> (~ forall(a : A), P a).
Proof.
  intros A P H H0.
  destruct H.
  apply H.
  apply H0.
Qed.


(* not needed *)
Lemma forall_not_iff_not_exists : forall{A : Type}{P : A -> Prop},
  (forall(a : A), ~ P a) <-> (~ exists a : A, P a).
Proof.
  intros A P.
  split.
    intros H H0.
    destruct H0.
    eapply H.
    eexact H0.
  intros H a H0.
  apply H.
  exists a.
  trivial.
Qed.


Lemma or_neg_neg_and : forall(P Q : Prop),
  ((~ P) \/ (~ Q)) -> ~ (P /\ Q).
Proof.
  intros P Q H H0.
  destruct H0.
  destruct H.
    apply H.
    trivial.
  apply H.
  trivial.
Qed.

(* 
 * Lemma not valid : forall(P Q : Prop),
 *    ~ (P /\ Q) -> ((~ P) \/ (~ Q)).
 *)

(* proved in classic.v
 * Lemma neg_or_and_neg : forall(P Q : Prop),
 *    ~ (P \/ Q) <-> ((~ P) /\ (~ Q)).
 *)

Lemma double_neg_impl_neg_or :
  forall(P Q : Prop), (~~(P -> Q)) <-> (~~(~P \/ Q)).
Proof.
  intros P Q.
  split.
    intros H H0.
    apply H; clear H; intros H.
    apply H0.
    left.
    intro H1.
    apply H in H1.
    apply H0.
    right.
    trivial.
  intros H H0.
  apply H; clear H; intros H.
  apply H0.
  intros H1.
  destruct H.
    contradiction.
  trivial.
Qed.


(* The following is the Goedel-Gentzen translation (or double negation 
 * translation) of classical disjunction. Using it to interpret sequents
 * (instead of intuitionistic disjunction) ensures that the rules in the
 * paper are upward correct, which is needed for completeness.
 *)
Definition dneg_or(P Q : Prop) : Prop := ~ (~ P /\ ~ Q).

Lemma dneg_or_intro : forall(P Q : Prop), P -> dneg_or P Q.
Proof.
  intros P Q H H0.
  destruct H0.
  contradiction.
Qed.

Lemma dneg_or_elim : forall(G P Q : Prop),
  (P -> G) -> (Q -> G) -> dneg_or P Q -> ~ ~ G.
Proof.
  intros G P Q H H0 H1 H2.
  apply H1.
  split.
    intros H3.
    apply H2.
    apply H.
    trivial.
  intros H3.
  apply H2.
  apply H0.
  trivial.
Qed.

Lemma single_prop_ax_correct : forall(P : Prop),
  dneg_or P (~ P).
Proof.
  intros P.
  intros H.
  destruct H.
  contradiction.
Qed.

Lemma prop_ax_down_correct : forall(G P : Prop),
  dneg_or P (dneg_or (~ P) G).
Proof.
  intros G P.
  unfold dneg_or in *.
  intros H.
  destruct H.
  apply H0.
  intros H1.
  destruct H1.
  apply H1.
  trivial.
Qed.

Lemma prop_ax_up_correct : forall(G P : Prop),
  dneg_or P (dneg_or (~ P) G) -> True.
Proof.
  trivial.
Qed.

Lemma prop_and_down_correct : forall(G P Q : Prop),
  dneg_or P G -> dneg_or Q G -> dneg_or (P /\ Q) G.
Proof.
  intros G P Q H H0.
  unfold dneg_or in *.
  intros H1.
  destruct H1.
  apply H.
  split.
    intros H3.
    apply H0.
    split.
      intros H4.
      apply H1.
      split.
        trivial.
      trivial.
    trivial.
  trivial.
Qed.

Lemma prop_and_up_correct : forall(G P Q : Prop),
  dneg_or (P /\ Q) G -> (dneg_or P G) /\ dneg_or Q G.
Proof.
  intros G P Q H.
  unfold dneg_or in *.
  split.
    intros H0.
    destruct H0.
    apply H.
    split.
      intros H2.
      destruct H2.
      apply H0.
      trivial.
    trivial.
  intros H0.
  destruct H0.
  apply H.
  split.
    intros H2.
    destruct H2.
    apply H0.
    trivial.
  trivial.
Qed.

Lemma prop_or_down_correct :  forall(G P Q : Prop),
  dneg_or (~ P) (dneg_or (~ Q) G) -> dneg_or (~ (P /\ Q)) G.
Proof.
  intros G P Q H.
  unfold dneg_or in *.
  intros H0.
  destruct H0.
  apply H0.
  intros H2.
  destruct H2.
  apply H.
  split.
    apply neg_double_neg.
    trivial.
  apply neg_double_neg.
  split.
    apply neg_double_neg.
    trivial.
  trivial.
Qed.

Lemma prop_or_up_correct_context :  forall(G P Q : Prop),
   dneg_or (~ (P /\ Q)) G -> dneg_or (~ P) (dneg_or (~ Q) G).
Proof.
  intros G P Q H.
  unfold dneg_or in *.
  intros H0.
  destruct H0.
  apply H0.
  intros H2.
  apply H1.
  intros H3.
  destruct H3.
  apply H3.
  intros H5.
  apply H.
  split.
    apply neg_double_neg.
    split.
      trivial.
    trivial.
  trivial.
Qed.

Lemma prop_or_up_correct :  forall(P Q : Prop),
   (~ (P /\ Q)) -> dneg_or (~ P) (~ Q).
Proof.
  unfold dneg_or in *.
  intros P Q H H0.
  destruct H0.
  apply H0; clear H0; intros H0.
  apply H1; clear H1; intros H1.
  apply H; clear H.
  split; trivial.
Qed.

Lemma prop_neg_down_correct : forall(G P : Prop),
  dneg_or P G -> dneg_or (~ ~ P) G.
Proof.
  intros G P H.
  unfold dneg_or in *.
  intros H0.
  destruct H0.
  apply H0.
  intros H2.
  apply H.
  split.
    trivial.
  trivial.
Qed.

Lemma prop_neg_up_correct : forall(G P : Prop),
  dneg_or (~ ~ P) G -> dneg_or P G.
Proof.
  intros G P H.
  unfold dneg_or in *.
  intros H0.
  destruct H0.
  apply H.
  split.
    apply neg_double_neg.
    trivial.
  trivial.
Qed.


(***************************************************************************)
(** ***  max  *)
(***************************************************************************)

Lemma max_mono_both : forall(n1 n2 n3 n4 : nat),
  n1 <= n3 -> n2 <= n4 -> max n1 n2 <= max n3 n4.
  intros n1 n2 n3 n4 H H0.
Proof.
  apply Max.max_lub.
    assert (H1 := Max.le_max_l n3 n4).
    omega.
  assert (H1 := Max.le_max_r n3 n4).
  omega.
Qed.


(***************************************************************************)
(** ***  counted lists  *)
(***************************************************************************)

Inductive counted_list(A : Type) : nat -> Type :=
  | counted_nil : counted_list A 0
  | counted_cons : forall(n : nat), 
      A -> counted_list A n -> counted_list A (S n).

Implicit Arguments counted_nil [[A]].
Implicit Arguments counted_cons [A n].


Definition counted_head{A : Type}{n : nat}(l : counted_list A (S n)) : A :=
  match l in (counted_list _ n')
          return (match n' return Type with
                    | 0 => nat
                    | S _ => A
                  end)
  with
    | counted_nil => 0
    | counted_cons _ a _ => a
  end.

Fixpoint counted_map{A B : Type}{n : nat}(f : A -> B)(al : counted_list A n) :
                                                           counted_list B n :=
  match al with
    | counted_nil => counted_nil
    | counted_cons n a al => counted_cons (f a) (counted_map f al)
  end. 


Fixpoint list_of_counted_list{A : Type}{n : nat}(l : counted_list A n) : 
                                                                      list A :=
  match l with
    | counted_nil => []
    | counted_cons n a l => a :: (list_of_counted_list l)
  end.

Lemma list_of_counted_list_map :
  forall(A B : Type)(n : nat)(f : A -> B)(al : counted_list A n),
    list_of_counted_list (counted_map f al) = map f (list_of_counted_list al).
Proof.
  induction al.
    trivial.
  simpl.
  rewrite IHal.
  trivial.
Qed.


Lemma length_list_of_counted_list : 
  forall(A : Type)(n : nat)(l : counted_list A n),
    length (list_of_counted_list l) = n.
Proof.
  induction l.
    trivial.
  simpl.
  rewrite IHl.
  trivial.
Qed.

Lemma less_length_counted_list :
  forall{A : Type}(i n : nat)(l : counted_list A n),
    i < n -> i < length (list_of_counted_list l).
Proof.
  intros A i n l H.
  rewrite length_list_of_counted_list.
  trivial.
Qed.


Lemma counted_list_eta : forall{A : Type}{n : nat}(l : counted_list A n),
  match n return counted_list A n -> Prop with
    | 0 => fun(l : counted_list A 0) => l = counted_nil
    | S n => fun(l : counted_list A (S n)) => 
      exists(a : A)(tl : counted_list A n), l = counted_cons a tl
  end l.
Proof.
  intros A n l.
  destruct l.
    trivial.
  exists a, l.
  trivial.
Qed.

Lemma counted_list_eta_1 : forall{A : Type}(l : counted_list A 1),
  exists(a : A), l = counted_cons a counted_nil.
Proof.
  intros A l.
  assert (H := counted_list_eta l).
  simpl in *.
  decompose [ex] H; clear H.
  rename x into a, x0 into tail.
  assert (H := counted_list_eta tail).
  simpl in *.
  subst tail.
  exists a.
  trivial.
Qed.

Lemma counted_0_destruction : 
  forall(A : Type)(P : counted_list A 0 -> Prop)(v : counted_list A 0),
    P counted_nil ->
      P v.
Proof.
  intros A P v H.
  (* (match v with counted_nil => H end) does also work *)
  exact (match v 
           as v0
           in (counted_list _ n)
           return (match n return (counted_list A n -> Prop) 
                   with
                     | 0 => fun v0 : counted_list A 0 => P v0
                     | S n0 => fun _ : counted_list A (S n0) => True
                   end v0)
          with
            | counted_nil => H
            | counted_cons _ _ _ => I
          end).
Qed.

Lemma counted_list_equal :
  forall{A : Type}{n : nat}(l1 l2 : counted_list A n),
    list_of_counted_list l1 = list_of_counted_list l2 -> 
      l1 = l2.
Proof.
  induction n.
    intros l1 l2 H.
    assert (H0 := counted_list_eta l1).
    assert (H1 := counted_list_eta l2).
    simpl in *.
    subst l1 l2.
    trivial.
  intros l1 l2 H.
  assert (H0 := counted_list_eta l1).
  assert (H1 := counted_list_eta l2).
  simpl in *.
  decompose [ex] H0; clear H0.
  decompose [ex] H1; clear H1.
  subst l1 l2.
  simpl in H.
  inversion H; clear H.
  subst x1.
  rewrite IHn with (1 := H2).
  trivial.
Qed.

