(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: prop_mod.v,v 1.11 2013/04/10 11:17:16 tews Exp $
 *)


(** * Completeness and semantic cut elimination, Section 4 *)

(** ** Non-decomposable formulas

      In the proof of 4.13, there appears a set of formulas that
      cannot be decomposed by the propositional rules. These formulas
      are either possibly negated propsitional variables or possibly
      negated modal formulas. Note the similarity with
      [plain_prop_mod]. 
*)

Require Export dsets build_prop_proof.

Section Prop_mod.

  Variable V : Type.
  Variable L : modal_operators.

  (** Need a decidable equality on propositional constants for 
      the properties on the proof built oracle.
   *)
  Variable v_eq : eq_type V.


  (***************************************************************************)
  (** ***  non-decomposable formulas  *)
  (***************************************************************************)

  (** See also neg_form for the Prop version. *)
  Definition dneg_form_maybe(F : dset (lambda_formula V L)) 
                                                : dset (lambda_formula V L) :=
    fun(f : lambda_formula V L) =>
      match f with
        | lf_neg f => F f
        | _ => F f
      end.

  Definition dprop_or_mod_formula : dset (lambda_formula V L) :=
    dunion (dneg_form_maybe is_prop) (dneg_form_maybe is_modal).

  Definition dprop_or_mod_sequent : dset (sequent V L) := 
    forallb dprop_or_mod_formula.


  (***************************************************************************)
  (** *** As Prop sets  *)
  (***************************************************************************)

  Definition prop_or_mod_formula : set (lambda_formula V L) :=
    union (neg_form_maybe prop_form) (@top_modal_form V L).

  Lemma simple_prop_from_prop : forall(f : lambda_formula V L),
    neg_form_maybe prop_form f <-> dneg_form_maybe is_prop f = true.
  Proof.
    clear. 
    split.
      intros H.
      destruct f; simpl in *; try contradiction; trivial.
      destruct f; simpl in *; try contradiction; trivial.
    intros H.
    destruct f; simpl in *; try discriminate; trivial.
    destruct f; simpl in *; try discriminate; trivial.
  Qed.

  Lemma simple_modal_from_prop : forall(f : lambda_formula V L),
    top_modal_form f <-> dneg_form_maybe is_modal f = true.
  Proof.
    clear. 
    split.
      intros H.
      destruct f; simpl in *; try contradiction; trivial.
      destruct f; simpl in *; try contradiction; trivial.
    intros H.
    destruct f; simpl in *; try discriminate; trivial.
    destruct f; simpl in *; try discriminate; trivial.
  Qed.

  Lemma prop_or_mod_formula_prop : forall(f : lambda_formula V L),
    prop_or_mod_formula f <-> dprop_or_mod_formula f = true.
  Proof.
    clear. 
    unfold prop_or_mod_formula, dprop_or_mod_formula in *.
    unfold union, dunion in *.
    intros f.
    rewrite orb_true_iff.
    rewrite <- simple_prop_from_prop.
    rewrite <- simple_modal_from_prop.
    apply iff_refl.
  Qed.


  Definition prop_or_mod_sequent : set (sequent V L) :=
    every_nth prop_or_mod_formula.

  Lemma prop_or_mod_sequent_empty : prop_or_mod_sequent [].  
  Proof.
    apply every_nth_empty.
  Qed.

  Lemma prop_or_mod_sequent_cons : 
    forall(f : lambda_formula V L)(s : sequent V L),
      prop_or_mod_sequent (f :: s) <-> 
        prop_or_mod_formula f /\ prop_or_mod_sequent s.
  Proof.
    intros f s.
    split.
      intros H.
      split.
        eapply every_nth_head.
        eexact H.
      eapply every_nth_tail.
      eexact H.
    intros H.
    destruct H.
    apply every_nth_cons.
      trivial.
    trivial.
  Qed.

  Lemma prop_or_mod_sequent_list_reorder : forall(s1 s2 : sequent V L),
    list_reorder s1 s2 ->
    prop_or_mod_sequent s1 ->
      prop_or_mod_sequent s2.
  Proof.
    intros s1 s2 H H0.
    unfold prop_or_mod_sequent in *.
    erewrite <- every_nth_list_reorder.
      eexact H0.
    trivial.
  Qed.

  Lemma prop_or_mod_sequent_append_left : forall(s1 s2 : sequent V L),
    prop_or_mod_sequent (s1 ++ s2) -> prop_or_mod_sequent s1.
  Proof.
    intros s1 s2 H.
    eapply every_nth_append_left.
    eexact H.
  Qed.

  Lemma prop_or_mod_sequent_append_right : forall(s1 s2 : sequent V L),
    prop_or_mod_sequent (s1 ++ s2) -> prop_or_mod_sequent s2.
  Proof.
    intros s1 s2 H.
    eapply every_nth_append_right.
    eexact H.
  Qed.

  Lemma prop_or_mod_sequent_prop : forall(s : sequent V L),
    prop_or_mod_sequent s <-> dprop_or_mod_sequent s = true.
  Proof.
    clear. 
    induction s.
      simpl.
      split.
        trivial.
      intros H.
      apply prop_or_mod_sequent_empty.
    unfold dprop_or_mod_sequent in *.
    rewrite prop_or_mod_sequent_cons.
    rewrite forallb_cons.
    rewrite prop_or_mod_formula_prop.
    tauto.
  Qed.


  (***************************************************************************)
  (** *** Prop oracle properties  *)
  (***************************************************************************)

  Lemma non_decomposable_is_prop_mod : forall(s : sequent V L),
    prop_G_oracle v_eq s = None -> prop_or_mod_sequent s.
  Proof.
    intros s H n n_less.
    apply prop_G_oracle_None in H.
    decompose [and] H; clear H.
    unfold prop_or_mod_formula, union in *.
    destruct (nth s n n_less) eqn:?.
          left.
          simpl.
          trivial.
        destruct l.
              left.
              simpl.
              trivial.
            exfalso.
            revert Heql.
            apply find_neg_neg_none.
            trivial.
          exfalso.
          revert Heql.
          apply find_neg_and_none.
          trivial.
        right.
        simpl.
        trivial.
      exfalso.
      revert Heql.
      apply find_and_none.
      trivial.
    right.
    simpl.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Partition prop-mod sequents  *)
  (***************************************************************************)

  Lemma prop_or_mod_partition : forall(s : sequent V L),
    prop_or_mod_sequent s ->
      exists(prop_s mod_s : sequent V L),
        prop_sequent prop_s /\
        top_modal_sequent mod_s /\
        list_reorder s (mod_s ++ prop_s).
  Proof.
    clear. 
    intros s H.
    assert (H0 := partition_result (dneg_form_maybe is_prop) s).
    destruct H0.
    exists (fst (partition (dneg_form_maybe is_prop) s)),
           (snd (partition (dneg_form_maybe is_prop) s)).
    assert (list_reorder s
              (snd (partition (dneg_form_maybe is_prop) s) ++
               fst (partition (dneg_form_maybe is_prop) s))).
      eapply list_reorder_trans.
        apply list_reorder_partition.
      apply list_reorder_append_swap.
    repeat split; trivial.
      clear H1 H2.
      rewrite forallb_every_nth in H0.
      intros i i_less.
      rewrite simple_prop_from_prop.
      apply H0.
    clear H0.
    assert (H3 := prop_or_mod_sequent_list_reorder _ _ H2 H).
    apply prop_or_mod_sequent_append_left in H3.
    clear H H2.
    rewrite forallb_every_nth in H1.
    intros i i_less.
    specialize (H1 i i_less).
    specialize (H3 i i_less).
    rewrite negb_true_iff in H1.
    rewrite prop_or_mod_formula_prop in H3.
    unfold dprop_or_mod_formula, dunion in *.
    rewrite H1 in H3.
    simpl in H3.
    rewrite simple_modal_from_prop.
    trivial.
  Qed.

End Prop_mod.

Implicit Arguments dprop_or_mod_sequent [V L].
Implicit Arguments prop_or_mod_sequent_prop [V L].
Implicit Arguments prop_or_mod_partition [V L].
