(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: rule_sets.v,v 1.41 2013/04/10 11:17:16 tews Exp $
 *)

(** ** Rule sets

      This module defines the various rule sets from section 3.
*)

Require Export propositional_formulas modal_formulas rules.

Section Rule_sets.

  Variable V : Type.
  Variable L : modal_operators.


  (**************************************************************************)
  (** ***  G rule set  *)
  (**************************************************************************)

  Definition G_struct_set(r : sequent_rule V L) : Prop :=
    is_and_rule r \/ is_neg_and_rule r \/ is_neg_neg_rule r.

  Definition G_set : set (sequent_rule V L) :=
    union is_ax_rule 
     (union is_and_rule
       (union is_neg_and_rule is_neg_neg_rule)).

  Lemma G_set_struct_union : 
    set_equal G_set (union is_ax_rule G_struct_set).
  Proof.
    clear. 
    unfold G_set, G_struct_set, union in *.
    intros r.
    apply iff_refl.
  Qed.


  (**************************************************************************)
  (** **** G rule set properties  *)
  (**************************************************************************)

  Lemma subst_closed_G_struct_set : subst_closed_rule_set G_struct_set.
  Proof.
    clear. 
    unfold subst_closed_rule_set in *.
    intros sigma r H.
    unfold G_struct_set in *.
    decompose [or] H; clear H.
        left.
        apply subst_closed_and.
        trivial.
      right.
      left.
      apply subst_closed_neg_and.
      trivial.
    right.
    right.
    apply subst_closed_neg_neg.
    trivial.
  Qed.

  Lemma G_struct_multiset : rule_multiset G_struct_set.
  Proof.
    unfold rule_multiset, G_struct_set in *.
    intros or s H H0.
    decompose [or] H; clear H.
        assert (H2 := and_rule_context _ H1).
        clear H1.
        decompose [ex] H2; clear H2.
        subst or.
        lapply (reorder_rule_add_context _ s x x0 H0).
          intros H.
          decompose [ex] H; clear H.
          eexists.
          split.
            eexact H2.
          left.
          apply context_and_rule.
        simpl.
        trivial.
      assert (H3 := neg_and_rule_context _ H2).
      clear H2.
      decompose [ex] H3; clear H3.
      subst or.
      lapply (reorder_rule_add_context _ s x x0 H0).
        intros H.
        decompose [ex] H; clear H.
        eexists.
        split.
          eexact H2.
        right.
        left.
        apply context_neg_and_rule.
      simpl.
      trivial.
    assert (H3 := neg_neg_rule_context _ H2).
    clear H2.
    decompose [ex] H3; clear H3.
    subst or.
    lapply (reorder_rule_add_context _ s x x0 H0).
      intros H.
      decompose [ex] H; clear H.
      eexists.
      split.
        eexact H2.
      right.
      right.
      apply context_neg_neg_rule.
    simpl.
    trivial.
  Qed.

  Lemma G_multiset : rule_multiset G_set.
  Proof.
    eapply set_equal_multiset.
      apply set_equal_symm.
      apply G_set_struct_union.
    apply multiset_union.
      unfold rule_multiset.
      intros or s H H0.
      exists {| assumptions := []; conclusion := s |}.
      split.
        unfold is_ax_rule in *.
        destruct H.
        unfold reordered_rule in *.
        repeat split.
          simpl.
          rewrite H.
          trivial.
        intros n n_less_rs n_less_rr.
        exfalso.
        rewrite H in *.
        simpl in n_less_rs.
        omega.
      unfold is_ax_rule in *.
      simpl.
      split.
        trivial.
      destruct H.
      eapply simple_tautology_reorder.
        eexact H1.
      trivial.
    apply G_struct_multiset.
  Qed.

  Lemma const_rank_G_set : 
    forall(n : nat)(r : sequent_rule V L),
      G_set r ->
      rank_sequent n (conclusion r) ->
        every_nth (rank_sequent n) (assumptions r).
  Proof.
    intros n r H H0.
    unfold G_set, union in *.
    decompose [or] H; clear H.
          unfold is_ax_rule in *.
          destruct H1.
          rewrite H.
          apply every_nth_empty.
        assert (H3 := and_rule_context _ H2).
        decompose [ex] H3; clear H3.
        eapply const_rank_add_context; eauto.
        apply const_rank_and_rule.
      assert (H2 := neg_and_rule_context _ H1).
      decompose [ex] H2; clear H2.
      eapply const_rank_add_context; eauto.
      apply const_rank_neg_and_rule.
    assert (H2 := neg_neg_rule_context _ H1).
    decompose [ex] H2; clear H2.
    eapply const_rank_add_context; eauto.
    apply const_rank_neg_neg_rule.
  Qed.


  (**************************************************************************)
  (** **** conclusions in G are not purely modal *)
  (**************************************************************************)

  Lemma ax_rule_no_simple_modal_conclusion :
    forall(r : sequent_rule V L),
      is_ax_rule r ->
      ~ simple_modal_sequent (conclusion r).
  Proof.
    intros r H H0.
    unfold is_ax_rule, simple_tautology, simple_tautology_witness in *.
    decompose [ex and or dep_and] H; clear H.
    specialize (H0 x a).
    rewrite H2 in *.
    contradiction.
  Qed.

  Lemma and_rule_no_simple_modal_conclusion :
    forall(r : sequent_rule V L),
      is_and_rule r ->
      ~ simple_modal_sequent (conclusion r).
  Proof.
    intros r H H0.
    unfold is_and_rule in *.
    decompose [ex and or dep_and] H; clear H.
    rewrite H3 in *.
    clear - H0.
    assert (H1 := simple_modal_sequent_append_right _ _ H0).
    assert (H2 := simple_modal_sequent_head _ _ H1).
    contradiction.
  Qed.

  Lemma neg_and_rule_no_simple_modal_conclusion :
    forall(r : sequent_rule V L),
      is_neg_and_rule r ->
      ~ simple_modal_sequent (conclusion r).
  Proof.
    intros r H H0.
    unfold is_neg_and_rule in *.
    decompose [ex and or dep_and] H; clear H.
    rewrite H3 in *.
    clear - H0.
    assert (H1 := simple_modal_sequent_append_right _ _ H0).
    assert (H2 := simple_modal_sequent_head _ _ H1).
    contradiction.
  Qed.

  Lemma neg_neg_rule_no_simple_modal_conclusion :
    forall(r : sequent_rule V L),
      is_neg_neg_rule r ->
      ~ simple_modal_sequent (conclusion r).
  Proof.
    intros r H H0.
    unfold is_neg_neg_rule in *.
    decompose [ex and or dep_and] H; clear H.
    rewrite H3 in *.
    clear - H0.
    assert (H1 := simple_modal_sequent_append_right _ _ H0).
    assert (H2 := simple_modal_sequent_head _ _ H1).
    contradiction.
  Qed.

  Lemma G_rules_no_simple_modal_conclusion :
    forall(r : sequent_rule V L),
      G_set r ->
      ~ simple_modal_sequent (conclusion r).
  Proof.
    intros r H.
    unfold G_set, union in *.
    decompose [or] H; clear H.
          apply ax_rule_no_simple_modal_conclusion; trivial.
        apply and_rule_no_simple_modal_conclusion; trivial.
      apply neg_and_rule_no_simple_modal_conclusion; trivial.
    apply neg_neg_rule_no_simple_modal_conclusion; trivial.
  Qed.

  Lemma G_set_no_empty_conclusion : forall(r : sequent_rule V L),
    G_set r -> conclusion r <> [].
  Proof.
    intros r H.
    unfold G_set, union in *.
    decompose [or] H; clear H.
          apply ax_rule_no_empty_conclusion; trivial.
        apply and_rule_no_empty_conclusion; trivial.
      apply neg_and_rule_no_empty_conclusion; trivial.
    apply neg_neg_rule_no_empty_conclusion; trivial.
  Qed.


  (**************************************************************************)
  (** *** G_n rule set *)
  (**************************************************************************)

  Definition G_n_set(n : nat) : set (sequent_rule V L) :=
    rank_rules n G_set.

  Lemma G_n_multiset : forall(n : nat),
    rule_multiset (G_n_set n).
  Proof.
    clear. 
    intros n.
    apply multiset_rank_rules.
    apply G_multiset.
  Qed.

  Lemma provable_depth_G_n_hyp_list_reorder : 
    forall(hyp : set (sequent V L))(n d : nat)(s1 s2 : sequent V L),
      sequent_multiset hyp ->
      list_reorder s1 s2 ->
      provable_at_depth (G_n_set n) hyp d s1 ->
        provable_at_depth (G_n_set n) hyp d s2.
  Proof.
    intros hyp n d s1 s2 H H0 H1.
    eapply multiset_depth_provability.
          apply G_n_multiset.
        trivial.
      eexact H0.
    trivial.
  Qed.

  Lemma provable_G_n_hyp_list_reorder : 
    forall(hyp : set (sequent V L))(n : nat)(s1 s2 : sequent V L),
      sequent_multiset hyp ->
      list_reorder s1 s2 ->
      provable (G_n_set n) hyp s1 ->
        provable (G_n_set n) hyp s2.
  Proof.
    intros hyp n s1 s2 H H0 H1.
    eapply multiset_provability.
          apply G_n_multiset.
        trivial.
      eexact H0.
    trivial.
  Qed.

  Lemma provable_G_n_list_reorder : 
    forall(n : nat)(s1 s2 : sequent V L),
      list_reorder s1 s2 ->
      provable (G_n_set n) (empty_sequent_set V L) s1 ->
        provable (G_n_set n) (empty_sequent_set V L) s2.
  Proof.
    intros n s1 s2 H H0.
    eapply provable_G_n_hyp_list_reorder.
        apply sequent_multiset_empty.
      eexact H.
    trivial.
  Qed.


  (**************************************************************************)
  (** **** G_n set subset properties *)
  (**************************************************************************)

  Lemma G_n_set_mono : forall(n1 n2 : nat),
    n1 <= n2 -> subset (G_n_set n1) (G_n_set n2).
  Proof.
    intros n1 n2 H.
    eapply rank_rules_subset_rank.
    trivial.
  Qed.

  Lemma ax_n_subset_G_n : forall(n : nat),
    subset (rank_rules n is_ax_rule) (G_n_set n).
  Proof.
    intros n.
    unfold G_n_set, G_set in *.
    apply rank_rules_mono.
    apply subset_union_left.
  Qed.

  Lemma and_n_subset_G_n : forall(n : nat),
    subset (rank_rules n is_and_rule) (G_n_set n).
  Proof.
    intros n.
    unfold G_n_set, G_set in *.
    apply rank_rules_mono.
    eapply subset_trans.
      apply subset_union_left.
    apply subset_union_right.
  Qed.

  Lemma neg_and_n_subset_G_n : forall(n : nat),
    subset (rank_rules n is_neg_and_rule) (G_n_set n).
  Proof.
    intros n.
    unfold G_n_set, G_set in *.
    apply rank_rules_mono.
    eapply subset_trans.
      apply subset_union_left.
    eapply subset_trans.
      apply subset_union_right.
    apply subset_union_right.
  Qed.

  Lemma neg_neg_n_subset_G_n : forall(n : nat),
    subset (rank_rules n is_neg_neg_rule) (G_n_set n).
  Proof.
    intros n.
    unfold G_n_set, G_set in *.
    apply rank_rules_mono.
    eapply subset_trans.
      apply subset_union_right.
    eapply subset_trans.
      apply subset_union_right.
    apply subset_union_right.
  Qed.


  (**************************************************************************)
  (** **** G_n decomposition *)
  (**************************************************************************)

  Lemma decompose_G_n_set_coarsly :
    forall(n : nat)(r : sequent_rule V L),
      G_n_set n r ->
        (assumptions r = [] /\
         simple_tautology (conclusion r) /\
         rank_sequent n (conclusion r)
        ) \/
        (exists(rb : sequent_rule V L)(sl sr : sequent V L)
               (concl_rb : lambda_formula V L),
          every_nth (rank_sequent n) (assumptions r) /\ (* rarely needed *)
          rank_sequent n (conclusion r) /\
          r = rule_add_context sl sr rb /\
          ((exists(f1 f2 : lambda_formula V L), rb = bare_and_rule f1 f2) \/
           (exists(f1 f2 : lambda_formula V L), rb = bare_neg_and_rule f1 f2) \/
           (exists(f : lambda_formula V L), rb = bare_neg_neg_rule f))
          /\ [concl_rb] = conclusion rb).
  Proof.
    intros n r H.
    unfold G_n_set, rank_rules, rule_has_rank, G_set, union in *.
    decompose [ex and or dep_and] H; clear H.
          left.
          unfold is_ax_rule in *.
          tauto.
        right.
        assert (H4 := and_rule_context _ H0).
        decompose [ex] H4; clear H4.
        exists (bare_and_rule x1 x2), x, x0, (lf_and x1 x2).
        repeat split; auto.
        left.
        exists x1, x2.
        trivial.
      right.
      assert (H4 := neg_and_rule_context _ H2).
      decompose [ex] H4; clear H4.
      exists (bare_neg_and_rule x1 x2), x, x0, (lf_neg (lf_and x1 x2)).
      repeat split; auto.
      right.
      left.
      exists x1, x2.
      trivial.
    right.
    assert (H4 := neg_neg_rule_context _ H2).
    decompose [ex] H4; clear H4.
    exists (bare_neg_neg_rule x1), x, x0, (lf_neg (lf_neg x1)).
    repeat split; auto.
    right.
    right.
    exists x1.
    trivial.
  Qed.


  Lemma decompose_G_n_set :
    forall(n : nat)(r : sequent_rule V L),
      G_n_set n r ->
        (assumptions r = [] /\
         simple_tautology (conclusion r) /\
         rank_sequent n (conclusion r)
        ) \/
        (exists(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
          assumptions r = [sl ++ f1 :: sr; sl ++ f2 :: sr] /\
          conclusion r = sl ++ (lf_and f1 f2) :: sr /\
          rank_sequent n (sl ++ f1 :: sr) /\
          rank_sequent n (sl ++ f2 :: sr) /\
          rank_sequent n (sl ++ (lf_and f1 f2) :: sr)
        ) \/
        (exists(sl sr : sequent V L)(f1 f2 : lambda_formula V L),
          assumptions r = [sl ++ (lf_neg f1) :: (lf_neg f2) :: sr] /\
          conclusion r = sl ++ (lf_neg (lf_and f1 f2)) :: sr /\
          rank_sequent n (sl ++ (lf_neg f1) :: (lf_neg f2) :: sr) /\
          rank_sequent n (sl ++ (lf_neg (lf_and f1 f2)) :: sr)
        ) \/
        (exists(sl sr : sequent V L)(f : lambda_formula V L),
          assumptions r = [sl ++ f :: sr] /\
          conclusion r = sl ++ (lf_neg (lf_neg f)) :: sr /\
          rank_sequent n (sl ++ f :: sr) /\
          rank_sequent n (sl ++ (lf_neg (lf_neg f)) :: sr)
        ).
  Proof.
    intros n r H.
    assert (H0 := decompose_G_n_set_coarsly _ _ H).
    clear H.
    decompose [ex and or dep_and] H0; clear H0.
          left.
          tauto.
        clear x2 H5.
        right.
        left.
        subst x r.
        simpl in *.
        unfold add_context in *.
        simpl in *.
        exists x0, x1, x3, x4.
        repeat split.
            eapply every_nth_head.
            eexact H1.
          eapply every_nth_head.
          eapply every_nth_tail.
          eexact H1.
        trivial.
      clear x2 H5.
      right.
      right.
      left.
      subst x r.
      simpl in *.
      unfold add_context in *.
      simpl in *.
      exists x0, x1, x3, x4.
      repeat split.
        eapply every_nth_head.
        eexact H1.
      trivial.
    clear x2 H5.
    right.
    right.
    right.
    subst x r.
    simpl in *.
    unfold add_context in *.
    simpl in *.
    exists x0, x1, x3.
    repeat split.
      eapply every_nth_head.
      eexact H1.
    trivial.
  Qed.


  (**************************************************************************)
  (** **** provability with G_n *)
  (**************************************************************************)

  Lemma provable_with_and :
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))
          (sl sr : sequent V L)(f1 f2 : lambda_formula V L)(n : nat),
      subset (G_n_set n) (rank_rules n rules) ->
      rank_sequent n (sl ++ (lf_and f1 f2) :: sr) ->
      provable (rank_rules n rules) hyp (sl ++ f1 :: sr) ->
      provable (rank_rules n rules) hyp (sl ++ f2 :: sr) ->
        provable (rank_rules n rules) hyp (sl ++ (lf_and f1 f2) :: sr).
  Proof.
    intros rules hyp sl sr f1 f2 n H H0 H1 H2.
    apply provable_with_rule with 
                         (assum := [sl ++ f1 :: sr; sl ++ f2 :: sr]).
      apply H.
      apply and_n_subset_G_n.
      split.
        apply context_and_rule.
      split.
        apply every_nth_cons.
          eapply const_rank_and_rule_left_context.
          eexact H0.
        apply every_nth_cons.
          eapply const_rank_and_rule_right_context.
          eexact H0.
        apply every_nth_empty.
      trivial.
    apply every_nth_cons; trivial.
    apply every_nth_cons; trivial.
    apply every_nth_empty.
  Qed.

  Lemma provable_with_neg_and :
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))
          (sl sr : sequent V L)(f1 f2 : lambda_formula V L)(n : nat),
      subset (G_n_set n) (rank_rules n rules) ->
      rank_sequent n (sl ++ (lf_neg (lf_and f1 f2)) :: sr) ->
      provable (rank_rules n rules) hyp 
               (sl ++ (lf_neg f1) :: (lf_neg f2) :: sr) ->
        provable (rank_rules n rules) hyp 
                 (sl ++ (lf_neg (lf_and f1 f2)) :: sr).
  Proof.
    intros rules hyp sl sr f1 f2 n H H0 H1.
    apply provable_with_rule with 
                         (assum := [sl ++ lf_neg f1 :: lf_neg f2 :: sr]).
      apply H.
      apply neg_and_n_subset_G_n.
      split.
        apply context_neg_and_rule.
      split.
        apply every_nth_cons.
          apply const_rank_neg_and_rule_context.
          trivial.
        apply every_nth_empty.
      trivial.
    apply every_nth_cons.
      trivial.
    apply every_nth_empty.
  Qed.

  Lemma provable_with_neg_neg :
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))
          (sl sr : sequent V L)(f : lambda_formula V L)(n : nat),
      subset (G_n_set n) (rank_rules n rules) ->
      rank_sequent n (sl ++ (lf_neg (lf_neg f)) :: sr) ->
      provable (rank_rules n rules) hyp (sl ++ f :: sr) ->
        provable (rank_rules n rules) hyp (sl ++ (lf_neg (lf_neg f)) :: sr).
  Proof.
    intros rules hyp sl sr f n H H0 H1.
    apply provable_with_rule with (assum := [sl ++ f :: sr]).
      apply H.
      apply neg_neg_n_subset_G_n.
      split.
        apply context_neg_neg_rule.
      split.
        apply every_nth_cons.
          apply const_rank_neg_neg_rule_context.
          trivial.
        apply every_nth_empty.
      trivial.
    apply every_nth_cons.
      trivial.
    apply every_nth_empty.
  Qed.


  (**************************************************************************)
  (** **** other G_n properties *)
  (**************************************************************************)

  Lemma G_n_set_no_simple_modal_conclusion :
    forall(n : nat)(r : sequent_rule V L),
      G_n_set n r ->
      ~ simple_modal_sequent (conclusion r).
  Proof.
    intros n r H.
    apply G_rules_no_simple_modal_conclusion.
    unfold G_n_set, rank_rules in *.
    destruct H.
    trivial.
  Qed.


  Lemma sequent_other_context_G_n_set :
    forall(n : nat)(sl1 sl2 sr1 sr2 : sequent V L)
          (rbase : sequent_rule V L),
      ((exists f1 f2 : lambda_formula V L, rbase = bare_and_rule f1 f2) \/
       (exists f1 f2 : lambda_formula V L, rbase = bare_neg_and_rule f1 f2) \/
       (exists f : lambda_formula V L, rbase = bare_neg_neg_rule f)) ->
      (rank_sequent n sl1 -> rank_sequent n sl2) ->
      (rank_sequent n sr1 -> rank_sequent n sr2) ->
      rank_sequent n (sl1 ++ conclusion rbase ++ sr1) ->
        G_n_set n (rule_add_context sl2 sr2 rbase).
  Proof.
    intros n sl1 sl2 sr1 sr2 rbase H H0 H1 H2.
    assert (G_set (rule_add_context sl2 sr2 rbase)).
      decompose [ex and or dep_and] H; clear H.
          subst rbase.
          right.
          left.
          apply context_and_rule.
        subst rbase.
        right.
        right.
        left.
        apply context_neg_and_rule.
      subst rbase.
      right.
      right.
      right.
      apply context_neg_neg_rule.
    assert (rank_sequent n (conclusion (rule_add_context sl2 sr2 rbase))).
      clear H H3.
      unfold rule_add_context, add_context in *.
      simpl.
      apply rank_sequent_append.
        apply H0.
        apply rank_sequent_append_left in H2.
        trivial.
      apply rank_sequent_append_right in H2.
      apply rank_sequent_append.
        apply rank_sequent_append_left in H2.
        trivial.
      apply H1.
      apply rank_sequent_append_right in H2.
      trivial.
    repeat split; trivial.
    apply const_rank_G_set; trivial.
  Qed.


  Lemma other_context_G_n_set :
    forall(n : nat)(f1 f2 : lambda_formula V L)(sl sr : sequent V L)
          (rbase : sequent_rule V L),
      ((exists f1 f2 : lambda_formula V L, rbase = bare_and_rule f1 f2) \/
       (exists f1 f2 : lambda_formula V L, rbase = bare_neg_and_rule f1 f2) \/
       (exists f : lambda_formula V L, rbase = bare_neg_neg_rule f)) ->
      rank_sequent n ((f1 :: sl) ++ conclusion rbase ++ sr) ->
      (rank_formula n f1 -> rank_formula n f2) ->
        G_n_set n (rule_add_context (f2 :: sl) sr rbase).
  Proof.
    intros n f1 f2 sl sr rbase H H0 H1.
    apply sequent_other_context_G_n_set with (sl1 := f1 :: sl)(sr1 := sr); 
          eauto.
    clear - H1.
    intros H.
    apply rank_sequent_cons.
      apply H1.
      apply rank_sequent_head in H.
      trivial.
    apply rank_sequent_tail in H.
    trivial.
  Qed.

  Lemma smaller_context_G_n_set :
    forall(n : nat)(f : lambda_formula V L)(sl sr : sequent V L)
          (rbase : sequent_rule V L),
      ((exists f1 f2 : lambda_formula V L, rbase = bare_and_rule f1 f2) \/
       (exists f1 f2 : lambda_formula V L, rbase = bare_neg_and_rule f1 f2) \/
       (exists f : lambda_formula V L, rbase = bare_neg_neg_rule f)) ->
      rank_sequent n ((f :: sl) ++ conclusion rbase ++ sr) ->
        G_n_set n (rule_add_context sl sr rbase).
  Proof.
    intros n f sl sr rbase H H0.
    apply sequent_other_context_G_n_set with (sl1 := f :: sl)(sr1 := sr); auto.
    clear.
    intros H.
    apply rank_sequent_tail in H.
    trivial.
  Qed.


  (**************************************************************************)
  (** ***  GC rule set  *)
  (**************************************************************************)

  Definition GC_set : set (sequent_rule V L) :=
    union G_set is_cut_rule.

  Lemma GC_multiset : rule_multiset GC_set.
  Proof.
    apply multiset_union.
      apply G_multiset.
    apply cut_rule_multiset.
  Qed.


  Definition GC_n_set(n : nat) : set (sequent_rule V L) :=
    rank_rules n GC_set.

  Lemma GC_n_multiset : forall(n : nat), rule_multiset (GC_n_set n).
  Proof.
    intros n.
    apply multiset_rank_rules.
    apply GC_multiset.
  Qed.

  Lemma G_n_subset_GC_n : forall(n : nat),
    subset (G_n_set n) (GC_n_set n).
  Proof.
    intros n.
    unfold G_n_set, GC_n_set in *.
    apply rank_rules_mono.
    unfold GC_set in *.
    apply subset_union_left.
  Qed.

  Lemma C_n_subset_GC_n : forall(n : nat),
    subset (bounded_cut_rules V L n) (GC_n_set n).
  Proof.
    intros n.
    unfold bounded_cut_rules, GC_n_set, GC_set in *.
    apply rank_rules_mono.
    apply subset_union_right.
  Qed.

  Lemma GC_n_as_G_C_union : forall(n : nat),
    set_equal (GC_n_set n) (union (bounded_cut_rules V L n) (G_n_set n)).
  Proof.
    intros n.
    apply set_equal_subset_char.
    split.
      unfold GC_n_set, bounded_cut_rules, G_n_set in *.
      eapply set_equal_subset.
          apply set_equal_refl.
        apply rank_rules_distribute_union.
      apply rank_rules_mono.
      unfold GC_set in *.
      eapply set_equal_subset.
          apply union_comm.
        apply set_equal_refl.
      apply subset_refl.
    apply subset_union_lub.
      apply C_n_subset_GC_n.
    apply G_n_subset_GC_n.
  Qed.


  (**************************************************************************)
  (** **** provability with GC_n *)
  (**************************************************************************)

  Lemma provable_with_cut :
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))
          (gl gr dl dr : sequent V L)(f : lambda_formula V L)(n : nat),
      subset (GC_n_set n) (rank_rules n rules) ->
      rank_formula n f ->
      rank_sequent n (gl ++ gr ++ dl ++ dr) ->
      provable (rank_rules n rules) hyp (gl ++ f :: gr) ->
      provable (rank_rules n rules) hyp (dl ++ (lf_neg f) :: dr) ->
        provable (rank_rules n rules) hyp (gl ++ gr ++ dl ++ dr).
  Proof.
    intros rules hyp gl gr dl dr f n H H0 H1 H2 H3.
    apply provable_with_rule with 
                   (assum := [gl ++ f :: gr; dl ++ (lf_neg f) :: dr]).
      apply H.
      apply C_n_subset_GC_n.
      split.
        exists gl, gr, dl, dr, f.
        split.
          auto.
        apply list_reorder_refl.
      split.
        apply every_nth_cons.
          eapply const_rank_cut_rule_left; eauto.
        apply every_nth_cons.
          eapply const_rank_cut_rule_right; eauto.
        apply every_nth_empty.
      trivial.
    apply every_nth_cons; trivial.
    apply every_nth_cons; trivial.
    apply every_nth_empty.
  Qed.


  (***************************************************************************)
  (** ***  One-step rules, Def 3.3, page 11  *)
  (***************************************************************************)

  Definition one_step_rule(r : sequent_rule V L) : Prop :=
    every_nth prop_sequent (assumptions r) /\
    simple_modal_sequent (conclusion r) /\
    every_nth (fun(s : sequent V L) => 
                 incl (prop_var_sequent s) (prop_var_sequent (conclusion r)))
      (assumptions r) /\
    conclusion r <> [].

  Lemma one_step_rule_multiset : rule_multiset one_step_rule.
  Proof.
    unfold rule_multiset in *.
    intros or s H H0.
    unfold one_step_rule in *.
    decompose [and] H; clear H.
    exists {|assumptions := assumptions or; conclusion := s |}.
    repeat split.
            simpl.
            intros n n_less_rs n_less_rr.
            erewrite nth_tcc_irr.
            apply list_reorder_refl.
          trivial.
        simpl.
        eapply simple_modal_sequent_list_reorder.
          eexact H0.
        trivial.
      simpl.
      intros n n_less.
      specialize (H2 n n_less).
      simpl in H2.
      eapply incl_tran.
        eexact H2.
      apply incl_list_reorder.
      apply prop_var_sequent_list_reorder.
      trivial.
    simpl.
    assert (H6 := list_reorder_length _ _ H0).
    clear - H5 H6.
    destruct s.
      destruct (conclusion or).
        apply H5.
      discriminate.
    destruct (conclusion or).
      discriminate.
    discriminate.
  Qed.


  Definition one_step_rule_set(rules : set (sequent_rule V L)) : Prop :=
    forall(r : sequent_rule V L), rules r -> one_step_rule r.


  Lemma one_step_rule_propositional_assumption :
    forall(r : sequent_rule V L),
      one_step_rule r ->
        every_nth propositional_sequent (assumptions r).
  Proof.
    intros r H n n_less.
    apply prop_sequent_is_propositional.
    unfold one_step_rule in *.
    decompose [ex and or dep_and] H; clear H.
    apply H0.
  Qed.

  Lemma rank_subst_assumptions :
    forall(r : sequent_rule V L)(sigma : lambda_subst V L)(n : nat),
      one_step_rule r ->
      rank_subst n sigma ->
      every_nth (rank_sequent n)
        (map (subst_sequent sigma) (assumptions r)).
  Proof.
    unfold every_nth in *.
    intros r sigma n H H0 n0 n_less.
    rewrite nth_map.
    apply rank_sequent_subst_prop_sequent.
      unfold one_step_rule in *.
      destruct H.
      apply H.
    trivial.
  Qed.

  Lemma rank_sequent_subst_nth_assumptions :
    forall(r : sequent_rule V L)(sigma : lambda_subst V L)(n i : nat)
          (i_less : i < length (assumptions r)),
      one_step_rule r ->
      rank_subst n sigma ->
      rank_sequent n (subst_sequent sigma (nth (assumptions r) i i_less)).
  Proof.
    intros r sigma n i i_less H H0.
    eapply rank_subst_assumptions in H; eauto.
    rewrite every_nth_map in H.
    apply H.
  Qed.
  
  Lemma rank_subst_conclusion :
    forall(r : sequent_rule V L)(sigma : lambda_subst V L)(n : nat),
      1 < n ->
      one_step_rule r ->
      rank_subst (pred n) sigma ->
      rank_sequent n (subst_sequent sigma (conclusion r)).
  Proof.
    intros r sigma n H H0 H1.
    apply rank_sequent_subst_simple_modal_sequent.
        unfold one_step_rule in *.
        decompose [and] H0; clear H0.
        trivial.
      trivial.
    trivial.
  Qed.

  Lemma one_step_rule_simple_modal_conclusion :
    forall(r : sequent_rule V L),
      one_step_rule r -> simple_modal_sequent (conclusion r).
  Proof.
    intros r H.
    unfold one_step_rule in *.
    tauto.
  Qed.

  Lemma one_step_rule_subst_top_modal_conclusion :
    forall(r : sequent_rule V L)(sigma : lambda_subst V L),
      one_step_rule r ->
        top_modal_sequent (subst_sequent sigma (conclusion r)).
  Proof.
    intros r sigma H.
    apply one_step_rule_simple_modal_conclusion in H.
    unfold simple_modal_sequent, top_modal_sequent in *.
    eapply every_nth_map.
    intros i i_less.
    specialize (H i i_less).
    destruct (nth (conclusion r) i i_less); try contradiction.
      destruct l; try contradiction.
      rewrite subst_form_char.
      rewrite subst_form_char.
      simpl.
      trivial.
    rewrite subst_form_char.
    simpl.
    trivial.
  Qed.

  Lemma one_step_rule_prop_modal_prop_conclusion :
    forall(r : sequent_rule V L),
      one_step_rule r -> prop_modal_prop_sequent (conclusion r).
  Proof.
    intros r H.
    apply simple_modal_sequent_is_prop_modal_prop.
    apply one_step_rule_simple_modal_conclusion.
    trivial.
  Qed.

  Lemma one_step_rule_incl_prop_var_sequent :
    forall(r : sequent_rule V L),
      one_step_rule r ->
        every_nth 
          (fun(a : sequent V L) => 
                   incl (prop_var_sequent a) (prop_var_sequent (conclusion r)))
          (assumptions r).
  Proof.
    intros r H.
    unfold one_step_rule in *.
    tauto.
  Qed.

  Lemma one_step_rule_nonempty_conclusion :
    forall(r : sequent_rule V L),
      one_step_rule r -> conclusion r <> [].
  Proof.
    intros r H.
    unfold one_step_rule in *.
    tauto.
  Qed.

  (* 
   * Lemma rank_one_step_rules : 
   *   forall(n : nat)(r : sequent_rule V L),
   *     one_step_rule r ->
   *     every_nth (rank_sequent 1) (assumptions r) /\
   *     rank_sequent 2 (conclusion r).
   * Proof.
   *   intros n r H.
   *   unfold one_step_rule in *.
   *   decompose [and] H; clear H.
   *   clear H1.
   *   split.
   *     clear H2 H4.
   *     intros i i_less.
   *     specialize (H0 i i_less).
   *     unfold prop_sequent, rank_sequent in *.
   *     intros j j_less.
   *     specialize (H0 j j_less).
   *     unfold rank_formula in *.
   *     destruct (nth (nth (assumptions r) i i_less) j j_less).
   *           rewrite modal_rank_char.
   *           trivial.
   *         simpl in H0.
   *         destruct l.
   *               rewrite modal_rank_char.
   *               rewrite modal_rank_char.
   *               trivial.
   *             contradiction.
   *           contradiction.
   *         contradiction.
   *       contradiction.
   *     contradiction.
   *   clear H0.
   *   unfold rank_sequent, simple_modal_sequent in *.
   *   intros i i_less.
   *   specialize (H2 i i_less).
   *   unfold rank_formula in *.
   *   destruct (nth (conclusion r) i i_less).
   *         contradiction.
   *       simpl in H2.
   *       destruct l.
   *             contradiction.
   *           contradiction.
   *         contradiction.
   *       rewrite modal_rank_char.
   *       rewrite modal_rank_char.
   *       apply le_n_S.
   *       simpl in H2.
   *       unfold nat_list_max in *.
   *       fold nat_list_max in *.
   *       apply Max.max_lub.
   *         trivial.
   *       apply nat_list_max_le.
   *       apply every_nth_map.
   *       intros j j_less.
   *       specialize (H2 j j_less).
   *       destruct (nth (list_of_counted_list c) j j_less).
   *             rewrite modal_rank_char.
   *             trivial.
   *           contradiction.
   *         contradiction.
   *       contradiction.
   *     contradiction.
   *     was not finished here
   * Qed.
   *)


  (***************************************************************************)
  (** ***  S(R), Def 3.5, page 11  *)
  (***************************************************************************)

  (* sequent rules S(R), Def 3.5, page 11 *)
  Definition weaken_subst_rule(rules : set (sequent_rule V L))
                                                   : set (sequent_rule V L) :=
    fun(r_subst : sequent_rule V L) =>
      exists(r_base : sequent_rule V L)(sigma : lambda_subst V L)
            (delta : sequent V L),
        rules r_base /\
        assumptions r_subst = map (subst_sequent sigma) (assumptions r_base) /\
        list_reorder (conclusion r_subst) 
                     ((subst_sequent sigma (conclusion r_base)) ++ delta).

  Lemma weaken_subst_rule_multiset : 
    forall(rules : set (sequent_rule V L)),
      rule_multiset (weaken_subst_rule rules).
  Proof.
    unfold rule_multiset, weaken_subst_rule in *.
    intros rules or s H H0.
    decompose [ex and or dep_and] H; clear H.
    rename x into r_base, x0 into sigma, x1 into delta.
    exists {| assumptions := assumptions or; conclusion := s |}.
    split.
      unfold reordered_rule in *.
      simpl.
      repeat split; auto.
      intros n n_less_rs n_less_rr.
      erewrite nth_tcc_irr.
      apply list_reorder_refl.
    exists r_base, sigma, delta.
    simpl.
    repeat split; auto.
    eapply list_reorder_trans.
      apply list_reorder_symm.
      eexact H0.
    trivial.
  Qed.

  Lemma subst_closed_weaken_subst :
    forall(rules : set (sequent_rule V L)),
      subst_closed_rule_set (weaken_subst_rule rules).
  Proof.
    clear. 
    unfold subst_closed_rule_set in *.
    intros rules sigma r H.
    unfold weaken_subst_rule, subst_sequent_rule in *.
    simpl in *.
    decompose [ex and or dep_and] H; clear H.
    rewrite H0.
    exists x, (subst_compose sigma x0), (subst_sequent sigma x1).
    repeat split.
        trivial.
      rewrite map_map.
      apply map_ext.
      intros a.
      rewrite subst_sequent_compose.
      trivial.
    rewrite subst_sequent_compose.
    rewrite <- subst_sequent_append.
    apply list_reorder_map.
    trivial.
  Qed.

  Lemma R_set_no_empty_conclusion : 
    forall(rules : set (sequent_rule V L))(r : sequent_rule V L),
      one_step_rule_set rules ->
      weaken_subst_rule rules r -> 
        conclusion r <> [].
  Proof.
    intros rules r H H0 H1.
    unfold weaken_subst_rule in *.
    decompose [ex and] H0; clear H0.
    rename x into rbase, x0 into sigma, x1 into delta.
    clear H2.
    apply H in H3.
    unfold one_step_rule in *.
    decompose [and] H3; clear H3.
    clear - H1 H5 H7.
    destruct (conclusion rbase).
      apply H7.
      trivial.
    clear H7.
    simpl in *.
    remember (subst_form sigma l :: subst_sequent sigma s ++ delta).
    destruct H5.
      discriminate.
    discriminate.
  Qed.

  Lemma R_set_1_empty :
    forall(rules : set (sequent_rule V L)),
      one_step_rule_set rules ->
        set_equal (rank_rules 1 (weaken_subst_rule rules)) (empty_set _).
  Proof.
    intros rules H r.
    split.
      intros H0.
      exfalso.
      destruct H0.
      unfold weaken_subst_rule in *.
      decompose [ex and] H0; clear H0.
      rename x into rbase, x0 into sigma, x1 into delta.
      destruct H1.
      clear H0 H2.
      rewrite (rank_sequent_list_reorder _ _ _ H5) in H1.
      clear r H5.
      apply rank_sequent_append_left in H1.
      apply subst_sequent_rank_increase in H1.
      specialize (H _ H3).
      unfold one_step_rule in *.
      decompose [and] H; clear H.
      clear - H1 H4 H6.
      apply minimal_rank_simple_modal_sequent in H1; trivial.
      omega.
    intros H0.
    contradiction.
  Qed.


  (***************************************************************************)
  (** ***  GR, convention 3.8, page 13  *)
  (***************************************************************************)

  Definition GR_set(rules : set (sequent_rule V L)) : set (sequent_rule V L) :=
    union G_set (weaken_subst_rule rules).

  Lemma GR_multiset : forall(rules : set (sequent_rule V L)),
    rule_multiset (GR_set rules).
  Proof.
    intros rules.
    apply multiset_union.
      apply G_multiset.
    apply weaken_subst_rule_multiset.
  Qed.

  Definition GR_set_wo_ax(rules : set (sequent_rule V L))
                                                    : set (sequent_rule V L) :=
    union G_struct_set (weaken_subst_rule rules).

  Lemma GR_set_wo_ax_multiset : forall(rules : set (sequent_rule V L)),
    rule_multiset (GR_set_wo_ax rules).
  Proof.
    intros rules.
    apply multiset_union.
      apply G_struct_multiset.
    apply weaken_subst_rule_multiset.
  Qed.

  Lemma GR_set_struct_union : forall(rules : set (sequent_rule V L)),
    set_equal (GR_set rules) (union is_ax_rule (GR_set_wo_ax rules)).
  Proof.
    clear. 
    assert (H := G_set_struct_union).
    unfold GR_set, GR_set_wo_ax, union in *.
    intros rules r.
    specialize (H r).
    simpl in *.
    tauto.
  Qed.

  Lemma subst_closed_GR_set_wo_ax :
    forall(rules : set (sequent_rule V L)),
      subst_closed_rule_set (GR_set_wo_ax rules).
  Proof.
    clear. 
    unfold subst_closed_rule_set in *.
    intros rules sigma r H.
    destruct H.
      left.
      apply subst_closed_G_struct_set.
      trivial.
    right.
    apply subst_closed_weaken_subst.
    trivial.
  Qed.

  Lemma GR_set_no_empty_conclusion : 
    forall(rules : set (sequent_rule V L))(r : sequent_rule V L),
      one_step_rule_set rules ->
      GR_set rules r -> 
        conclusion r <> [].
  Proof.
    intros rules r H H0.
    destruct H0.
      apply G_set_no_empty_conclusion; trivial.
    eapply R_set_no_empty_conclusion; eauto.
  Qed.

  Lemma G_subset_GR : forall(rules : set (sequent_rule V L)),
    subset G_set (GR_set rules).
  Proof.
    intros rules.
    unfold GR_set in *.
    apply subset_union_left.
  Qed.


  Definition GR_n_set(rules : set (sequent_rule V L))(n : nat)
                                                    : set (sequent_rule V L) :=
    rank_rules n (GR_set rules).

  Lemma GR_n_multiset : forall(rules : set (sequent_rule V L))(n : nat),
    rule_multiset (GR_n_set rules n).
  Proof.
    intros rules n.
    apply multiset_rank_rules.
    apply GR_multiset.
  Qed.

  (* not needed, only used in provable_GR_n_list_reorder, but not Elsewhere *)
  Lemma provable_GR_n_hyp_list_reorder : 
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))
          (n : nat)(s1 s2 : sequent V L),
      list_reorder s1 s2 ->
      sequent_multiset hyp ->
      provable (GR_n_set rules n) hyp s1 ->
        provable (GR_n_set rules n) hyp s2.
  Proof.
    intros rules hyp n s1 s2 H H0 H1.
    eapply multiset_provability; eauto.
    apply GR_n_multiset.
  Qed.

  Lemma provable_GR_n_list_reorder : 
    forall(rules : set (sequent_rule V L))(n : nat)(s1 s2 : sequent V L),
      list_reorder s1 s2 ->
      provable (GR_n_set rules n) (empty_sequent_set V L) s1 ->
        provable (GR_n_set rules n) (empty_sequent_set V L) s2.
  Proof.
    intros rules n s1 s2 H H0.
    eapply provable_GR_n_hyp_list_reorder; eauto.
    apply sequent_multiset_empty.
  Qed.

  Lemma GR_n_set_struct_union : 
    forall(rules : set (sequent_rule V L))(n : nat),
      set_equal (GR_n_set rules n)
                (rank_rules n (union is_ax_rule (GR_set_wo_ax rules))).
  Proof.
    intros rules n.
    unfold GR_n_set in *.
    apply rank_rules_set_eq.
    apply GR_set_struct_union.
  Qed.

  Lemma G_n_subset_GR_n :
    forall(rules : set (sequent_rule V L))(n : nat),
      subset (G_n_set n) (GR_n_set rules n).
  Proof.
    intros rules n.
    unfold G_n_set, GR_n_set in *.
    apply rank_rules_mono.
    unfold GR_set in *.
    apply subset_union_left.
  Qed.

  Lemma R_n_subset_GR_n :
    forall(rules : set (sequent_rule V L))(n : nat),
      subset (rank_rules n (weaken_subst_rule rules)) (GR_n_set rules n).
  Proof.
    intros rules n.
    unfold GR_n_set in *.
    apply rank_rules_mono.
    unfold GR_set in *.
    apply subset_union_right.
  Qed.


  (***************************************************************************)
  (** ***  GRC, page 14  *)
  (***************************************************************************)

  Definition GRC_set(rules : set (sequent_rule V L))
                                                    : set (sequent_rule V L) :=
    union (GR_set rules) is_cut_rule.

  Lemma GRC_multiset : forall(rules : set (sequent_rule V L)),
    rule_multiset (GRC_set rules).
  Proof.
    intros rules.
    apply multiset_union.
      apply GR_multiset.
    apply cut_rule_multiset.
  Qed.

  Definition GRC_set_wo_ax(rules : set (sequent_rule V L))
                                                    : set (sequent_rule V L) :=
    union (GR_set_wo_ax rules) is_cut_rule.

  Lemma GRC_set_wo_ax_multiset : forall(rules : set (sequent_rule V L)),
    rule_multiset (GRC_set_wo_ax rules).
  Proof.
    intros rules.
    apply multiset_union.
      apply GR_set_wo_ax_multiset.
    apply cut_rule_multiset.
  Qed.

  Lemma GRC_set_struct_union : forall(rules : set (sequent_rule V L)),
    set_equal (GRC_set rules) (union is_ax_rule (GRC_set_wo_ax rules)).
  Proof.
    clear. 
    assert (H := GR_set_struct_union).
    unfold GRC_set, GRC_set_wo_ax, union in *.
    intros rules r.
    specialize (H rules r).
    tauto.
  Qed.

  Lemma subst_closed_GRC_set_wo_ax :
    forall(rules : set (sequent_rule V L)),
      subst_closed_rule_set (GRC_set_wo_ax rules).
  Proof.
    clear. 
    unfold subst_closed_rule_set in *.
    intros rules sigma r H.
    destruct H.
      left.
      apply subst_closed_GR_set_wo_ax.
      trivial.
    right.
    apply subst_cut_rule.
    trivial.
  Qed.

  Lemma GR_subset_GRC :
    forall(rules : set (sequent_rule V L)),
      subset (GR_set rules) (GRC_set rules).
  Proof.
    unfold GRC_set in *.
    intros rules.
    apply subset_union_left.
  Qed.


  Definition GRC_n_set(rules : set (sequent_rule V L))(n : nat)
                                                    : set (sequent_rule V L) :=
    rank_rules n (GRC_set rules).

  Lemma GRC_n_multiset : forall(rules : set (sequent_rule V L))(n : nat),
    rule_multiset (GRC_n_set rules n).
  Proof.
    intros rules n.
    apply multiset_rank_rules.
    apply GRC_multiset.
  Qed.

  Lemma provable_GRC_n_list_reorder : 
    forall(rules : set (sequent_rule V L))(n : nat)(s1 s2 : sequent V L),
      list_reorder s1 s2 ->
      provable (GRC_n_set rules n) (empty_sequent_set V L) s1 ->
        provable (GRC_n_set rules n) (empty_sequent_set V L) s2.
  Proof.
    intros rules n s1 s2 H H0.
    eapply multiset_provability.
          apply GRC_n_multiset.
        apply sequent_multiset_empty.
      eexact H.
    trivial.
  Qed.

  Definition GRC_n_set_wo_ax(rules : set (sequent_rule V L))(n : nat)
                                                    : set (sequent_rule V L) :=
    rank_rules n (GRC_set_wo_ax rules).

  Lemma GRC_n_set_wo_ax_multiset : 
    forall(rules : set (sequent_rule V L))(n : nat),
      rule_multiset (GRC_n_set_wo_ax rules n).
  Proof.
    intros rules n.
    apply multiset_rank_rules.
    apply GRC_set_wo_ax_multiset.
  Qed.

  Lemma GRC_n_set_struct_union : 
    forall(rules : set (sequent_rule V L))(n : nat),
      set_equal (GRC_n_set rules n)
                (rank_rules n (union is_ax_rule (GRC_set_wo_ax rules))).
  Proof.
    intros rules n.
    unfold GRC_n_set in *.
    apply rank_rules_set_eq.
    apply GRC_set_struct_union.
  Qed.

  Lemma GRC_n_set_empty :
    forall(rules : set (sequent_rule V L)),
      one_step_rule_set rules ->
      set_equal (GRC_n_set rules 0) (empty_set (sequent_rule V L)).
  Proof.
    unfold set_equal, GRC_n_set, GRC_set in *.
    split.
      rename a into r.
      intros H0.
      destruct H0.
      destruct H0.
        destruct H1.
        apply GR_set_no_empty_conclusion in H0; trivial.
        destruct (conclusion r).
          exfalso.
          apply H0.
          trivial.
        apply rank_sequent_head in H2.
        apply rank_formula_zero in H2.
        contradiction.
      unfold is_cut_rule in *.
      decompose [ex and] H0; clear H0.
      destruct H1.
      rewrite H3 in *.
      clear - H0.
      apply every_nth_head in H0.
      apply rank_sequent_append_right in H0.
      apply rank_sequent_head in H0.
      apply rank_formula_zero in H0.
      contradiction.
    contradiction.
  Qed.


  Lemma GR_n_subset_GRC_n :
    forall(rules : set (sequent_rule V L))(n : nat),
      subset (GR_n_set rules n) (GRC_n_set rules n).
  Proof.
    unfold GR_n_set, GRC_n_set in *.
    intros rules n.
    apply rank_rules_mono.
    apply GR_subset_GRC.
  Qed.

  Lemma GC_n_subset_GRC_n :
    forall(rules : set (sequent_rule V L))(n : nat),
      subset (GC_n_set n) (GRC_n_set rules n).
  Proof.
    intros rules n.
    unfold GC_n_set, GRC_n_set in *.
    apply rank_rules_mono.
    unfold GRC_set, GC_set in *.
    apply subset_union_both.
      apply G_subset_GR.
    apply subset_refl.
  Qed.

  Lemma G_n_subset_GRC_n :
    forall(rules : set (sequent_rule V L))(n : nat),
      subset (G_n_set n) (GRC_n_set rules n).
  Proof.
    intros rules n.
    unfold G_n_set, GRC_n_set in *.
    apply rank_rules_mono.
    unfold GRC_set in *.
    eapply subset_trans.
      apply G_subset_GR.
    apply subset_union_left.
  Qed.

  Lemma R_n_subset_GRC_n :
    forall(rules : set (sequent_rule V L))(n : nat),
      subset (rank_rules n (weaken_subst_rule rules)) (GRC_n_set rules n).
  Proof.
    intros rules n.
    unfold GRC_n_set in *.
    apply rank_rules_mono.
    unfold GRC_set, GR_set in *.
    eapply subset_trans.
      apply subset_union_right.
    apply subset_union_left.
  Qed.

  Lemma GRC_n_as_GR_C_union : 
    forall(rules : set (sequent_rule V L))(n : nat),
      set_equal (GRC_n_set rules n) 
                (union (bounded_cut_rules V L n) (GR_n_set rules n)).
  Proof.
    intros rules n.
    apply set_equal_subset_char.
    split.
      unfold GRC_n_set, bounded_cut_rules, GR_n_set in *.
      eapply set_equal_subset.
          apply set_equal_refl.
        apply rank_rules_distribute_union.
      apply rank_rules_mono.
      unfold GRC_set in *.
      eapply set_equal_subset.
          apply union_comm.
        apply set_equal_refl.
      apply subset_refl.
    apply subset_union_lub.
      unfold bounded_cut_rules, GRC_n_set, GRC_set in *.
      apply rank_rules_mono.
      apply subset_union_right.
    apply GR_n_subset_GRC_n.
  Qed.

End Rule_sets.


Implicit Arguments const_rank_G_set [V L].
Implicit Arguments decompose_G_n_set_coarsly [V L].
Implicit Arguments decompose_G_n_set [V L].
Implicit Arguments other_context_G_n_set [V L].
Implicit Arguments sequent_other_context_G_n_set [V L].
Implicit Arguments smaller_context_G_n_set [V L].
Implicit Arguments one_step_rule [V L].
Implicit Arguments one_step_rule_set [V L].
Implicit Arguments one_step_rule_propositional_assumption [V L].
Implicit Arguments one_step_rule_nonempty_conclusion [V L].
Implicit Arguments one_step_rule_prop_modal_prop_conclusion [V L].
Implicit Arguments one_step_rule_incl_prop_var_sequent [V L].
Implicit Arguments weaken_subst_rule [V L].
Implicit Arguments GR_set [V L].
Implicit Arguments GR_n_set [V L].
Implicit Arguments GRC_set [V L].
Implicit Arguments GRC_n_set [V L].
Implicit Arguments GRC_n_set_struct_union [V L].
Implicit Arguments GRC_n_multiset [V L].
