(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: slice.v,v 1.13 2013/04/10 11:17:16 tews Exp $
 *)

(** ** Notions from the slice category Set/P(V), 4.11

      The semantics of coalgebric logics is defined by coalgebras in
      the simple slice category [Set / P(V)]. This category has tuples
      [(X, f)] as objects, where [X] is a set and [f : X -> P(V)] is a
      function (which, in the semantics, gives the covaluation of
      propositional variables). A morphism from [(X,f)] to [(Y,g)] is
      function [h : X -> Y] such that [g ∘ h = f].

      A [T]-coalgebra, for a functor [T], is a function [X -> T(X)].
      For coalgebras in the slice we need to lift the functor [T] to
      the slice category. This happens via [TL(X) = (T(X) * P(V),
      \pi_2)] and [TL(f) = T(f) * id], where [TL] is the lifted
      functor on the slice.

      This is a lot of categorical noice for just the following
      point: A model is a triple [(X,f,coval)] with
      - [X] a set of states
      - [f : X -> T(X)] the transition function
      - [coval : X -> P(V)] the covaluation of propositional variables
      - [p] holds for [x] iff [p] holds for [f(x)], for a variable p and a state [x]
      (I don't know, if the last point is needed somewhere.)

      In the semantics, I simply define a model to be a triple,
      instead of using the notions from the slice category. However,
      the completeness proof uses the terminal cone in the slice and a
      specific property (lemma 4.11) about it. 

      This module defines the terminal cone and proves 4.11.

      For the definitions and the proof I work in the embedding of the
      slice in Set. That is, instead of pairs [(X, f)] I only consider
      sets or rather types (in Coq) [X]. The [f] is often not needed,
      and when, it is passed as additional argument. Instead of
      commuting triangles [g ∘ h = f] (as morphisms in the slice) I
      only consider functions [h]. If necessary, the commutation is
      proved as lemma. 
*)


Require Export functor sets cast.

Section Slice.

  Variable V : Type.
  Variable T : functor.

  (***************************************************************************)
  (** ***  Lift a functor T to the slice  *)
  (***************************************************************************)

  Definition slice_obj_T(T : functor)(X V : Type) : Type := 
    prod (obj T X) (set V).

  Definition slice_map_T{X Y : Type}(f : X -> Y) 
                                   : slice_obj_T T X V -> slice_obj_T T Y V :=
    ftimes (fmap T f) id.

  Lemma slice_map_T_id : forall(X : Type),
    slice_map_T (@id X) ≡ @id (slice_obj_T T X V).
  Proof.
    intros X.
    unfold slice_map_T in *.
    eapply feq_transitive.
      apply feq_ftimes_left.
      apply (id_law T).
    apply feq_ftimes_id.
  Qed.

  Lemma feq_slice_map_T : forall(X Y : Type)(f g : X -> Y),
    f ≡ g -> slice_map_T f ≡ slice_map_T g.
  Proof.
    intros X Y f g H.
    apply feq_ftimes_left.
    apply (fmap_feq_law T).
    trivial.
  Qed.

  Lemma feq_slice_map_T_compose : 
    forall(X Y Z : Type)(f : X -> Y)(g : Y -> Z),
      (slice_map_T g) ∘ (slice_map_T f) ≡ slice_map_T (g ∘ f).
  Proof.
    intros X Y Z f g.
    eapply feq_transitive.
    apply feq_ftimes_compose.
    eapply feq_ftimes_both.
      apply feq_symmetric.
      apply (comp_law T).
    apply feq_id_left.
  Qed.


  (***************************************************************************)
  (** ***  n-fold application of T in the slice  *)
  (***************************************************************************)

  Fixpoint iter_obj_T(X : Type)(n : nat) : Type :=
    match n with
      | 0 => X
      | S n => slice_obj_T T (iter_obj_T X n) V
    end.

  Lemma slice_obj_T_eq_lift : forall(X Y : Type),
    X = Y -> slice_obj_T T X V = slice_obj_T T Y V.
  Proof.
    intros X Y H.
    rewrite H.
    trivial.
  Defined.

  Lemma slice_obj_T_eq_lift_refl : forall(X : Type),
    slice_obj_T_eq_lift X X eq_refl = eq_refl.
  Proof.
    intros X.
    trivial.
  Qed.

  Lemma iter_obj_T_S_n : forall(X : Type)(n : nat),
    iter_obj_T (slice_obj_T T X V) n = iter_obj_T X (S n).
  Proof.
    induction n.
      trivial.
    simpl in *.
    apply slice_obj_T_eq_lift.
    exact IHn.
  Defined.

  Fixpoint iter_fmap_T{X Y : Type}(f : X -> Y)(n : nat) : 
                                      (iter_obj_T X n) -> (iter_obj_T Y n) :=
    match n with
      | 0 => f
      | S n => slice_map_T (iter_fmap_T f n)
    end.


  (***************************************************************************)
  (** ***  final object in the slice  *)
  (***************************************************************************)

  Definition slice_final : Type := prod unit (set V).

  Lemma nonempty_slice_final : non_empty_type slice_final.
  Proof.
    exists (tt, empty_set V).
    trivial.
  Qed.

  Definition slice_final_map{X : Type}(coval : X -> set V) 
                                                          : X -> slice_final :=
    pair (final_map X) coval.
    
  Lemma slice_final_map_id :
    @slice_final_map slice_final (@snd unit (set V)) ≡ @id slice_final.
  Proof.
    unfold slice_final_map, slice_final in *.
    eapply feq_rw_r.
      apply feq_pair_proj.
    apply feq_pair_both.
      apply feq_symmetric.
      apply final_map_prop.
    apply feq_symmetric.
    apply feq_id_right.
  Qed.

  Lemma feq_slice_final_map_coval :
    forall(X : Type)(coval_1 coval_2 : X -> set V),
      coval_1 ≡ coval_2 -> slice_final_map coval_1 ≡ slice_final_map coval_2.
  Proof.
    intros X coval_1 coval_2 H.
    eapply feq_pair_right.
    trivial.
  Qed.

  Lemma slice_final_map_pair_prop :
    forall(X Y : Type)(f : X -> Y * (set V)),
      slice_final_map (@snd Y (set V)) ∘ f ≡
      slice_final_map ((@snd Y (set V)) ∘ f).
  Proof.
    intros X Y f.
    unfold slice_final_map, slice_final in *.
    rewrite pair_compose_right.
    eapply feq_pair_left.
    apply final_map_prop.
  Qed.

  Lemma slice_final_map_times_prop :
    forall(X Y : Type)(f : X -> Y),
      slice_final_map (@snd Y (set V)) ∘ (ftimes f id) ≡
      slice_final_map (@snd X (set V)).
  Proof.
    intros X Y f.
    rewrite <- ftimes_def.
    eapply feq_transitive.
      apply slice_final_map_pair_prop.
    apply feq_slice_final_map_coval.
    rewrite pair_proj_right.
    apply feq_id_left.
  Qed.


  (***************************************************************************)
  (** ***  terminal sequence and cone  *)
  (***************************************************************************)

  Definition terminal_obj_sequence(n : nat) : Type :=
    iter_obj_T slice_final n.

  (** I need to define the projections of the terminal sequence cone
      by case analysis on n. Otherwise Coq is not able to recognice
      that the term is a product.

      The first projection has a different type for [n = 0]. Therefore
      I only define it here for [n > 0].
   *)
  Definition terminal_obj_sequence_pi_1(n : nat) :
             terminal_obj_sequence (S n) -> obj T (terminal_obj_sequence n) :=
    match n 
      return terminal_obj_sequence (S n) -> obj T (terminal_obj_sequence n)
    with
      | 0 => @fst (obj T (terminal_obj_sequence 0)) _ 
      | S n => @fst (obj T (terminal_obj_sequence (S n))) _
    end.

  Lemma terminal_obj_sequence_pi_1_char :
    forall(n : nat)(tx : obj T (terminal_obj_sequence n))(pv : set V),
      terminal_obj_sequence_pi_1 n (tx, pv) = tx.
  Proof.
    intros n tx pv.
    destruct n.
      trivial.
    trivial.
  Qed.

  Definition terminal_obj_sequence_pi_2(n : nat) :
                                         terminal_obj_sequence n -> set V :=
    match n return terminal_obj_sequence n -> set V
    with
      | 0 => @snd _ _ 
      | S n => @snd (obj T (terminal_obj_sequence n)) _
    end.

  Lemma nonempty_terminal_obj_sequence :
    forall(n : nat)(coval : set V),
      non_trivial_functor T ->
        exists(x : terminal_obj_sequence n),
          terminal_obj_sequence_pi_2 n x = coval.
  Proof.
    induction n.
      intros coval H.
      exists (tt, coval).
      trivial.
    intros coval H.
    specialize (IHn (empty_set V) H).
    destruct IHn.
    clear H0.
    specialize (H (terminal_obj_sequence n)).
    destruct H.
      exists x.
      trivial.
    clear H.
    exists (x0, coval).
    trivial.
  Qed.


  Definition terminal_morph_sequence(n : nat) : 
                 (terminal_obj_sequence (S n)) -> terminal_obj_sequence n :=
    fun_dom_cast (iter_obj_T_S_n slice_final n)
      (iter_fmap_T 
         (slice_final_map (X := terminal_obj_sequence 1) (snd (B := set V))) 
         n).

  Lemma terminal_morph_sequence_char : forall(n : nat),
    terminal_morph_sequence n =
      match n return terminal_obj_sequence (S n) -> terminal_obj_sequence n
      with
        | 0 => slice_final_map 
                 (X := terminal_obj_sequence 1) (snd (B := set V))
        | S n => slice_map_T (terminal_morph_sequence n)
      end.
  Proof.
    destruct n.
      trivial.
    unfold terminal_morph_sequence, terminal_obj_sequence in *.
    simpl.
    apply eq_sym.
    eapply dom_map_cast with (T := fun X => slice_obj_T T X V).
    trivial.
  Qed.


  Fixpoint terminal_seq_cone{X : Type}(slice_c : X -> slice_obj_T T X V)
                                   (n : nat) : X -> terminal_obj_sequence n :=
    match n with
      | 0 => slice_final_map ((snd (B := set V)) ∘ slice_c)
      | S n => (slice_map_T (terminal_seq_cone slice_c n)) ∘ slice_c
    end.

  Lemma feq_terminal_seq_cone :
    forall(X : Type)(c1 c2 : X -> slice_obj_T T X V)(n : nat),
      c1 ≡ c2 -> terminal_seq_cone c1 n ≡ terminal_seq_cone c2 n.
  Proof.
    induction n.
      intros H.
      simpl.
      apply feq_slice_final_map_coval.
      apply feq_compose_left_both.
      trivial.
    intros H.
    simpl.
    apply feq_compose_both.
      apply feq_slice_map_T.
      apply IHn.
      trivial.
    trivial.
  Qed.

  Lemma terminal_seq_cone_property :
    forall(X : Type)(slice_c : X -> slice_obj_T T X V)(n : nat),
      (terminal_morph_sequence n) ∘ (terminal_seq_cone slice_c (S n))
      ≡ terminal_seq_cone slice_c n.
  Proof.
    induction n.
      simpl.
      rewrite terminal_morph_sequence_char.
      eapply feq_transitive.
        apply feq_compose_associative.
      eapply feq_left_compose_left.
        apply slice_final_map_times_prop.
      apply slice_final_map_pair_prop.
    simpl in *.
    rewrite terminal_morph_sequence_char.
    eapply feq_transitive.
      apply feq_compose_associative.
    eapply feq_left_compose_left.
      apply feq_slice_map_T_compose.
    eapply feq_compose_right_both.
    apply feq_slice_map_T.
    trivial.
  Qed.


  (***************************************************************************)
  (** ***  towards Lemma 4.11 in the slice  *)
  (***************************************************************************)

  Definition slice_unit_coalg : Type := slice_final -> obj T slice_final.

  Definition unit_coalg_sequence(c : slice_unit_coalg)(n : nat) : 
           terminal_obj_sequence n -> terminal_obj_sequence (S n) :=
    fun_codom_cast (iter_obj_T_S_n slice_final n)
      (iter_fmap_T (pair c (snd (B := set V))) n).

  Lemma unit_coalg_sequence_char : 
    forall(c : slice_unit_coalg)(n : nat),
      unit_coalg_sequence c n =
        match n return terminal_obj_sequence n -> terminal_obj_sequence (S n)
        with
          | 0 => pair c (snd (B := set V))
          | S n => slice_map_T (unit_coalg_sequence c n)
        end.
  Proof.
    intros c n.
    destruct n.
      trivial.
    unfold unit_coalg_sequence, terminal_obj_sequence in *.
    simpl.
    apply eq_sym.
    eapply codom_map_cast with (T := fun X => slice_obj_T T X V).
    trivial.
  Qed.


  Definition unit_coalg_seq_cone(c : slice_unit_coalg)(i n : nat) : 
                         terminal_obj_sequence i -> terminal_obj_sequence n :=
    terminal_seq_cone (unit_coalg_sequence c i) n.

  Lemma unit_coalg_seq_cone_after_c : 
    forall(c : slice_unit_coalg)(i n : nat),
      (unit_coalg_seq_cone c (S i) n) ∘ (unit_coalg_sequence c i) 
        ≡ unit_coalg_seq_cone c i n.
  Proof.
    induction n.
      unfold unit_coalg_seq_cone in *.
        rewrite unit_coalg_sequence_char.
        simpl.
        eapply feq_left_compose_left.
          apply slice_final_map_times_prop.
      apply slice_final_map_pair_prop.
    unfold unit_coalg_seq_cone in *.
    simpl.
    rewrite unit_coalg_sequence_char at 2.
    eapply feq_left_compose_left.
      eapply feq_transitive.
        apply feq_slice_map_T_compose.
      apply feq_slice_map_T.
      exact IHn.
    apply feq_reflexive.
  Qed.

  (** Lemma 4.11 in the slice, page 22  *)
  Lemma unit_coalg_seq_cone_identity : 
    forall(c : slice_unit_coalg)(n : nat),
      unit_coalg_seq_cone c n n ≡ @id (terminal_obj_sequence n).
  Proof.
    induction n.
      unfold unit_coalg_seq_cone, terminal_obj_sequence in *.
      simpl.
      rewrite unit_coalg_sequence_char.
      rewrite pair_proj_right.
      apply slice_final_map_id.
    unfold unit_coalg_seq_cone.
    simpl.
    rewrite unit_coalg_sequence_char at 2.
    eapply feq_transitive.
      apply feq_slice_map_T_compose.
    eapply feq_rw_r.
      apply feq_symmetric.
      apply slice_map_T_id.
    apply feq_slice_map_T.
    eapply feq_transitive.
      apply unit_coalg_seq_cone_after_c.
    trivial.
  Qed.

End Slice.

Implicit Arguments terminal_obj_sequence_pi_2 [V T].
Implicit Arguments terminal_obj_sequence_pi_1 [V T].
Implicit Arguments nonempty_terminal_obj_sequence [V T].
Implicit Arguments terminal_seq_cone [V T X].
Implicit Arguments unit_coalg_sequence [V T].
