(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: some_neg_form.v,v 1.10 2013/04/10 12:06:16 tews Exp $
 *)

(** ** Abstract sequent semantics

      This module proves the correctness of the [some_neg] sequent
      semantics on an abstract level. 

      There are five semantics for
      formulas and sequents in the formalization: propositional
      semantics, the real semantics, the step semantics, the X,tau
      semantics and the TX,tau semantics. Each of them constructs the
      sequent semantics with a finite disjunction from the formula
      semantics. For each of these sequent semantics I need
      concatenation lemmas and some other utility results. The
      predicate [some_neg] provides a simplified sequent semantics,
      for which concatenation and other results can be proved easily.
      This model proves the equivalence of the
      finite-disjunction-semantics with the simplified semantics. 

      As in module 
      #<A HREF="some_neg.html"><spanclass="inlinecode">some_neg</span></A>#
       I first prove the dependent version and
      derive the non-dependent version from it.
 *)

Require Export some_neg formulas.


  (***************************************************************************)
  (** *** Correctness of simplified some_neg_dep semantics I
         This is the dependent version for nonempty sequents
   *)
  (***************************************************************************)

Section Some_neg_dep_form.

  Variable V : Type.
  Variable L : modal_operators.
  Variable form_prop : lambda_formula V L -> Prop.
  Variable P : forall(f : lambda_formula V L), form_prop f -> Prop.

  Definition sequent_prop(s : sequent V L) : Prop := a_list_prop form_prop s.

  (** First hypothesis: well-formedness of formulas is closed under
      disjunction. 
   *)
  Hypothesis form_prop_or :
    forall(f1 f2 : lambda_formula V L), 
      form_prop f1 -> form_prop f2 -> form_prop (lambda_or f1 f2).

  (** This is more a parameter than an hypothesis. It declares the
      function that converts a proof for the well-formedness of a
      sequent into the well-formdness of its disjunction. This
      function is needed for the main statement below and will get
      instantiated. 
   *)
  Hypothesis form_prop_or_formula_of_sequent_iter :
    forall(f : lambda_formula V L)(s : sequent V L),
      sequent_prop (f :: s) ->
        form_prop (or_formula_of_sequent_iter f s).

  (** Second hypothesis: The well-formedness argument is irrelevant
      for the formula semantics.
   *)
  Hypothesis P_tcc_irr : 
    forall(f : lambda_formula V L)(tf1 tf2 : form_prop f),
      P f tf1 -> P f tf2.

  (** Third hypothesis: The formula semantics is closed under
      disjunction.
   *)
  Hypothesis P_or_formula :
    forall(f1 f2 : lambda_formula V L)(tf1 : form_prop f1)(tf2 : form_prop f2),
      P (lambda_or f1 f2) (form_prop_or f1 f2 tf1 tf2)
      <-> ~(~P f1 tf1 /\ ~P f2 tf2).

  Lemma a_list_prop_first_or :
    forall(f1 f2 f3 : lambda_formula V L)(s : sequent V L),
      sequent_prop (f1 :: f2 :: f3 :: s) -> 
        sequent_prop ((lambda_or f1 f2) :: f3 :: s).
  Proof.
    intros f1 f2 f3 s H n n_less.
    destruct n.
      simpl.
      apply form_prop_or.
        eapply a_list_prop_head.
        eexact H.
      eapply a_list_prop_head.
      eapply a_list_prop_tail.
      eexact H.
    remember (f3 :: s) as f3s.
    simpl.
    subst f3s.
    apply a_list_prop_tail in H.
    apply a_list_prop_tail in H.
    apply H.
  Qed.

  Lemma p_or_formula_of_sequent_iter :
    forall(s : sequent V L)(f1 f2 : lambda_formula V L)
          (t_f_s : sequent_prop (f1 :: f2 :: s)),
      P (or_formula_of_sequent_iter f1 (f2 :: s))
         (form_prop_or_formula_of_sequent_iter f1 (f2 :: s) t_f_s)
      <->
      ~(~P f1 (a_list_prop_head t_f_s) /\
        ~P (or_formula_of_sequent_iter f2 s) 
           (form_prop_or_formula_of_sequent_iter f2 s 
                                      (a_list_prop_tail t_f_s))).
  Proof.
    clear - P_tcc_irr P_or_formula.
    induction s.
      intros f1 f2 t_f_s.
      simpl.
      split.
        intros H.
        apply P_or_formula.
        eapply P_tcc_irr.
        eexact H.
      intros H.
      eapply P_tcc_irr.
      apply P_or_formula.
      eexact H.
    rename a into f3.
    split.
      intros H.
      remember (f3 :: s) as f3s.
      simpl in H.
      subst f3s.
      apply P_tcc_irr with
        (tf2 := form_prop_or_formula_of_sequent_iter _ _
                    (a_list_prop_first_or _ _ _ _ t_f_s)) in H.
      apply IHs in H.
      rewrite IHs.
      intros H0.
      destruct H0.
      apply H; clear H; split; intro H.
        eapply P_tcc_irr in H.
        apply P_or_formula in H.
        apply H; clear H; split; intro H.
          apply H0.
          eexact H.
        apply H1; clear H1; intros H1.
        destruct H1.
        apply H1.
        eexact H.
      apply H1; clear H1; intros H1.
      destruct H1.
      apply H2.
      eapply P_tcc_irr.
      eexact H.
    intros H.
    remember (f3 :: s) as f3s.
    simpl.
    subst f3s.
    apply P_tcc_irr with (tf1 := form_prop_or_formula_of_sequent_iter _ _
                    (a_list_prop_first_or _ _ _ _ t_f_s)).
    apply IHs.
    rewrite IHs in H.
    clear IHs.
    intros H0.
    destruct H0.
    apply H0; clear H0.
    eapply P_tcc_irr.
    apply P_or_formula.
    intros H2.
    destruct H2.
    apply H; clear H; split.
      eexact H0.
    intros H3; apply H3; clear H3.
    split.
      eexact H2.
    intros H3.
    apply H1; clear H1.
    eapply P_tcc_irr.
    eexact H3.
  Qed.

  Lemma some_neg_dep_cons_correct :
    forall(s : sequent V L)(f : lambda_formula V L)
          (dep_s : sequent_prop (f :: s)),
      some_neg_dep form_prop P (f :: s) dep_s <-> 
        P (or_formula_of_sequent_iter f s) 
          (form_prop_or_formula_of_sequent_iter f s dep_s).
  Proof.
    clear - P_tcc_irr. 
    induction s.
      simpl.
      split.
        intros H.
        apply some_neg_dep_singleton in H; trivial.
        eapply P_tcc_irr.
        eexact H.
      intros H.
      eapply some_neg_dep_singleton_rev; trivial.
      eexact H.
    rename a into f'.
    split.
      intros H.
      apply some_neg_dep_cons_cons_elim in H; trivial.
      apply p_or_formula_of_sequent_iter; trivial.
      intros H1; destruct H1.
      apply H; clear H; split.
        trivial.
      rewrite IHs.
      trivial.
    intros H.
    apply some_neg_dep_long_neg_intro; trivial.
      discriminate.
    intros H0.
    eapply p_or_formula_of_sequent_iter in H; trivial.
    apply H; clear H; split; intro H.
      apply H0.
      eapply some_neg_dep_head; trivial.
      eexact H.
    eapply IHs in H.
    apply H0.
    apply some_neg_dep_tail; trivial.
  Qed.

  (* 
   * Hypothesis P_false : 
   *   forall(nonempty_v : V)(t_false : form_prop (lambda_false nonempty_v)),
   *     P (lambda_false nonempty_v) t_false -> False.
   * 
   * Hypothesis form_prop_or_formula_of_sequent :
   *   forall(nonempty_v : V)(s : sequent V L),
   *     sequent_prop s ->
   *       form_prop (or_formula_of_sequent s nonempty_v).
   * 
   * Lemma some_neg_dep_correct :
   *   forall(nonempty_v : V)(s : sequent V L)(dep_s : sequent_prop s),
   *     some_neg_dep form_prop P s dep_s <-> 
   *       P (or_formula_of_sequent s nonempty_v) 
   *         (form_prop_or_formula_of_sequent nonempty_v s dep_s).
   * Proof.
   *   clear - P_false.
   *   intros nonempty_v s dep_s.
   *   destruct s.
   *     split.
   *       intros H.
   *       apply some_neg_dep_empty in H.
   *       contradiction.
   *     simpl.
   *     intros H.
   *     apply P_false in H.
   *     contradiction.
   *   split.
   *     intros H.
   *     assert (H0 := some_neg_dep_cons_correct s l dep_s).
   *)

End Some_neg_dep_form.

Implicit Arguments sequent_prop [V L].


  (***************************************************************************)
  (** *** Correctness of simplified some_neg_dep semantics II
         This is the dependent version for possibly empty sequents
   *)
  (***************************************************************************)

Section Some_neg_dep_form_2.

  Variable V : Type.
  Variable L : modal_operators.
  Variable form_prop : lambda_formula V L -> Prop.
  Variable P : forall(f : lambda_formula V L), form_prop f -> Prop.

  (** first hypothesis *)
  Hypothesis form_prop_or :
    forall(f1 f2 : lambda_formula V L), 
      form_prop f1 -> form_prop f2 -> form_prop (lambda_or f1 f2).
  
  (** Declaration of the well-formedness proof conversion function *)
  Hypothesis form_prop_or_formula_of_sequent :
    forall(nonempty_v : V)(s : sequent V L),
      sequent_prop form_prop s ->
        form_prop (or_formula_of_sequent s nonempty_v).

  (** second hypothesis *)
  Hypothesis P_tcc_irr : 
    forall(f : lambda_formula V L)(tf1 tf2 : form_prop f),
      P f tf1 -> P f tf2.

  (** third hypothesis *)
  Hypothesis P_or_formula :
    forall(f1 f2 : lambda_formula V L)(tf1 : form_prop f1)(tf2 : form_prop f2),
      P (lambda_or f1 f2) (form_prop_or f1 f2 tf1 tf2)
      <-> ~(~P f1 tf1 /\ ~P f2 tf2).

  (** Fourth hypothesis: The formula semantics is false for false. *)
  Hypothesis P_false : 
    forall(nonempty_v : V)(t_false : form_prop (lambda_false nonempty_v)),
      P (lambda_false nonempty_v) t_false -> False.

  Lemma form_prop_or_formula_of_sequent_iter :
    forall(nonempty_v : V)(f : lambda_formula V L)(s : sequent V L),
      sequent_prop form_prop (f :: s) ->
        form_prop (or_formula_of_sequent_iter f s).
  Proof.
    clear - form_prop_or_formula_of_sequent.
    intros nonempty_v f s H.
    specialize (form_prop_or_formula_of_sequent nonempty_v _ H).
    trivial.
  Qed.

  Lemma some_neg_dep_correct :
    forall(nonempty_v : V)(s : sequent V L)(dep_s : sequent_prop form_prop s),
      some_neg_dep form_prop P s dep_s <-> 
        P (or_formula_of_sequent s nonempty_v) 
          (form_prop_or_formula_of_sequent nonempty_v s dep_s).
  Proof.
    intros nonempty_v s dep_s.
    destruct s.
      split.
        intros H.
        apply some_neg_dep_empty in H.
        contradiction.
      simpl.
      intros H.
      apply P_false in H.
      contradiction.
    split.
      intros H.
      eapply some_neg_dep_cons_correct with (P := P) 
               (form_prop_or_formula_of_sequent_iter := 
                    form_prop_or_formula_of_sequent_iter nonempty_v) 
               in H; eauto.
    intros H.
    eapply some_neg_dep_cons_correct with (P := P)
               (form_prop_or_formula_of_sequent_iter := 
                    form_prop_or_formula_of_sequent_iter nonempty_v);
               eauto.
  Qed.

End Some_neg_dep_form_2.


Section Some_neg_form.

  (***************************************************************************)
  (** *** Correctness of simplified some_neg semantics
         This is the simple (non-dependent) version for arbitrary
         sequents. 
    *)
  (***************************************************************************)

  Variable V : Type.
  Variable L : modal_operators.
  Variable P : lambda_formula V L -> Prop.

  (** There remain only two hypothesis for the simple version. *)
  Hypothesis P_false : 
    forall(nonempty_v : V),
      P (lambda_false nonempty_v) -> False.

  Hypothesis P_or_formula : forall(f1 f2 : lambda_formula V L),
    P (lambda_or f1 f2) <-> ~(~P f1 /\ ~P f2).

  Lemma noprop_or : forall(f1 f2 : lambda_formula V L), 
    noprop (lambda_formula V L) f1 -> 
    noprop (lambda_formula V L) f2 -> 
    noprop (lambda_formula V L) (lambda_or f1 f2).
  Proof.
    trivial.
  Qed.

  Lemma noprop_or_formula_of_sequnet :
    forall(nonempty_v : V)(s : sequent V L),
      a_list_prop (noprop (lambda_formula V L)) s -> 
        noprop (lambda_formula V L) (or_formula_of_sequent s nonempty_v).
  Proof.
    intros nonempty_v s H.
    cbv.
    trivial.
  Qed.

  Lemma some_neg_correct :
    forall(nonempty_v : V)(s : sequent V L),
      some_neg P s <-> P (or_formula_of_sequent s nonempty_v).
  Proof.
    intros nonempty_v s.
    unfold some_neg in *.
    rewrite some_neg_dep_correct with (form_prop_or := noprop_or) 
            (form_prop_or_formula_of_sequent := noprop_or_formula_of_sequnet).
          apply iff_refl.
        apply lift_p_tcc_irr.
      intros f1 f2 tf1 tf2.
      apply P_or_formula.
    intros nonempty_v0 t_false.
    apply P_false.
  Qed.

End Some_neg_form.
