(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: sound.v,v 1.21 2013/04/10 12:06:16 tews Exp $
 *)


(** ** Soundness, 4.7

      This module proves the soundness theorem for coalgebraic logics.

      I use two definitions of downward correct rules here. The first
      and weaker one is needed for one-step rules. The second is
      based on the simplified sequent semantics. It is simpler but
      also slightly stronger. This second definition is used for the
      propositional rules and also for contraction.
*)


Require Export classic one_step_conditions.

Section Soundness.

  Variable V : Type.
  Variable L : modal_operators.
  Variable T : functor.


  (***************************************************************************)
  (** *** Downward validity  *)
  (***************************************************************************)

  Definition downward_correct_rule(nonempty_v : V)
                    (LS : lambda_structure L T)(r : sequent_rule V L) : Prop :=
    forall(m : model V T),
      every_nth (valid_all_states nonempty_v LS m) (assumptions r) ->
        valid_all_states nonempty_v LS m (conclusion r).

  Definition downward_correct_rule_set(nonempty_v : V)
          (LS : lambda_structure L T)(rules : set (sequent_rule V L)) : Prop :=
    forall(r : sequent_rule V L),
      rules r -> downward_correct_rule nonempty_v LS r.

  Lemma sound_derivation :
    forall(nonempty_v : V)(LS : lambda_structure L T)
          (rules : set (sequent_rule V L))(hypotheses : set (sequent V L))
          (m : model V T)(s : sequent V L),
      downward_correct_rule_set nonempty_v LS rules ->
      (forall(h : sequent V L),
         hypotheses h -> valid_all_states nonempty_v LS m h) ->
      proof rules hypotheses s ->
        valid_all_states nonempty_v LS m s.
  Proof.
    intros nonempty_v LS rules hypotheses m s H H0 p.
    induction p using proof_sequent_ind.
      apply H0.
      trivial.
    apply H.
      trivial.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Stronger downward validity  *)
  (***************************************************************************)

  Definition strong_downward_correct_rule(LS : lambda_structure L T)
                                              (r : sequent_rule V L) : Prop :=
    forall(m : model V T)(x : state m),
      every_nth (state_seq_semantics LS m x) (assumptions r) ->
        state_seq_semantics LS m x (conclusion r).

  Definition downward_correct_rule_strengthen :
    forall(nonempty_v : V)(LS : lambda_structure L T)(r : sequent_rule V L),
      strong_downward_correct_rule LS r ->
        downward_correct_rule nonempty_v LS r.
  Proof.
    unfold downward_correct_rule, strong_downward_correct_rule in *.
    intros  nonempty_v LS r H m H0.
    unfold valid_all_states, is_full_set in *.
    intros a.
    rewrite <- state_seq_semantics_correct; trivial.
    apply H.
    intros i i_less.
    rewrite state_seq_semantics_correct; trivial.
    apply H0.
  Qed.

  Lemma assumption_add_context_state_seq_semantics :
    forall(LS : lambda_structure L T)(m : model V T)(x : state m)
          (sl sr : sequent V L)(sml : list (sequent V L)),
      sml <> [] ->
      every_nth (state_seq_semantics LS m x)
                (map (add_context sl sr) sml) ->
        ~ ( ~ (state_seq_semantics LS m x sl) /\
            ~ (state_seq_semantics LS m x sr) /\
            ~ (every_nth (state_seq_semantics LS m x) sml)).
  Proof.
    induction sml.
      intros H H0.
      auto.
    rename a into s1.
    destruct sml as [| s2 ].
      clear IHsml.
      intros H H0.
      clear H.
      apply every_nth_head in H0.
      unfold add_context in *.
      intros H1.
      decompose [and] H1; clear H1.
      apply state_seq_semantics_append in H0.
      apply H0; clear H0.
      split.
        trivial.
      intros H5.
      apply state_seq_semantics_append in H5.
      apply H5; clear H5.
      split.
        intros H5.
        apply H4; clear H4.
        apply every_nth_cons.
          trivial.
        apply every_nth_empty.
      trivial.
    intros H H0.
    clear H.
    assert (H1 := every_nth_head _ _ _ H0).
    remember (s2 :: sml) as s2sml.
    simpl in H0.
    apply every_nth_tail in H0.
    subst s2sml.
    apply IHsml in H0; clear IHsml.
      intros H2.
      decompose [and] H2; clear H2.
      apply H0; clear H0.
      split; trivial.
      split; trivial.
      intros H6.
      unfold add_context in *.
      apply state_seq_semantics_append in H1.
      apply H1; clear H1.
      split; trivial.
      intros H1.
      apply state_seq_semantics_append in H1.
      apply H1; clear H1.
      split; trivial.
      intros H1.
      apply H5.
      apply every_nth_cons.
        trivial.
      trivial.
    discriminate.
  Qed.

  Lemma strong_downward_correct_context :
    forall(LS : lambda_structure L T)
          (r : sequent_rule V L)(sl sr : sequent V L),
      conclusion r <> [] ->
      strong_downward_correct_rule LS r ->
        strong_downward_correct_rule LS (rule_add_context sl sr r).
  Proof.
    unfold strong_downward_correct_rule in *.
    intros LS r sl sr H H0 m x H1.
    destruct (assumptions r) eqn:H2.
      clear H1.
      simpl.
      unfold add_context in *.
      apply state_seq_semantics_append_left.
      apply state_seq_semantics_append_right.
      apply H0.
      apply every_nth_empty.
    simpl in *.
    apply state_seq_semantics_length_case_intro.
      intros H3.
      destruct (conclusion r) as [|fcr cr].
        exfalso.
        auto.
      unfold add_context in *.
      destruct sl.
        destruct sr.
          simpl in *.
          rewrite app_nil_r in *.
          rewrite <- H2 in *; clear H2.
          assert (every_nth (state_seq_semantics LS m x) (assumptions r)).
            clear - H1.
            intros i i_less.
            rewrite every_nth_map in H1.
            specialize (H1 i i_less).
            simpl in *.
            rewrite app_nil_r in *.
            trivial.
          apply H0 in H2.
          clear - H3 H2.
          apply state_seq_semantics_singleton.
          destruct cr.
            trivial.
          simpl in H3.
          omega.
        clear - H3.
        simpl in *.
        rewrite app_length in H3.
        simpl in *.
        omega.
      clear - H3.
      simpl in *.
      rewrite app_length in H3.
      simpl in *.
      omega.
    intros H3 H4.
    apply assumption_add_context_state_seq_semantics in H1.
      apply H1; clear H1.
      repeat split; intro H1; apply H4.
          apply state_seq_semantics_append_right.
          trivial.
        apply state_seq_semantics_append_left.
        apply state_seq_semantics_append_left.
        trivial.
      apply state_seq_semantics_append_left.
      apply state_seq_semantics_append_right.
      apply H0.
      rewrite <- H2.
      trivial.
    rewrite H2.
    discriminate.
  Qed.


  (**************************************************************************)
  (** *** Strong downward correctness of G  *)
  (**************************************************************************)

  Lemma strong_downward_correct_ax :
    forall(LS : lambda_structure L T)(r : sequent_rule V L),
      is_ax_rule r -> 
        strong_downward_correct_rule LS r.
  Proof.
    unfold is_ax_rule, strong_downward_correct_rule in *.
    unfold simple_tautology, simple_tautology_witness in *.
    intros LS r H m x H0.
    decompose [ex and or dep_and] H; clear H.
    rename x0 into n1, x1 into n2, x2 into v, a into n1_less, a0 into n2_less.
    clear H0 H1.
    apply state_seq_semantics_length_case_intro.
      intros s_len.
      exfalso.
      assert (n1 <> n2).
        intros H4.
        subst n2.
        rewrite nth_tcc_irr with (inside_2 := n1_less) in H3.
        rewrite H2 in H3.
        discriminate.
      omega.
    intros H H4.
    clear H.
    assert (~ (~ form_semantics LS m (nth (conclusion r) n1 n1_less) x /\
               ~ form_semantics LS m (nth (conclusion r) n2 n2_less) x)).
      rewrite H2.
      rewrite H3.
      repeat rewrite form_semantics_char.
      unfold set_inverse in *.
      tauto.
    apply H; clear H; split; intro H; apply H4; clear H4.
      eapply state_seq_semantics_nth_intro.
      eexact H.
    eapply state_seq_semantics_nth_intro.
    eexact H.
  Qed.

  Lemma strong_downward_correct_and :
    forall(LS : lambda_structure L T)(r : sequent_rule V L),
      is_and_rule r -> strong_downward_correct_rule LS r.
  Proof.
    intros LS r H.
    apply and_rule_context in H.
    decompose [ex] H; clear H.
    rename x into sl, x0 into sr, x1 into f1, x2 into f2.
    subst r.
    apply strong_downward_correct_context.
      discriminate.
    clear. 
    unfold bare_and_rule, strong_downward_correct_rule in *.
    simpl.
    intros m x H.
    assert (H0 := every_nth_head _ _ _ H).
    apply every_nth_tail in H.
    apply every_nth_head in H.
    rewrite state_seq_semantics_singleton in *.
    rewrite form_semantics_char.
    unfold intersection in *.
    auto.
  Qed.


  Lemma strong_downward_correct_neg_and :
    forall(LS : lambda_structure L T)(r : sequent_rule V L),
      is_neg_and_rule r -> strong_downward_correct_rule LS r.
  Proof.
    intros LS r H.
    apply neg_and_rule_context in H.
    decompose [ex] H; clear H.
    rename x into sl, x0 into sr, x1 into f1, x2 into f2.
    subst r.
    apply strong_downward_correct_context.
      discriminate.
    clear. 
    unfold bare_neg_and_rule, strong_downward_correct_rule in *.
    simpl.
    intros m x H.
    apply every_nth_head in H.
    rewrite state_seq_semantics_singleton.
    rewrite form_semantics_char.
    rewrite form_semantics_char.
    intros H0.
    destruct H0.
    apply state_seq_semantics_cons_case_elim in H.
    decompose [and or] H; clear H.
      discriminate.
    apply H4; clear H4; split; intros H4.
      contradiction.
    apply state_seq_semantics_singleton in H4.
    contradiction.
  Qed.

  Lemma strong_downward_correct_neg_neg :
    forall(LS : lambda_structure L T)(r : sequent_rule V L),
      is_neg_neg_rule r -> strong_downward_correct_rule LS r.
  Proof.
    intros LS r H.
    apply neg_neg_rule_context in H.
    decompose [ex and or dep_and] H; clear H.
    rename x into sl, x0 into sr, x1 into f.
    subst r.
    apply strong_downward_correct_context.
      discriminate.
    clear. 
    unfold bare_neg_neg_rule, strong_downward_correct_rule in *.
    simpl.
    intros m x H.
    apply every_nth_head in H.
    rewrite state_seq_semantics_singleton in *.
    rewrite form_semantics_char.
    rewrite form_semantics_char.
    unfold set_inverse in *.
    auto.
  Qed.


  (**************************************************************************)
  (** *** Downward correctness other rules *)
  (**************************************************************************)

  (** The cut rule is not sound in intuitionistic logic, because it 
      permits to derive  B  from  A  and  ~A, B , which amounts to
      A /\ ~(~~A /\ ~B) -> B  from which one can derive classical 
      logic, see 
      #<A HREF="classic.html##classic_axiom_sound_cut_left"><spanclass="inlinecode">classic_axiom_sound_cut_left</span></A>#
       in module 
       #<A HREF="classic.html"><spanclass="inlinecode">classic</span></A>#.
   *)
  Lemma strong_downward_correct_cut :
    forall(LS : lambda_structure L T)(r : sequent_rule V L),
      classical_logic ->
      is_cut_rule r -> 
        strong_downward_correct_rule LS r.
  Proof.
    unfold is_cut_rule, strong_downward_correct_rule in *.
    intros LS r classic H m x H0.
    decompose [ex and or dep_and] H; clear H.
    rename x0 into gl, x1 into gr, x2 into dl, x3 into dr, x4 into f.
    rewrite H2 in *; clear H2.
    eapply state_seq_semantics_reorder.
      apply list_reorder_symm.
      eexact H3.
    clear H3.
    assert (H1 := every_nth_head _ _ _ H0).
    apply every_nth_tail in H0.
    apply every_nth_head in H0.
    apply state_seq_semantics_reorder with (s2 := lf_neg f :: dl ++ dr) in H0.
      apply state_seq_semantics_reorder with (s2 := f :: gl ++ gr) in H1.
        apply state_seq_semantics_length_case_intro.
          intros H2.
          generalize (nth_head_tcc (gl ++ gr ++ dl ++ dr) H2).
          destruct (dl ++ dr) as [| d1 dlr].
            rewrite app_nil_r in *.
            destruct (gl ++ gr) as [| g1 glr].
              discriminate.
            rewrite state_seq_semantics_singleton in H0.
            apply state_seq_semantics_cons_case_elim in H1.
            decompose [and or] H1; clear H1.
              discriminate.
            apply double_neg_or in H4; trivial.
            destruct H4.
              contradiction.
            destruct glr.
              apply state_seq_semantics_singleton in H.
              intros l.
              trivial.
            discriminate.
          rewrite app_assoc in H2 |-*.
          destruct (gl ++ gr) as [| g1 glr].
            simpl in *.
            rewrite state_seq_semantics_singleton in H1.

            apply state_seq_semantics_cons_case_elim in H0.
            decompose [and or] H0; clear H0.
              discriminate.
            apply double_neg_or in H4; trivial.
            destruct H4.
              contradiction.
            destruct dlr.
              apply state_seq_semantics_singleton in H.
              auto.
            discriminate.
          exfalso.
          rewrite app_length in H2.
          simpl in H2.
          omega.
        intros H2 H3.
        rewrite append_single_rev in *.
        rewrite app_assoc in H3.
        apply state_seq_semantics_append in H0.
        apply state_seq_semantics_append in H1.
        apply H0; clear H0; split; intro H0.
          apply H1; clear H1; split; intro H1.
            rewrite state_seq_semantics_singleton in *.
            contradiction.
          apply H3; clear H3.
          apply state_seq_semantics_append_right.
          trivial.
        apply H3; clear H3.
        apply state_seq_semantics_append_left.
        trivial.
      apply list_reorder_symm.
      apply list_reorder_cons_parts.
      apply list_reorder_refl.
    apply list_reorder_symm.
    apply list_reorder_cons_parts.
    apply list_reorder_refl.
  Qed.

  Lemma downward_correct_one_step_rule :
    forall(nonempty_v : V)(LS : lambda_structure L T)
          (rules : set (sequent_rule V L))(osr : one_step_rule_set rules)
          (r : sequent_rule V L),
      one_step_sound nonempty_v LS rules osr ->
      weaken_subst_rule rules r ->
        downward_correct_rule nonempty_v LS r.
  Proof.
    unfold downward_correct_rule in *.
    intros nonempty_v LS rules osr r H H0 m H1.
    unfold weaken_subst_rule in *.
    decompose [ex and or dep_and] H0; clear H0.
    rename x into rbase, x0 into sigma, x1 into delta.
    eapply valid_all_states_reorder.
      apply list_reorder_symm.
      eexact H5.
    clear H5.
    apply valid_all_states_append_right.
    rewrite valid_all_states_subst_change_coval.
    assert (H4 := osr _ H3).
    assert (H5 := one_step_rule_nonempty_conclusion _ H4).
    apply one_step_rule_prop_modal_prop_conclusion in H4.
    apply mod_val_semantics_valid with (nonempty_s := H5) (propm_s := H4).
    erewrite mod_seq_val_valid_tcc_irr.
    apply H with (rules_r := H3).
    clear H H4 H5.
    intros n n_less.
    rewrite <- prop_val_semantics_valid.
    rewrite <- valid_all_states_subst_change_coval.
    rewrite H2 in *.
    erewrite every_nth_map in H1.
    apply H1.
  Qed.


  (** The contraction rule is not part of the soundness theorem,
      but its soundness is needed for the semantics admissibility proof 
      in Theorem 4.15.

      Contraction is not sound in intuitionistic logic, because, with 
      empty Gamma, it amounts to ~(~A /\ ~A) -> A, which implies 
      classical logic, see 
      #<A HREF="classic.html##classic_axiom_sound_contraction"><spanclass="inlinecode">classic_axiom_sound_contraction</span></A>#
       in module
      #<A HREF="classic.html"><spanclass="inlinecode">classic</span></A>#.
   *)
  Lemma downward_correct_contraction :
    forall(nonempty_v : V)(LS : lambda_structure L T)(r : sequent_rule V L),
      classical_logic ->
      is_contraction_rule r -> 
        downward_correct_rule nonempty_v LS r.
  Proof.
    intros nonempty_v LS r classic H.
    apply downward_correct_rule_strengthen; trivial.
    unfold is_contraction_rule, strong_downward_correct_rule in *.
    decompose [ex and or dep_and] H; clear H.
    rename x into n, a into n_less, b into H.
    rewrite H; clear H.
    intros m x H.
    apply every_nth_head in H.
    apply state_seq_semantics_length_case_intro.
      intros H0.

      apply state_seq_semantics_cons_case_elim in H.
      decompose [and or] H; clear H.
        exfalso.
        rewrite H2 in *.
        discriminate.
      apply double_neg_or in H3; trivial.
      destruct H3.
        destruct n.
          erewrite nth_tcc_irr.
          eexact H.
        clear - n_less H0.
        rewrite H0 in *.
        omega.
      destruct (conclusion r).
        discriminate.
      destruct s.
        apply state_seq_semantics_singleton in H.
        trivial.
      discriminate.
    intros H0 H1.
    rewrite append_single_rev in H.
    apply state_seq_semantics_append in H.
    apply H; clear H; split; intro H; apply H1; clear H1.
      apply state_seq_semantics_singleton in H.
      apply state_seq_semantics_nth_intro in H.
      trivial.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Towards soundness, theorem 4.7 *)
  (***************************************************************************)

  Lemma downward_correct_GR :
    forall(nonempty_v : V)(LS : lambda_structure L T)
          (rules : set (sequent_rule V L))(osr : one_step_rule_set rules),
      one_step_sound nonempty_v LS rules osr ->
        downward_correct_rule_set nonempty_v LS (GR_set rules).
  Proof.
    intros nonempty_v LS rules osr H r H0.
    destruct H0.
      apply downward_correct_rule_strengthen.
      unfold G_set, union in *.
      decompose [or] H0; clear H0.
            apply strong_downward_correct_ax; trivial.
          apply strong_downward_correct_and; trivial.
        apply strong_downward_correct_neg_and; trivial.
      apply strong_downward_correct_neg_neg; trivial.
    eapply downward_correct_one_step_rule; eauto.
  Qed.

  Lemma downward_correct_GRC :
    forall(nonempty_v : V)(LS : lambda_structure L T)
          (rules : set (sequent_rule V L))(osr : one_step_rule_set rules),
      classical_logic ->
      one_step_sound nonempty_v LS rules osr ->
        downward_correct_rule_set nonempty_v LS (GRC_set rules).
  Proof.
    intros nonempty_v LS rules osr classic H r H0.
    destruct H0.
      eapply downward_correct_GR; eauto.
    apply downward_correct_rule_strengthen; trivial.
    apply strong_downward_correct_cut; trivial.
  Qed.


  (** **** Soundness theorem 4.7, GRC part 

          Soundness for GRC is only provable in classical logic
          because of the cut rule.
   *)
  Theorem sound_GRC :
    forall(nonempty_v : V)(LS : lambda_structure L T)
          (rules : set (sequent_rule V L))(osr : one_step_rule_set rules)
          (s : sequent V L),
      classical_logic ->
      one_step_sound nonempty_v LS rules osr ->
      provable (GRC_set rules) (empty_sequent_set V L) s ->
        valid_all_models nonempty_v LS s.
  Proof.
    intros nonempty_v LS rules osr s classical H H0 H1.
    destruct H0.
    clear H0.
    apply sound_derivation with (3 := x).
      eapply downward_correct_GRC.
        trivial.
      eexact H.
    intros h H0.
    contradiction.
  Qed.

  (** **** soundness theorem 4.7, GR part 
          Soundness of GR can be proven in intuitionistic logic.
   *)
  Theorem sound_GR :
    forall(nonempty_v : V)(LS : lambda_structure L T)
          (rules : set (sequent_rule V L))(osr : one_step_rule_set rules)
          (s : sequent V L),
      one_step_sound nonempty_v LS rules osr ->
      provable (GR_set rules) (empty_sequent_set V L) s ->
        valid_all_models nonempty_v LS s.
  Proof.
    intros nonempty_v LS rules osr s H H0 H1.
    destruct H0.
    clear H0.
    apply sound_derivation with (3 := x).
      eapply downward_correct_GR.
      eexact H.
    intros h H0.
    contradiction.
  Qed.


End Soundness.

Implicit Arguments downward_correct_rule_set [V L T].
Implicit Arguments downward_correct_rule_strengthen [V L T].
Implicit Arguments strong_downward_correct_cut [V L T].
Implicit Arguments sound_GR [V L T].
