(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: substitution.v,v 1.35 2013/04/10 11:17:17 tews Exp $
 *)

(** ** Substitutions *)

Require Export formulas list_set functions.

Section Substitutions.

  Variable V : Type.
  Variable L : modal_operators.

  (** need a decidable equality on propositional constants for limit_subst *)
  Variable v_eq : eq_type V.


  (*************************************************************************)
  (** *** Substitutions on formulas *)
  (*************************************************************************)

  (* substitutions, page 6 *)
  Definition lambda_subst : Type := V -> lambda_formula V L.


  Definition subst_form(sigma : lambda_subst)(f : lambda_formula V L) : 
                                                        lambda_formula V L :=
    lambda_formula_rec sigma lf_neg lf_and lf_modal f.

  Lemma subst_form_char : 
    forall(sigma : lambda_subst)(f : lambda_formula V L),
      subst_form sigma f =
        match f with
          | lf_prop v => sigma v
          | lf_neg f => lf_neg (subst_form sigma f)
          | lf_and f1 f2 => lf_and (subst_form sigma f1) (subst_form sigma f2)
          | lf_modal op args =>
            lf_modal op (counted_map (subst_form sigma) args)
        end.
  Proof.
    intros sigma f.
    unfold subst_form at 1.
    rewrite lambda_formula_rec_char.
    trivial.
  Qed.

  Lemma subst_form_or : 
    forall(f1 f2 : lambda_formula V L)(sigma : lambda_subst),
      subst_form sigma (lambda_or f1 f2) = 
        lambda_or (subst_form sigma f1) (subst_form sigma f2).
  Proof.
    intros f1 f2 sigma.
    unfold lambda_or in *.
    rewrite subst_form_char.
    f_equal.
  Qed.


  Lemma subst_form_rank_increase :
    forall(f : lambda_formula V L)(sigma : lambda_subst),
      modal_rank f <= modal_rank (subst_form sigma f).
  Proof.
    clear. 
    induction f.
          intros sigma.
          rewrite modal_rank_char.
          rewrite subst_form_char.
          assert (H :=  modal_rank_ge_1 (sigma v)).
          omega.
        intros sigma.
        rewrite modal_rank_char.
        rewrite subst_form_char.
        rewrite (modal_rank_char (lf_neg _)).
        auto.
      intros sigma.
      rewrite modal_rank_char.
      rewrite subst_form_char.
      rewrite (modal_rank_char (lf_and _ _)).
      apply max_mono_both; auto.
    intros sigma.
    rewrite modal_rank_char.
    rewrite subst_form_char.
    rewrite (modal_rank_char (lf_modal _ _)).
    apply le_n_S.
    unfold nat_list_max.
    fold nat_list_max.
    apply max_mono_both; auto.
    assert (length (map (modal_rank (L:=L)) (list_of_counted_list args)) =
            length (map (modal_rank (L:=L))
                 (list_of_counted_list (counted_map (subst_form sigma) args)))).
      rewrite map_length.
      rewrite length_list_of_counted_list.
      rewrite map_length.
      rewrite length_list_of_counted_list.
      trivial.
    apply nat_list_max_pairwise_le with (len_eq := H0).
    intros n n_less.
    rewrite nth_map.
    generalize 
      (eq_rect (length (map (modal_rank (L:=L)) (list_of_counted_list args)))
        (fun m : nat => n < m) n_less
        (length
           (map (modal_rank (L:=L))
              (list_of_counted_list (counted_map (subst_form sigma) args))))
        H0).
    rewrite list_of_counted_list_map.
    intros l.
    rewrite nth_map.
    rewrite nth_map.
    erewrite nth_tcc_irr.
    apply H.
  Qed.


  Definition id_subst(v : V) : lambda_formula V L := lf_prop v.

  Lemma subst_form_id : forall(f : lambda_formula V L),
    subst_form id_subst f = f.
  Proof.
    induction f.
          trivial.
        rewrite subst_form_char.
        rewrite IHf.
        trivial.
      rewrite subst_form_char.
      rewrite IHf1.
      rewrite IHf2.
      trivial.
    rewrite subst_form_char.
    f_equal.
    apply counted_list_equal.
    rewrite list_of_counted_list_map.
    apply list_equal_nth_char.
      rewrite map_length.
      trivial.
    intros n n1_less n2_less.
    rewrite nth_map.
    erewrite nth_tcc_irr.
    apply H.
  Qed.


  (*************************************************************************)
  (** *** Composition of substitutions *)
  (*************************************************************************)

  Definition subst_compose(sigma_1 sigma_2 : lambda_subst) : lambda_subst :=
    fun(v : V) => subst_form sigma_1 (sigma_2 v).

  Lemma subst_form_compose : 
    forall(f : lambda_formula V L)(sigma_1 sigma_2 : lambda_subst),
      subst_form (subst_compose sigma_1 sigma_2) f =
        subst_form sigma_1 (subst_form sigma_2 f).
  Proof.
    clear. 
    induction f.
          intros sigma_1 sigma_2.
          unfold subst_compose in *.
          trivial.
        intros sigma_1 sigma_2.
        rewrite subst_form_char.
        rewrite (subst_form_char sigma_2).
        rewrite (subst_form_char sigma_1).
        rewrite IHf.
        trivial.
      intros sigma_1 sigma_2.
      rewrite subst_form_char.
      rewrite (subst_form_char sigma_2).
      rewrite (subst_form_char sigma_1).
      rewrite IHf1.
      rewrite IHf2.
      trivial.
    intros sigma_1 sigma_2.
    rewrite subst_form_char.
    rewrite (subst_form_char sigma_2).
    rewrite (subst_form_char sigma_1).
    f_equal.
    induction args.
      simpl.
      trivial.
    simpl in *.
    f_equal.
      assert (H0 := every_nth_head _ _ _ H).
      simpl in H0.
      apply H0.
    apply IHargs.
    eapply every_nth_tail.
    eexact H.
  Qed.

  Lemma subst_compose_id_right :
    forall(sigma : lambda_subst),
      subst_compose sigma id_subst ≡ sigma.
  Proof.
    unfold subst_compose, id_subst in *.
    intros sigma v.
    apply subst_form_char.
  Qed.

  Lemma subst_compose_assoc :
    forall(sigma1 sigma2 sigma3 : lambda_subst),
      subst_compose (subst_compose sigma1 sigma2) sigma3 ≡
        subst_compose sigma1 (subst_compose sigma2 sigma3).
  Proof.
    intros sigma1 sigma2 sigma3 v.
    unfold subst_compose in |-* at 1.
    rewrite subst_form_compose.
    trivial.
  Qed.


  (*************************************************************************)
  (** *** Substitutions on sequents *)
  (*************************************************************************)

  Definition subst_sequent(sigma : lambda_subst)(s : sequent V L) : 
                                                                sequent V L :=
    map (subst_form sigma) s.

  Lemma length_subst_sequent : forall(sigma : lambda_subst)(s : sequent V L),
    length (subst_sequent sigma s) = length s.
  Proof.
    intros sigma s.
    unfold subst_sequent in *.
    rewrite map_length.
    trivial.
  Qed.

  Lemma subst_sequent_id : 
    forall(s : sequent V L), subst_sequent id_subst s = s. 
  Proof.
    intros s.
    unfold subst_sequent in *.
    apply list_equal_nth_char.
      rewrite map_length.
      trivial.
    intros n n1_less n2_less.
    rewrite nth_map.
    rewrite subst_form_id.
    apply nth_tcc_irr.
  Qed.

  Lemma subst_sequent_append : 
    forall(sigma : lambda_subst)(s1 s2 : sequent V L),
      subst_sequent sigma (s1 ++ s2) = 
        (subst_sequent sigma s1) ++ (subst_sequent sigma s2).
  Proof.
    clear. 
    intros sigma s1 s2.
    apply map_app.
  Qed.

  Lemma neg_sequent_subst_sequent :
    forall(sigma : lambda_subst)(s : sequent V L),
      every_nth neg_form s -> every_nth neg_form (subst_sequent sigma s).
  Proof.
    intros sigma s H.
    apply every_nth_map.
    intros i i_less.
    specialize (H i i_less).
    destruct (nth s i i_less); try contradiction.
    rewrite subst_form_char.
    trivial.
  Qed.

  Lemma nth_subst_sequent : 
    forall(sigma : lambda_subst)(s : sequent V L)
          (n : nat)(n_less_subst : n < length (subst_sequent sigma s))
          (n_less_s : n < length s),
      nth (subst_sequent sigma s) n n_less_subst = 
        subst_form sigma (nth s n n_less_s).
  Proof.
    clear. 
    intros sigma s n n_less_subst n_less_s.
    unfold subst_sequent in *.
    rewrite nth_map.
    f_equal.
    apply nth_tcc_irr.
  Qed.

  Lemma list_reorder_subst_sequent :
    forall(sigma : lambda_subst)(s1 s2 : sequent V L),
      list_reorder s1 s2 ->
        list_reorder (subst_sequent sigma s1) (subst_sequent sigma s2).
  Proof.
    clear. 
    unfold subst_sequent in *.
    intros sigma s1 s2 H.
    apply list_reorder_map.
    trivial.
  Qed.

  Lemma or_formula_subst_sequent_tcc :
    forall(s : sequent V L)(sigma : lambda_subst),
      s <> [] -> subst_sequent sigma s <> [].
  Proof.
    intros s sigma H.
    destruct s.
      exfalso.
      apply H.
      trivial.
    intros H0.
    discriminate.
  Qed.
  
  Lemma or_formula_subst_sequent :
    forall(s : sequent V L)(sigma : lambda_subst)(nonempty_s : s <> []),
      or_formula_of_ne_sequent (subst_sequent sigma s) 
               (or_formula_subst_sequent_tcc s sigma nonempty_s) =
        subst_form sigma (or_formula_of_ne_sequent s nonempty_s).
  Proof.
    clear. 
    intros s sigma nonempty_s.
    destruct s.
      exfalso.
      apply nonempty_s.
      trivial.
    simpl.
    clear nonempty_s.
    revert l.
    induction s.
      intros l.
      simpl.
      trivial.
    intros l.
    simpl.
    rewrite <- subst_form_or.
    apply IHs.
  Qed.


  Lemma subst_sequent_rank_increase :
    forall(n : nat)(s : sequent V L)(sigma : lambda_subst),
      rank_sequent n (subst_sequent sigma s) -> rank_sequent n s.
  Proof.
    clear. 
    unfold rank_sequent, every_nth in *.
    intros n s sigma H n0 n_less.
    assert (n0 < length (subst_sequent sigma s)).
      rewrite length_subst_sequent.
      trivial.
    specialize (H n0 H0).
    unfold rank_formula in *.
    rewrite nth_subst_sequent with (n_less_s := n_less) in H.
    assert (H1 := subst_form_rank_increase (nth s n0 n_less) sigma). 
    omega.
  Qed.

  Lemma subst_sequent_compose :
    forall(s : sequent V L)(sigma_1 sigma_2 : lambda_subst),
      subst_sequent (subst_compose sigma_1 sigma_2) s =
        subst_sequent sigma_1 (subst_sequent sigma_2 s).
  Proof.
    clear. 
    intros s sigma_1 sigma_2.
    unfold subst_sequent in *.
    rewrite map_map.
    apply map_ext.
    intros a.
    apply subst_form_compose.
  Qed.


  (***************************************************************************)
  (** ***  substitution rank  *)
  (***************************************************************************)

  Definition rank_subst(n : nat)(sigma : lambda_subst) : Prop :=
    forall(v : V), rank_formula n (sigma v).

  Lemma nonempty_rank_subst : forall(v : V)(n : nat)(sigma : lambda_subst),
    rank_subst n sigma -> 0 < n.
  Proof.
    intros v n sigma H.
    specialize (H v).
    unfold rank_formula in *.
    assert (H0 := modal_rank_ge_1 (sigma v)).
    omega.
  Qed.

  Lemma rank_subst_ge :
    forall(n1 n2 : nat), 
      n1 <= n2 ->
        subset (rank_subst n1) (rank_subst n2).
  Proof.
    unfold subset, rank_subst in *.
    intros n1 n2 H a H0 v.
    eapply rank_formula_ge; eauto.
  Qed.

  Lemma rank_subst_update :
   forall(sigma : lambda_subst)(n : nat)(v : V)(f : lambda_formula V L),
     rank_formula n f ->
     rank_subst n sigma ->
       rank_subst n (function_update v_eq sigma v f).
  Proof.
    unfold rank_subst, function_update in *.
    intros sigma n v f H H0 v0.
    destruct (v_eq v0 v).
      trivial.
    trivial.
  Qed.

  Lemma rank_formula_subst_formula_add :
    forall(f : lambda_formula V L)(sigma : lambda_subst)(n k : nat),
      1 <= k ->
      rank_formula n f ->
      rank_subst k sigma ->
        rank_formula (n + k - 1) (subst_form sigma f).
  Proof.
    unfold rank_formula in *.
    induction f.
          intros sigma n k H H0 H1.
          rewrite modal_rank_char in H0.
          rewrite subst_form_char.
          specialize (H1 v).
          unfold rank_formula in *.
          omega.
        intros sigma n k H H0 H1.
        rewrite subst_form_char.
        rewrite modal_rank_char in H0.
        rewrite modal_rank_char.
        apply IHf.
            trivial.
          trivial.
        trivial.
      intros sigma n k H H0 H1.
      rewrite subst_form_char.
      rewrite modal_rank_char.
      rewrite modal_rank_char in H0.
      apply Max.max_lub.
        apply IHf1.
            trivial.
          eapply Max.max_lub_l.
          eexact H0.
        trivial.
      apply IHf2.
          trivial.
        eapply Max.max_lub_r.
        eexact H0.
      trivial.
    intros sigma n k H0 H1 H2.
    rewrite subst_form_char.
    rewrite modal_rank_char.
    rewrite modal_rank_char in H1.
    rewrite list_of_counted_list_map.
    destruct n.
      omega.
    assert (H3 := le_S_n _ _ H1).
    clear H1.
    lapply (plus_minus_assoc 1 (n + k) 1).
      intros H1.
      change (1 + (n + k)) with (S n + k) in H1.
      rewrite H1; clear H1.
      apply le_n_S.
      unfold nat_list_max in *.
      fold nat_list_max in *.
      assert (H4 := Max.max_lub_l _ _ _ H3).
      apply Max.max_lub.
        omega.
      assert (H5 := Max.max_lub_r _ _ _ H3).
      clear H3.
      apply nat_list_max_le.
      apply every_nth_map.
      apply every_nth_map.
      assert (H6 := nat_list_max_le_inv _ _ H5).
      rewrite every_nth_map in H6.
      intros i i_less.
      apply H.
          trivial.
        apply H6.
      trivial.
    omega.
  Qed.

  (* 
   * Lemma subst_formula_modal_rank_ge_2 :
   *   forall(op : operator L)
   *         (args : counted_list (lambda_formula V L) (arity L op))
   *         (sigma : lambda_subst)(n : nat),
   *     rank_formula n (subst_form sigma (lf_modal op args)) ->
   *       1 < n.
   * Proof.
   *   intros op args sigma n H.
   *   rewrite subst_form_char in *.
   *   eapply rank_formula_modal_ge_2.
   *   eexact H.
   * Qed.
   *)

  Lemma rank_sequent_subst_add :
    forall(s : sequent V L)(sigma : lambda_subst)(n k npk : nat),
      rank_sequent n s ->
      rank_subst k sigma ->
      1 <= k ->
      n + k - 1 = npk ->
        rank_sequent npk (subst_sequent sigma s).
  Proof.
    clear. 
    intros s sigma n k npk H H0 H1 H2.
    subst npk.
    unfold rank_sequent, subst_sequent in *.
    rewrite every_nth_map.
    intros i i_less.
    apply rank_formula_subst_formula_add.
        trivial.
      apply H.
    trivial.
  Qed.

  Lemma rank_sequent_subst_prop_sequent :
    forall(s : sequent V L)(sigma : lambda_subst)(n : nat),
      prop_sequent s ->
      rank_subst n sigma ->
        rank_sequent n (subst_sequent sigma s).
  Proof.
    clear. 
    intros s sigma n H H0.
    destruct s.
      apply rank_sequent_empty.
    assert (n = 1 + n - 1).
      omega.
    rewrite H1.
    clear H1.
    eapply rank_sequent_subst_add.
          apply rank_prop_sequent.
          trivial.
        eexact H0.
      eapply nonempty_rank_subst.
        assert (H1 := every_nth_head _ _ _ H).
        clear H.
        destruct l; try contradiction.
          trivial.
        destruct l; try contradiction.
        trivial.
      eexact H0.
    trivial.
  Qed.

  (* 
   * Lemma subst_conclusion_rank_ge_2 :
   *   forall(r : sequent_rule V L)(sigma : lambda_subst)(n : nat),
   *     one_step_rule v_eq r ->
   *     rank_sequent n (subst_sequent sigma (conclusion r)) ->
   *       1 < n.
   * Proof.
   *   intros r sigma n H H0.
   *   unfold one_step_rule in *.
   *   decompose [ex and or dep_and] H; clear H.
   *   clear H1 H2.
   *   destruct (conclusion r).
   *     exfalso.
   *     apply H5.
   *     trivial.
   *   specialize (H0 0 (lt_0_Sn (length (subst_sequent sigma s)))).
   *   unfold subst_sequent in *.
   *   rewrite nth_map in H0.
   *   specialize (H3 0 (lt_0_Sn (length s))).
   *   simpl in *.
   *   destruct l.
   *         contradiction.
   *       simpl in *.
   *       rewrite subst_form_char in *.
   *       rewrite rank_formula_lf_neg in *.
   *       destruct l.
   *             contradiction.
   *           contradiction.
   *         contradiction.
   *       eapply subst_formula_modal_rank_ge_2; eauto.
   *     contradiction.
   *   eapply subst_formula_modal_rank_ge_2; eauto.
   * Qed.
   *)

  Lemma rank_subst_subst_compose :
    forall(n k npk : nat)(sigma_1 sigma_2 : lambda_subst),
      rank_subst (S n) sigma_1 ->
      rank_subst (S k) sigma_2 ->
      S (n + k) = npk ->
        rank_subst npk (subst_compose sigma_1 sigma_2).
  Proof.
    unfold rank_subst, subst_compose in *.
    intros n k npk sigma_1 sigma_2 H H0 H1 v.
    subst npk.
    lapply (rank_formula_subst_formula_add (sigma_2 v) sigma_1 (S k) (S n)).
      intros H1; lapply H1; clear H1; auto.
      intros H1; lapply H1; clear H1; auto.
      intros H1.
      simpl in H1.
      rewrite <- minus_n_O in H1.
      rewrite plus_comm in H1.
      trivial.
    apply le_n_S.
    apply le_0_n.
  Qed.


  (***************************************************************************)
  (** *** Substitution on sequent sets *)
  (***************************************************************************)

  Definition subst_sequent_set(sigma : lambda_subst)(ss : set (sequent V L))
                                                        : set (sequent V L) :=
    fun(s : sequent V L) =>
      exists(o : sequent V L),
        ss o /\ s = subst_sequent sigma o.

  Lemma subst_sequent_set_empty :
    forall(sigma : lambda_subst),
      set_equal (subst_sequent_set sigma (empty_sequent_set V L))
                (empty_sequent_set V L).
  Proof.
    intros sigma s.
    split.
      intros H.
      destruct H.
      destruct H.
      contradiction.
    contradiction.
  Qed.


  (***************************************************************************)
  (** *** Substituted rules *)
  (***************************************************************************)

  Definition subst_sequent_rule(sigma : lambda_subst)(r : sequent_rule V L)
                                                         : sequent_rule V L :=
    {| assumptions := map (subst_sequent sigma) (assumptions r);
       conclusion := subst_sequent sigma (conclusion r)
    |}.


  Lemma rule_has_rank_subst_rule :
    forall(sigma : lambda_subst)(r : sequent_rule V L)(n k : nat),
      rule_has_rank n r ->
      rank_subst (S k) sigma ->
        rule_has_rank (n + k) (subst_sequent_rule sigma r).
  Proof.
    clear. 
    intros sigma r n k H H0.
    unfold rule_has_rank in *.
    destruct H.
    split.
      unfold subst_sequent_rule in *.
      simpl in *.
      intros i i_less.
      rewrite nth_map.
      eapply rank_sequent_subst_add.
            apply H.
          eexact H0.
        omega.
      omega.
    clear H.
    unfold subst_sequent_rule in *.
    simpl in *.
    eapply rank_sequent_subst_add.
          eexact H1.
        eexact H0.
      omega.
    omega.
  Qed.


  Definition subst_closed_rule_set(rules : set (sequent_rule V L)) : Prop :=
    forall(sigma : lambda_subst)(r : sequent_rule V L),
      rules r -> rules (subst_sequent_rule sigma r).


  (***************************************************************************)
  (** *** Limit substitutions  *)
  (***************************************************************************)

  Lemma rank_formula_id_subst : forall(n : nat)(v : V),
    0 < n -> rank_formula n (id_subst v).
  Proof.
    intros n v H.
    unfold rank_formula,id_subst in *.
    rewrite modal_rank_char.
    omega.
  Qed.

  Lemma rank_subst_id_subst : forall(n : nat),
    0 < n -> rank_subst n id_subst.
  Proof.
    intros n H v.
    apply rank_formula_id_subst.
    trivial.
  Qed.

  Definition limit_subst(pv : list V)(sigma : lambda_subst) : lambda_subst :=
    fun(v : V) => 
      if member v_eq v pv
      then sigma v
      else id_subst v.

  Lemma rank_subst_limit_lf_modal :
    forall(n : nat)(v : V)(sigma : lambda_subst)
          (op : operator L)
          (args : counted_list (lambda_formula V L) (arity L op)),
      every_nth prop_form (list_of_counted_list args) ->
      In v (prop_var_formula (lf_modal op args)) ->
      rank_formula n (subst_form sigma (lf_modal op args)) ->
        rank_formula (pred n) (sigma v).
  Proof.
    intros n v sigma op args H H0 H1.
    unfold rank_formula in *.
    rewrite prop_var_formula_char in H0.
    assert (H2 := In_flatten _ _ H0).
    clear H0.
    decompose [ex and or dep_and] H2; clear H2.
    rename x into i, a into i_less, b into H2.
    rewrite nth_map in H2.
    remember (nth_map_tcc (prop_var_formula (L:=L))
                          (list_of_counted_list args) i i_less)
       as nth_tcc.
    specialize (H i nth_tcc).
    rewrite subst_form_char in H1.
    rewrite modal_rank_char in H1.
    destruct n.
      exfalso.
      omega.
    apply le_S_n in H1.
    simpl.
    unfold nat_list_max in H1.
    fold nat_list_max in H1.
    apply Max.max_lub_r in H1.
    apply nat_list_max_le_inv in H1.
    rewrite list_of_counted_list_map in H1.
    rewrite every_nth_map in H1.
    rewrite every_nth_map in H1.
    specialize (H1 i nth_tcc).
    simpl in H1.
    destruct (nth (list_of_counted_list args) i nth_tcc); try contradiction.
    rewrite prop_var_formula_char in H2.
    destruct H2.
      subst v0.
      rewrite subst_form_char in H1.
      trivial.
    contradiction.
  Qed.


  (***************************************************************************)
  (** *** Substitution equality  *)
  (***************************************************************************)

  Definition subst_eq_on_vars(sigma1 sigma2 : lambda_subst)(pv : list V)
                                                                     : Prop :=
    forall(v : V), In v pv -> sigma1 v = sigma2 v.

  Lemma subst_eq_on_vars_empty :
    forall(sigma1 sigma2 : lambda_subst),
      subst_eq_on_vars sigma1 sigma2 [].
  Proof.
    clear. 
    unfold subst_eq_on_vars in *.
    intros sigma1 sigma2 v H.
    contradiction.
  Qed.

  Lemma subst_eq_on_vars_symm : 
    forall(sigma1 sigma2 : lambda_subst)(pv : list V),
      subst_eq_on_vars sigma1 sigma2 pv ->
        subst_eq_on_vars sigma2 sigma1 pv.
  Proof.
    intros sigma1 sigma2 pv H v H0.
    rewrite H.
      trivial.
    trivial.
  Qed.

  Lemma subst_eq_on_vars_trans : 
    forall(sigma1 sigma2 sigma3: lambda_subst)(pv : list V),
      subst_eq_on_vars sigma1 sigma2 pv ->
      subst_eq_on_vars sigma2 sigma3 pv ->
        subst_eq_on_vars sigma1 sigma3 pv.
  Proof.
    intros sigma1 sigma2 sigma3 pv H H0 v H1.
    rewrite H.
      apply H0.
      trivial.
    trivial.
  Qed.

  Lemma subst_eq_on_vars_trans_rev : 
    forall(sigma1 sigma2 sigma3: lambda_subst)(pv : list V),
      subst_eq_on_vars sigma3 sigma2 pv ->
      subst_eq_on_vars sigma1 sigma2 pv ->
        subst_eq_on_vars sigma1 sigma3 pv.
  Proof.
    intros sigma1 sigma2 sigma3 pv H H0.
    eapply subst_eq_on_vars_trans.
      eexact H0.
    apply subst_eq_on_vars_symm.
    trivial.
  Qed.

  Lemma subst_eq_on_vars_seq :
    forall(sigma1 sigma2 : lambda_subst)(pv : list V),
      sigma1 ≡ sigma2 ->
        subst_eq_on_vars 
          sigma1
          sigma2
          pv.
  Proof.
    intros sigma1 sigma2 pv H v H0.
    apply H.
  Qed.

  Lemma subst_eq_on_vars_compose_right_change :
    forall(subst1 subst2 subst3 : lambda_subst)(pv : list V),
      subst_eq_on_vars subst2 subst3 pv ->
        subst_eq_on_vars 
          (subst_compose subst1 subst2)
          (subst_compose subst1 subst3) pv.
  Proof.
    unfold subst_eq_on_vars, subst_compose in *.
    intros subst1 subst2 subst3 pv H v H0.
    rewrite H.
      trivial.
    trivial.
  Qed.

  Lemma subst_eq_on_vars_append :
    forall(pv1 pv2 : list V)(sigma1 sigma2 : lambda_subst),
      subst_eq_on_vars sigma1 sigma2 pv1 ->
      subst_eq_on_vars sigma1 sigma2 pv2 ->
        subst_eq_on_vars sigma1 sigma2 (pv1 ++ pv2).
  Proof.
    intros pv1 pv2 sigma1 sigma2 H H0 v H1.
    apply in_app_or in H1.
    destruct H1.
      apply H.
      trivial.
    apply H0.
    trivial.
  Qed.


  Lemma subst_formula_eq :
    forall(sigma1 sigma2 : lambda_subst)
          (f : lambda_formula V L)(prop_vars : list V),
      incl (prop_var_formula f) prop_vars ->
      subst_eq_on_vars sigma1 sigma2 prop_vars ->
        subst_form sigma1 f = subst_form sigma2 f.
  Proof.
    clear. 
    induction f.
          intros prop_vars H H0.
          rewrite prop_var_formula_char in H.
          rewrite subst_form_char.
          rewrite subst_form_char.
          apply H0.
          apply H.
          apply in_eq.
        intros prop_vars H H0.
        rewrite prop_var_formula_char in H.
        rewrite subst_form_char.
        rewrite (subst_form_char _ (lf_neg _)).
        f_equal.
        eapply IHf; eauto.
      intros prop_vars H H0.
      rewrite prop_var_formula_char in H.
      rewrite subst_form_char.
      rewrite (subst_form_char _ (lf_and _ _)).
      f_equal.
        eapply IHf1; eauto.
        eapply incl_lappl.
        eexact H.
      eapply IHf2; eauto.
      eapply incl_lappr.
      eexact H.
    intros prop_vars H0 H1.
    rewrite prop_var_formula_char in H0.
    rewrite subst_form_char.
    rewrite (subst_form_char _ (lf_modal _ _)).
    f_equal.
    unfold prop_var_modal_args in *.
    induction args.
      simpl.
      trivial.
    simpl in *.
    f_equal.
      eapply (every_nth_head _ _ _ H); eauto.
      eapply incl_lappl.
      eexact H0.
    apply IHargs.
      eapply every_nth_tail.
      eexact H.
    eapply incl_lappr.
    eexact H0.
  Qed.

  (** Proof the case for modal arguments again, because it is needed 
      separately. 
   *)
  Lemma subst_modal_args_eq :
    forall(sigma1 sigma2 : lambda_subst)
          (n : nat)(args : counted_list (lambda_formula V L) n)
          (prop_vars : list V),
      incl (prop_var_modal_args n args) prop_vars ->
      subst_eq_on_vars sigma1 sigma2 prop_vars ->
        counted_map (subst_form sigma1) args = 
        counted_map (subst_form sigma2) args.
  Proof.
    induction args.
      intros prop_vars H H0.
      trivial.
    intros prop_vars H H0.
    unfold prop_var_modal_args in *.
    simpl in *.
    f_equal.
      eapply subst_formula_eq.
        eapply incl_lappl.
        eexact H.
      trivial.
    eapply IHargs.
      eapply incl_lappr.
      eexact H.
    trivial.
  Qed.

  Lemma subst_sequent_eq :
    forall(sigma1 sigma2 : lambda_subst)(s : sequent V L)(prop_vars : list V),
      incl (prop_var_sequent s) prop_vars ->
      subst_eq_on_vars sigma1 sigma2 prop_vars ->
        subst_sequent sigma1 s = subst_sequent sigma2 s.
  Proof.
    clear. 
    induction s.
      intros prop_vars H H0.
      trivial.
    intros prop_vars H H0.
    unfold prop_var_sequent in *.
    simpl in *.
    f_equal.
      eapply subst_formula_eq.
        eapply incl_lappl.
        eexact H.
      trivial.
    eapply IHs.
      eapply incl_lappr.
      eexact H.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Equality for limited substitutions  *)
  (***************************************************************************)

  Lemma subst_eq_on_vars_limit :
    forall(sigma : lambda_subst)(pv : list V),
      subst_eq_on_vars (limit_subst pv sigma) sigma pv.
  Proof.
    unfold subst_eq_on_vars, limit_subst in *.
    intros sigma pv v H.
    rewrite (iff_left (member_In _ v pv) H).
    trivial.
  Qed.

  Lemma subst_sequent_limit_eq :
    forall(s : sequent V L)(sigma : lambda_subst)(prop_vars : list V),
      incl (prop_var_sequent s) prop_vars ->
        subst_sequent (limit_subst prop_vars sigma) s = subst_sequent sigma s.
  Proof.
    clear. 
    intros s sigma prop_vars H.
    eapply subst_sequent_eq.
      eexact H.
    apply subst_eq_on_vars_limit.
  Qed.

  Lemma map_subst_sequent_limit_eq :
    forall(ss : list (sequent V L))(sigma : lambda_subst)(prop_vars : list V),
      incl (flatten (map prop_var_sequent ss)) prop_vars ->
        map (subst_sequent (limit_subst prop_vars sigma)) ss =
        map (subst_sequent sigma) ss.
  Proof.
    clear. 
    induction ss.
      intros sigma prop_vars H.
      trivial.
    intros sigma prop_vars H.
    simpl in *.
    f_equal.
      apply subst_sequent_limit_eq.
      eapply incl_lappl.
      eexact H.
    apply IHss.
    eapply incl_lappr.
    eexact H.
  Qed.


End Substitutions.


(* Implicit Arguments lambda_subst [V L]. *)
Implicit Arguments subst_form [V L].
Implicit Arguments subst_form_char [V L].
Implicit Arguments subst_compose [V L].
Implicit Arguments subst_sequent [V L].
Implicit Arguments rank_subst [V L].
Implicit Arguments rank_formula_subst_formula_add [V L].
Implicit Arguments or_formula_subst_sequent_tcc [V L].
Implicit Arguments nonempty_rank_subst [V L].
Implicit Arguments rank_subst_ge [V L].
Implicit Arguments rank_sequent_subst_add [V L].
Implicit Arguments subst_sequent_set [V L].
Implicit Arguments subst_sequent_rule [V L].
Implicit Arguments subst_closed_rule_set [V L].
Implicit Arguments id_subst [[V] [L]].
Implicit Arguments subst_eq_on_vars [V L].
Implicit Arguments limit_subst [V L].
