(*
 * Memcheck -- ocaml runtime type checking
 *
 * Copyright (C) 2006, Hendrik Tews, all right reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this directory
 * for more details.
 *
 * $Id: memcheck.mli,v 1.4 2007/04/19 12:37:01 tews Exp $
 *
 * Description: 
 * 
 * Public runtime interface of Memcheck.
 *)


     (* Unique tags used internally to mark types and visited blocks *)
type type_tag

    (* User constructed runtime type descriptions must have 
     * invalid_type_tag in all type_tag slots. During type checking other 
     * tags will be used.
     *)
val invalid_type_tag : type_tag


     (* Type of the runtime type descriptions.
      * Fill type_tag slots with invalid_type_tag.
      * 
      * IMPORTANT: Most of ocaml_type's constructors are for internal
      * use only. All of 
      *   Int_type
      *   Float_type
      *   String_type
      *   Int32_type 
      *   Nativeint_type
      *   Array_type
      *   Tuple_type
      * should never be used in user code. Use the predefined type 
      * constructors below instead (see the technology section in the docs
      * for an explanation).
      * 
      * The constructors 
      *   Type_parameter
      *   Static_variant
      *   Record_type
      * should only be used when defining new type constructors in 
      * the way generate_type_descr does. To build and manipulate 
      * type expression one can use Type_constructor_use or the 
      * predefined and camlp4 generated type constructor definitions.
      *)
type ocaml_type =
  | Int_type of type_tag
  | Float_type of type_tag
  | String_type of type_tag
  | Int32_type of type_tag
  | Nativeint_type of type_tag
      (* Type_parameter refers to a bound type variable ('a) inside a 
       * type definition. Don't use it for free type variables (which
       * are currently unsupported). Type_parameter(n) is a placeholder 
       * for the n-th argument of a parametric type definition. Before 
       * the type definition is used arguments must be provided and 
       * a substitution replaces all Type_parameter's with the actual 
       * arguments
       *)
  | Type_parameter of int
  | Array_type of type_tag * ocaml_type
  | Tuple_type of type_tag * ocaml_type list
    (* Static_variant refers to normal variant types, in contrast to 
     * polymorphic variants 
     *)
  | Static_variant of type_tag * string list * (string * ocaml_type list) list
  | Record_type of type_tag * (string * ocaml_type) list
    (* Use of some other type defintion. It is a ref, because it will 
     * be updated (1) when resolving rekursive type definitions, 
     * and (2) when during type checking.
     *)
  | Type_constructor_use of ocaml_type_constructor ref

and ocaml_type_constructor =
    (* Recursive_application(rec_type_constructor_numer, arguments) 
     * A runtime type description for a recursive type should be 
     * recursive as well. To construct it, fill Type_constructor_use 
     * with Recursive_application(n, arguments), where n is the (zero based) 
     * number of the type definition in the current type definition block. 
     * Use fix_rec_type_def to update Recursive_application into 
     * Resolved_application.
     *)
  | Recursive_application of int * ocaml_type list
    (* Resolved_application(type_constructor, arguments)
     * Refers to a different type definition.
     *)
  | Resolved_application of ocaml_type_descr * ocaml_type list
    (* Applied is only used internally *)
  | Applied of ocaml_type


and ocaml_type_descr =
    (* Type_constructor_use(type_constructor_id, arity, type_expression) 
     * Triple describing a type constructor (type definition). 
     * type_expression is the type description of the body of the 
     * type constructor. It may contain Type_parameter to refer to 
     * type arguments of parametric type definitions
     *)
  | Type_constructor_def of string * int * ocaml_type


    (* Resolve recursive type descriptions. The numbers used inside 
     * in Recursive_application must currespond to the indes in the 
     * input list.
     *)
val fix_rec_type_def : ocaml_type_descr list -> unit

(* definition of some builtin types and type constructors *)

val bool_type_descr : ocaml_type
val int_type_descr : ocaml_type
val int32_type_descr : ocaml_type
val nativeint_type_descr : ocaml_type
val float_type_descr : ocaml_type
val string_type_descr : ocaml_type
val array_type_descr : ocaml_type -> ocaml_type
val tuple_type_descr : ocaml_type list -> ocaml_type
val list_type_descr : ocaml_type -> ocaml_type
val option_type_descr : ocaml_type -> ocaml_type
val ref_type_descr : ocaml_type -> ocaml_type


  (* various flags to adopt the behaviour of check below *)
type check_flags =
  | Channel of out_channel		(* for printing messages *)
  | Verbose_blocks			(* messages for check memory blocks *)
  | Verbose_types			(* messages for type applications *)
  | Verbose_statistics			(* a final statistic *)
  | Verbose_spinner			(* some kind of alive message *)
  | Verbose_type_ids			(* print type ids on type application *)
  | Verbose_trace			(* give trace for errors *)
  | Start_indent of int			(* initial indentation of messages *)

    (* check_flags flags value type_descr 
     * checks the value against type_descr returning true if they match
     * and false otherwise. Flags can be used to regulate the verbosity.
     *)
val check : check_flags list -> 'a -> ocaml_type -> bool
