(* Otags reloaded
 * 
 * Hendrik Tews Copyright (C) 2010 - 2016
 * 
 * This file is part of "Otags reloaded".
 * 
 * "Otags reloaded" is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * "Otags reloaded" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with "Otags reloaded". If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * $Id: translate_location.ml,v 1.5 2016/01/14 20:17:46 tews Exp $
 * 
 * translate locations after line directives
 * 
 *)

(** See translate_loc below for more information. *)

open Monitor_line_directive
open Types
open Source_channel

module Loc = Camlp4.PreCast.Loc


let line_directive_array = ref [| |]
let current_index = ref(-1)

(** Copy the contents of parsed_line_directives into the 
    line_directive_array, suitable for binary search. Ensure that the
    array is sorted and that the ld_file_offset fields are invalid,
    i.e., -1
 *)
let prepare_line_directives () =
  line_directive_array := Array.of_list (List.rev !parsed_line_directives);
  parsed_line_directives := [];
  last_filename := None;
  current_index := -1;
  assert(
    snd(Array.fold_left
	  (fun (prev_off, res) ld -> 
	    (ld.ld_parse_offset, 
	     res && prev_off < ld.ld_parse_offset && ld.ld_file_offset = -1))
	  (0, true)
	  !line_directive_array))


(** Search the last line directive before parse_off using binary search 
    in line_directive_array. Return the index of the found line directive
    in line_directive_array. Can only be called if there is at least one 
    line directive and if parse_off is behind the first line directive.
    Assumes that line_directive_array is ordered.
 *)
let rec search_directive_array parse_off si ei =
  assert(parse_off >= !line_directive_array.(si).ld_parse_offset &&
	   (ei + 1 = Array.length !line_directive_array ||
	       parse_off < !line_directive_array.(ei + 1).ld_parse_offset));
  if si = ei then si
  else
    let middle = (si + ei + 1) / 2 in
    if parse_off < !line_directive_array.(middle).ld_parse_offset
    then search_directive_array parse_off si (middle - 1)
    else search_directive_array parse_off middle ei


(** Search the last line directive before parse_off using binary search 
    in line_directive_array. Return the index of the found line directive
    in line_directive_array or -1 if parse_off is before the first line 
    directive.
 *)
let get_directive_index parse_off =
  if Array.length !line_directive_array = 0 ||
    parse_off < !line_directive_array.(0).ld_parse_offset 
  then -1
  else if !current_index >= 0 &&
      !line_directive_array.(!current_index).ld_parse_offset <= parse_off &&
      (!current_index + 1 = Array.length !line_directive_array ||
	  parse_off < 
	  !line_directive_array.(!current_index + 1).ld_parse_offset)
  then
    !current_index
  else begin
    current_index := 
      search_directive_array parse_off 0
      (Array.length !line_directive_array - 1);
    (* Printf.eprintf "GDI search %d gives %d\n%!" parse_off !current_index; *)
    !current_index
  end


(** Return the character offset of the start of line line *)
let get_line_offset file_name line =
  (* Printf.eprintf "GLO %s line %d\n%!" file_name line; *)
  let loc = Loc.of_tuple (file_name, line, 0, 0, line, 0, 0, true) in
  let ic = get_channel loc in
  let line_count = ref 1 in
  let pos = ref 0 in
  seek_in ic 0;
  try
    while !line_count < line do
      if input_char ic = '\n'
      then incr line_count;
      incr pos;
    done;
    !pos
  with 
    | End_of_file -> raise(Otags_parsing_error(loc, "Invalid location"))


(** Lookup the character offset of the beginning of the line mentioned 
    in directive. This offset is cached in the ld_file_offset field in 
    the directive. If not present (i.e., if ld_file_offset = -1) the
    offset is computed from the file.
 *)
let source_line_offset directive =
  if directive.ld_file_offset >= 0 then directive.ld_file_offset
  else 
    let file_offset = get_line_offset directive.ld_file directive.ld_line in
    directive.ld_file_offset <- file_offset;
    file_offset


(** Correct character offsets in loc for line directives.

    OCaml and Camlp4 locations have the problem that character offsets are
    wrong after line directives, see OCaml bug #5159, the documentation
    for Lexing.position and Camlp4.Sig.Loc.t. Line directives only update
    the file and the line number but not the offset. Therefore, after a
    line directive, the offset cannot be used to find a position in a
    file. This functions corrects all the character offsets in loc,
    relying on character offset information of the line directives that
    was recorded during lexing from the Camlp4 token filter in
    Monitor_line_directives.Line_directive_monitor and on character
    offsets of the beginning of lines in the original source file that is
    computed on the fly here.

    To correct an offset, I take the character difference between a
    location and the character following a line directive. This latter
    character is the first character of the start of the line that the
    line directive refers to. I just need its position in the original
    source file to correct the character offsets.

 *)
let translate_loc loc =
  let stop_parse_off = Loc.stop_off loc in
  let stop_dir_index = get_directive_index stop_parse_off in
  if stop_dir_index = -1 
  then begin
    (* Printf.eprintf "TL stop -1\n%!"; *)
    loc
  end
  else
    let stop_dir = !line_directive_array.(stop_dir_index) in
    let stop_diff = stop_dir.ld_parse_offset - (source_line_offset stop_dir) in
    let stop_off = stop_parse_off - stop_diff in
    let stop_bol = (Loc.stop_bol loc) - stop_diff in
    let start_parse_off = Loc.start_off loc in
    let start_dir_index = get_directive_index start_parse_off in
    let (start_off, start_bol) =
      if start_dir_index = -1 
      then (start_parse_off, Loc.start_bol loc)
      else
	let start_dir = !line_directive_array.(start_dir_index) in
	let start_diff = 
	  start_dir.ld_parse_offset - (source_line_offset start_dir) in
	let start_off = start_parse_off - start_diff in
	let start_bol = (Loc.start_bol loc) - start_diff in
	(start_off, start_bol)
    in
    let (file_name, start_line, _bol1, _off1, stop_line, _bol2, _off2, ghost) =
      Loc.to_tuple loc
    in
    (* 
     * Printf.eprintf "TL start index %d stop index %d\n%!" 
     *   start_dir_index stop_dir_index;
     *)
    Loc.of_tuple (file_name, start_line, start_bol, start_off, 
		  stop_line, stop_bol, stop_off, ghost)
