(* Otags reloaded
 * 
 * Hendrik Tews Copyright (C) 2010 - 2012
 * 
 * This file is part of "Otags reloaded".
 * 
 * "Otags reloaded" is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * "Otags reloaded" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with "Otags reloaded". If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * $Id: emacs.ml,v 1.8 2012-01-14 21:31:39 tews Exp $
 * 
 * write emacs tags files
 * 
 *)

open Types

module Loc = Camlp4.PreCast.Loc


(* For files with INCLUDE or line directives we suddenly get tags with
 * locations from different files. We cannot tag with the main source file
 * because, for the line directives case, we don't know their location and,
 * for the INCLUDE case, they do not occur there. We therefore maintain 
 * several buffers to which we append tags. They are created as needed 
 * and stored in a hash table, because, for the line directive case, 
 * it is possible that one jumps several times between two files.
 * 
 * This buffers hash table is empty between compilation units. And the 
 * current buffer is some default empty buffer that is never used.
 * 
 * If file = "" then the current_buf, file and ic fields hold only
 * placeholders. Therefore filenames must be different from "".
 *)
type emacs_tag_state = {
  tags_oc : out_channel;
  buffers : (string, Buffer.t) Hashtbl.t;
  mutable current_buf : Buffer.t;
  mutable file : string;
  mutable ic : in_channel;
}


(* Dig out the buffer into which the tags for file are written, to
 * append the next tag to the right buffer. If that buffer does not yet
 * exist it is created. If the file has already been opened inc_opt
 * should be Some(ic) for the corresponding input channel ic. If no
 * input channel for file is available this function (re-)opens the
 * file. If there is some problem, for instance, because of a strange
 * line directive, then a Sys_error exception from open_in escapes.
 *)
let make_current_buffer es file inc_opt =
  let buf =
    try
      Hashtbl.find es.buffers file
    with
      | Not_found -> 
	let buf = Buffer.create 4095 in
	Hashtbl.add es.buffers file buf;
	buf
  in
  let ic = match inc_opt with
    | Some ic -> ic
    | None -> open_in file
  in
  es.current_buf <- buf;
  if es.file <> "" then close_in es.ic;
  es.file <- file;
  es.ic <- ic


let emacs_tag_line line tag line_number char =
  Printf.sprintf "%s\127%s\001%d,%d\n" line tag line_number char


let start_unit es file ic =
  assert(file <> "");
  assert(Hashtbl.length es.buffers = 0);
  let mod_name = Misc.module_name file in
  let module_tag = emacs_tag_line "" mod_name 1 0 in
  make_current_buffer es file (Some ic);
  Buffer.add_string es.current_buf module_tag


let write_tag es loc tag = 
  (* 
   * Printf.eprintf "%s: def %s line %d: %s\n"
   *   (Loc.to_string loc)
   *   (cut_out source_ic (Loc.start_off loc) (Loc.stop_off loc))
   *   (Loc.start_bol loc)
   *   (cut_out source_ic (Loc.start_bol loc) (Loc.stop_off loc));
   *)
  if es.file <> Loc.file_name loc 
  then 
    (try
       make_current_buffer es (Loc.file_name loc) None
     with
       | Sys_error sys_msg ->
	 let msg = "Original source not available: " ^ sys_msg in
	 raise(Otags_parsing_error(loc, msg))
    );
  Buffer.add_string es.current_buf 
    (emacs_tag_line
       (Misc.cut_out es.ic (Loc.start_bol loc) (Loc.stop_off loc))
       tag
       (Loc.start_line loc)
       (Loc.start_bol loc))


  (* Buffer, never to be really used, serves as a placeholder for the
   * current_buf field between compilation units.
   *)
let default_empty_buffer = Buffer.create 1


let finish_unit es () =
  Hashtbl.iter 				(* XXX don't write empty buffers *)
    (fun file buf ->
      Printf.fprintf es.tags_oc "\012\n%s,%d\n" 
	file
	(Buffer.length buf);
      Buffer.output_buffer es.tags_oc buf;
    )
    es.buffers;
  Hashtbl.clear es.buffers;
  es.current_buf <- default_empty_buffer;
  if es.file <> "" then close_in es.ic;
  es.file <- "";
  es.ic <- stdin;
  ()
  

let finish_tagging _es () = ()


let init oc =
  let es = {
    tags_oc = oc;
    buffers = Hashtbl.create 23;
    current_buf = default_empty_buffer;
    file = "";
    ic = stdin;
  }
  in {
    start_unit = start_unit es;
    write_tag = write_tag es;
    finish_unit = finish_unit es;
    finish_tagging = finish_tagging es;
  }
